package ru.yandex.direct.core.entity.addition.callout.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.addition.callout.container.CalloutSelection;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.model.CalloutsStatusModerate;
import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutValidationService;
import ru.yandex.direct.core.entity.addition.callout.service.validation.DeleteCalloutValidationService;
import ru.yandex.direct.core.entity.addition.callout.service.validation.DetachCalloutValidationService;
import ru.yandex.direct.core.entity.banner.type.banneradditions.BannerAdditionsRepository;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.addition.converter.AdditionConverter.toModerateAdditions;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Сервис работы с уточнениями callout
 */
@Service
@ParametersAreNonnullByDefault
public class CalloutService implements EntityService<Callout, Long> {

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final CalloutRepository calloutRepository;
    private final BannerAdditionsRepository bannerAdditionsRepository;
    private final CalloutValidationService calloutValidationService;
    private final DeleteCalloutValidationService deleteCalloutValidationService;
    private final DetachCalloutValidationService detachCalloutValidationService;
    private final ModerationRepository moderationRepository;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public CalloutService(ShardHelper shardHelper,
                          DslContextProvider dslContextProvider, CalloutRepository calloutRepository,
                          BannerAdditionsRepository bannerAdditionsRepository,
                          CalloutValidationService calloutValidationService,
                          DeleteCalloutValidationService deleteCalloutValidationService,
                          DetachCalloutValidationService detachCalloutValidationService,
                          ModerationRepository moderationRepository) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.calloutRepository = calloutRepository;
        this.bannerAdditionsRepository = bannerAdditionsRepository;
        this.calloutValidationService = calloutValidationService;
        this.deleteCalloutValidationService = deleteCalloutValidationService;
        this.detachCalloutValidationService = detachCalloutValidationService;
        this.moderationRepository = moderationRepository;
    }

    /**
     * Добавление уточнений, если все уточнения прошли этап валидации
     *
     * @param clientId - ид клиента, которому будут добавлены уточнения
     * @param callouts - список уточнений
     */
    public MassResult<Long> addCalloutsFull(ClientId clientId, List<Callout> callouts) {
        return addCallouts(clientId, callouts, Applicability.FULL);
    }

    /**
     * Добавление уточнений. Добавление только тех уточнений, которые прошли этап валидации
     *
     * @param clientId - ид клиента, которому будут добавлены уточнения
     * @param callouts - список уточнений
     */
    public MassResult<Long> addCalloutsPartial(ClientId clientId, List<Callout> callouts) {
        return addCallouts(clientId, callouts, Applicability.PARTIAL);
    }

    private MassResult<Long> addCallouts(ClientId clientId, List<Callout> callouts, Applicability applicability) {
        return createAddOperation(clientId, callouts, applicability).prepareAndApply();
    }

    private CalloutAddOperation createAddOperation(ClientId clientId, List<Callout> callouts,
                                                   Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new CalloutAddOperation(applicability, callouts, calloutRepository, calloutValidationService, shard,
                clientId);
    }

    private MassResult<Long> addCallouts(
            int shard, ClientId clientId, List<Callout> callouts, Applicability applicability) {
        return createAddOperation(shard, clientId, callouts, applicability).prepareAndApply();
    }

    private CalloutAddOperation createAddOperation(int shard, ClientId clientId, List<Callout> callouts,
                                                   Applicability applicability) {
        return new CalloutAddOperation(
                applicability, callouts, calloutRepository, calloutValidationService, shard, clientId);
    }

    /**
     * Помечает уточнения удаленными, если они не привязаны к какому-либо баннеру
     */
    public MassResult<Long> deleteCallouts(List<Long> calloutIds, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        ValidationResult<List<Long>, Defect> validationResult =
                deleteCalloutValidationService.validateDelete(shard, clientId, calloutIds);
        Collection<Long> validItems = getValidItems(validationResult);
        if (validationResult.hasErrors() || validItems.isEmpty()) {
            return MassResult.brokenMassAction(calloutIds, validationResult);
        }

        calloutRepository.setDeleted(shard, validItems, Boolean.TRUE);

        return MassResult.successfulMassAction(calloutIds, validationResult);
    }

    public List<Callout> getCallouts(ClientId clientId, CalloutSelection selection, LimitOffset limitOffset) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return calloutRepository.get(shard, clientId, selection, limitOffset);
    }

    public Map<Long, List<Long>> getExistingCalloutIdsByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return calloutRepository.getExistingCalloutIdsByAdGroupIds(shard, clientId, adGroupIds);
    }

    /**
     * Переотправка уточнений на модерацию
     */
    public void remoderateCallouts(ClientId clientId, Collection<Long> calloutIds, boolean moderateAccept) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        calloutRepository.setStatusModerate(shard, calloutIds, CalloutsStatusModerate.READY);

        moderationRepository.addModerateAdditions(shard, toModerateAdditions(moderateAccept, calloutIds));
    }

    /**
     * Отвязывает уточнения с переданными id от всех баннеров, а потом удаляет их
     *
     * @param clientId   - id клиента, уточнения которого будут отвзяаны
     * @param calloutIds - список id уточнений
     */
    public MassResult<Long> detachAndDeleteCallouts(List<Long> calloutIds, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        ValidationResult<List<Long>, Defect> validationResult =
                detachCalloutValidationService.validateDetach(shard, clientId, calloutIds);
        List<Long> validItems = getValidItems(validationResult);
        if (validationResult.hasErrors() || validItems.isEmpty()) {
            return MassResult.brokenMassAction(calloutIds, validationResult);
        }
        dslContextProvider.ppc(shard).transaction(t -> {
            DSLContext txContext = t.dsl();

            bannerAdditionsRepository.deleteBannersAdditionsByAdditionIds(txContext, validItems);
            calloutRepository.setDeleted(txContext, validItems, Boolean.TRUE);
        });

        return MassResult.successfulMassAction(calloutIds, validationResult);
    }

    @Override
    public List<Callout> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return getCallouts(clientId,
                new CalloutSelection().withIds(List.copyOf(ids)),
                new LimitOffset(Integer.MAX_VALUE, 0));
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid, List<Callout> entities,
                                Applicability applicability) {
        return addCallouts(clientId, entities, applicability);
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer, List<Callout> entities,
                                 Applicability applicability) {
        return addCallouts(copyContainer.getShardTo(), copyContainer.getClientIdTo(), entities, applicability);
    }
}
