package ru.yandex.direct.core.entity.addition.callout.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.constraint.AdmissibleCharsConstraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutConstants.ALLOW_CALLOUT_LETTERS;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.maxClientCallouts;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.maxClientCalloutsWithDeleted;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class CalloutConstraints {

    private CalloutConstraints() {
    }

    static Constraint<String, Defect> allowedCalloutChars() {
        return new AdmissibleCharsConstraint(ALLOW_CALLOUT_LETTERS);
    }

    static Constraint<List<Callout>, Defect> maxCalloutsOnClient(Collection<Callout> existingCallouts) {
        Set<String> existingCalloutTexts = existingCallouts.stream()
                .filter(callout -> !callout.getDeleted())
                .map(Callout::getText)
                .collect(Collectors.toSet());
        return fromPredicate(
                maxCalloutOnClientPredicate(existingCalloutTexts, CalloutConstants.MAX_CALLOUTS_COUNT_ON_CLIENT),
                maxClientCallouts());
    }

    static Constraint<List<Callout>, Defect> maxCalloutsOnClientWithDeleted(
            Collection<Callout> existingCallouts) {
        Set<String> existingCalloutTexts = existingCallouts.stream()
                .map(Callout::getText)
                .collect(Collectors.toSet());
        return fromPredicate(maxCalloutOnClientPredicate(existingCalloutTexts,
                CalloutConstants.MAX_CALLOUTS_COUNT_ON_CLIENT_WITH_DELETED),
                maxClientCalloutsWithDeleted());
    }

    private static Predicate<List<Callout>> maxCalloutOnClientPredicate(Set<String> existingCalloutTexts, int max) {
        return callouts -> StreamEx.of(callouts)
                .map(Callout::getText)
                .append(existingCalloutTexts)
                .collect(Collectors.toSet()).size()
                <= max;
    }

}

