package ru.yandex.direct.core.entity.addition.callout.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutConstraints.allowedCalloutChars;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutConstraints.maxCalloutsOnClient;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutConstraints.maxCalloutsOnClientWithDeleted;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.adExtensionAlreadyExists;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.allowedSymbolsCalloutText;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.calloutTextIsEmpty;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.calloutTextLengthExceeded;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.duplicateCalloutTexts;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
public class CalloutValidationService {

    private final ShardHelper shardHelper;
    private final CalloutRepository calloutRepository;

    @Autowired
    public CalloutValidationService(ShardHelper shardHelper,
                                    CalloutRepository calloutRepository) {
        this.shardHelper = shardHelper;
        this.calloutRepository = calloutRepository;
    }

    public ValidationResult<List<Callout>, Defect> validate(ClientId clientId, List<Callout> callouts) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Collection<Callout> existingCallouts =
                calloutRepository.getClientExistingCallouts(shard, clientId);
        return generateValidation(callouts, existingCallouts);
    }

    public ValidationResult<List<Callout>, Defect> generateValidation(List<Callout> callouts,
                                                                      Collection<Callout> existingCallouts) {
        Set<String> existingNotDeletedCalloutTexts = existingCallouts.stream()
                .filter(c -> !c.getDeleted())
                .map(Callout::getText).collect(toSet());

        return ListValidationBuilder.<Callout, Defect>of(callouts)
                .check(notNull())
                .checkEach(notNull())
                .check(maxCalloutsOnClient(existingCallouts), When.isValid())
                .check(maxCalloutsOnClientWithDeleted(existingCallouts), When.isValid())
                .checkEach(unique(Callout::getText), duplicateCalloutTexts())
                .checkEachBy(c -> validateOneCallout(c, existingNotDeletedCalloutTexts), When.notNull())
                .getResult();
    }

    private ValidationResult<Callout, Defect> validateOneCallout(Callout callout, Set<String> existingTexts) {
        ModelItemValidationBuilder<Callout> v = ModelItemValidationBuilder.of(callout);

        v.item(Callout.TEXT)
                .check(notNull())
                .check(notBlank(), calloutTextIsEmpty(), When.isValid())
                .check(maxStringLength(CalloutConstants.MAX_CALLOUT_TEXT_LENGTH), calloutTextLengthExceeded())
                .check(allowedCalloutChars(), allowedSymbolsCalloutText(), When.isValid())
                .weakCheck(notInSet(existingTexts), adExtensionAlreadyExists());
        v.item(Callout.CLIENT_ID)
                .check(notNull())
                .check(greaterThan(0L));

        return v.getResult();
    }

}
