package ru.yandex.direct.core.entity.addition.callout.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.core.entity.banner.type.banneradditions.BannerAdditionsRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.adExtensionIsDeleted;
import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.adExtensionIsInUse;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class DeleteCalloutValidationService {

    private final CalloutRepository calloutRepository;
    private final BannerAdditionsRepository bannerAdditionsRepository;

    @Autowired
    public DeleteCalloutValidationService(
            CalloutRepository calloutRepository,
            BannerAdditionsRepository bannerAdditionsRepository) {
        this.calloutRepository = calloutRepository;
        this.bannerAdditionsRepository = bannerAdditionsRepository;
    }

    public ValidationResult<List<Long>, Defect> validateDelete(int shard, ClientId clientId,
                                                               List<Long> calloutIds) {
        Collection<Callout> existsCallouts = calloutRepository.getClientExistingCallouts(shard, clientId);

        Set<Long> existsCalloutIds = existsCallouts.stream()
                .map(Callout::getId)
                .collect(Collectors.toSet());

        Set<Long> deletedCalloutIds = existsCallouts.stream()
                .filter(Callout::getDeleted)
                .map(Callout::getId)
                .collect(Collectors.toSet());

        Set<Long> linkedBannersAdditions = bannerAdditionsRepository.getLinkedBannersAdditions(shard, existsCalloutIds);

        return ListValidationBuilder.of(calloutIds, Defect.class)
                .check(notNull())
                .checkEach(notNull())
                .checkEach(validId())
                .checkEach(unique(), When.isValid())
                .weakCheckEach(notInSet(deletedCalloutIds), adExtensionIsDeleted())
                .checkEach(inSet(existsCalloutIds), objectNotFound(), When.isValid())
                .checkEach(notInSet(linkedBannersAdditions), adExtensionIsInUse(), When.isValid())
                .getResult();
    }
}
