package ru.yandex.direct.core.entity.addition.callout.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.addition.callout.service.validation.CalloutDefinitions.adExtensionIsDeleted;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class DetachCalloutValidationService {

    private final CalloutRepository calloutRepository;

    @Autowired
    public DetachCalloutValidationService(CalloutRepository calloutRepository) {
        this.calloutRepository = calloutRepository;
    }

    public ValidationResult<List<Long>, Defect> validateDetach(int shard, ClientId clientId,
                                                               List<Long> calloutIds) {
        Collection<Callout> existsCallouts = calloutRepository.getClientExistingCallouts(shard, clientId);

        Set<Long> existsCalloutIds = listToSet(existsCallouts, Callout::getId);
        Set<Long> deletedCalloutIds = filterAndMapToSet(existsCallouts, Callout::getDeleted, Callout::getId);

        return ListValidationBuilder.of(calloutIds, Defect.class)
                .check(notNull())
                .checkEach(notNull())
                .checkEach(validId())
                .checkEach(unique(), When.isValid())
                .weakCheckEach(notInSet(deletedCalloutIds), adExtensionIsDeleted())
                .checkEach(inSet(existsCalloutIds), objectNotFound(), When.isValid())
                .getResult();
    }
}
