package ru.yandex.direct.core.entity.additionaltargetings.repository

import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.additionaltargetings.model.CampAdditionalTargeting
import ru.yandex.direct.dbschema.ppc.tables.CampAdditionalTargetings.CAMP_ADDITIONAL_TARGETINGS
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

@Repository
class CampAdditionalTargetingsRepository(
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
) {
    /**
     * Добавляет таргетинг {@param data} для кампании {@param cid}.
     */
    fun insertTargetingToCampaigns(shard: Int, cids: Collection<Long>, data: String, comment: String?) {
        var batch = dslContextProvider.ppc(shard)
            .insertInto(
                CAMP_ADDITIONAL_TARGETINGS,
                CAMP_ADDITIONAL_TARGETINGS.ID,
                CAMP_ADDITIONAL_TARGETINGS.CID,
                CAMP_ADDITIONAL_TARGETINGS.DATA,
                CAMP_ADDITIONAL_TARGETINGS.COMMENT
            )
        val ids = shardHelper.generateCampAdditionalTargetingsIds(cids.size)
        cids.zip(ids).forEach { pair ->
            batch = batch.values(pair.second, pair.first, data, comment)
        }
        batch.execute()
    }

    fun insertCampAdditionalTargetings(shard: Int, targetings: List<CampAdditionalTargeting.Companion.Builder>) {
        var batch = dslContextProvider.ppc(shard)
            .insertInto(
                CAMP_ADDITIONAL_TARGETINGS,
                CAMP_ADDITIONAL_TARGETINGS.ID,
                CAMP_ADDITIONAL_TARGETINGS.CID,
                CAMP_ADDITIONAL_TARGETINGS.DATA,
                CAMP_ADDITIONAL_TARGETINGS.COMMENT
            )
        val ids = shardHelper.generateCampAdditionalTargetingsIds(targetings.size)
        targetings.zip(ids).forEach { pair ->
            batch = batch.values(pair.second, pair.first.cid, pair.first.data, pair.first.comment)
        }
        batch.execute()
    }

    /**
     * Возвращает записи по списку id кампаний {@param cids}.
     */
    fun findByCids(shard: Int, cids: Collection<Long>): List<CampAdditionalTargeting> =
        dslContextProvider.ppc(shard).select(
            CAMP_ADDITIONAL_TARGETINGS.ID,
            CAMP_ADDITIONAL_TARGETINGS.CID,
            CAMP_ADDITIONAL_TARGETINGS.DATA,
            CAMP_ADDITIONAL_TARGETINGS.CREATE_TIME,
            CAMP_ADDITIONAL_TARGETINGS.COMMENT
        ).from(CAMP_ADDITIONAL_TARGETINGS)
            .where(CAMP_ADDITIONAL_TARGETINGS.CID.`in`(cids))
            .orderBy(CAMP_ADDITIONAL_TARGETINGS.CID, CAMP_ADDITIONAL_TARGETINGS.ID)
            .fetchInto(CAMP_ADDITIONAL_TARGETINGS)
            .map(CampAdditionalTargeting::fromRecord)

    /**
     * Удаляет записи по списку id {@param ids}.
     * @return количество удаленных строк
     */
    fun deleteByIds(shard: Int, ids: Collection<Long>) =
        dslContextProvider.ppc(shard)
            .deleteFrom(CAMP_ADDITIONAL_TARGETINGS)
            .where(CAMP_ADDITIONAL_TARGETINGS.ID.`in`(ids))
            .execute()

    /**
     * Удаляет записи по списку id кампаний {@param cids}.
     * @return количество удаленных строк
     */
    fun deleteByCids(shard: Int, cids: Collection<Long>?) =
        dslContextProvider.ppc(shard)
            .deleteFrom(CAMP_ADDITIONAL_TARGETINGS)
            .where(CAMP_ADDITIONAL_TARGETINGS.CID.`in`(cids))
            .execute()
}
