package ru.yandex.direct.core.entity.additionaltargetings.repository

import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.additionaltargetings.model.ClientAdditionalTargeting
import ru.yandex.direct.dbschema.ppc.tables.ClientAdditionalTargetings.CLIENT_ADDITIONAL_TARGETINGS
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

@Repository
class ClientAdditionalTargetingsRepository(
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
) {

    /**
     * Добавляет таргетинг {@param data} для клиентов {@param clientIds}.
     */
    fun insertTargetingToClients(shard: Int, clientIds: Collection<Long>, data: String, comment: String?) {
        var batch = dslContextProvider.ppc(shard)
            .insertInto(
                CLIENT_ADDITIONAL_TARGETINGS,
                CLIENT_ADDITIONAL_TARGETINGS.ID,
                CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID,
                CLIENT_ADDITIONAL_TARGETINGS.DATA,
                CLIENT_ADDITIONAL_TARGETINGS.COMMENT
            )
        val ids = shardHelper.generateClientAdditionalTargetingsIds(clientIds.size)
        clientIds.zip(ids).forEach { pair ->
            batch = batch.values(pair.second, pair.first, data, comment)
        }
        batch.execute()
    }

    fun insertClientAdditionalTargetings(
        shard: Int,
        targetings: Collection<ClientAdditionalTargeting.Companion.Builder>
    ) {
        var batch = dslContextProvider.ppc(shard)
            .insertInto(
                CLIENT_ADDITIONAL_TARGETINGS,
                CLIENT_ADDITIONAL_TARGETINGS.ID,
                CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID,
                CLIENT_ADDITIONAL_TARGETINGS.DATA,
                CLIENT_ADDITIONAL_TARGETINGS.COMMENT
            )
        val ids = shardHelper.generateClientAdditionalTargetingsIds(targetings.size)
        targetings.zip(ids).forEach { pair ->
            batch = batch.values(pair.second, pair.first.clientId, pair.first.data, pair.first.comment)
        }
        batch.execute()
    }

    /**
     * Возвращает записи шарда {@param shard} по списку id клиентов {@param clientIds}.
     */
    fun findByClientIds(shard: Int, clientIds: Collection<Long>): List<ClientAdditionalTargeting> =
        dslContextProvider.ppc(shard).select(
            CLIENT_ADDITIONAL_TARGETINGS.ID,
            CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID,
            CLIENT_ADDITIONAL_TARGETINGS.DATA,
            CLIENT_ADDITIONAL_TARGETINGS.CREATE_TIME,
            CLIENT_ADDITIONAL_TARGETINGS.COMMENT
        ).from(CLIENT_ADDITIONAL_TARGETINGS)
            .where(CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID.`in`(clientIds))
            .orderBy(CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID, CLIENT_ADDITIONAL_TARGETINGS.ID)
            .fetchInto(CLIENT_ADDITIONAL_TARGETINGS)
            .map(ClientAdditionalTargeting::fromRecord)

    /**
     * Удаляет записи шарда {@param shard} по списку id {@param ids}.
     * @return количество удаленных строк
     */
    fun deleteByIds(shard: Int, ids: Collection<Long>) =
        dslContextProvider.ppc(shard)
            .deleteFrom(CLIENT_ADDITIONAL_TARGETINGS)
            .where(CLIENT_ADDITIONAL_TARGETINGS.ID.`in`(ids))
            .execute()

    /**
     * Удаляет записи шарда {@param shard} по списку id клиентов {@param clientIds}.
     * @return количество удаленных строк
     */
    fun deleteByClientIds(shard: Int, clientIds: Collection<Long>) =
        dslContextProvider.ppc(shard)
            .deleteFrom(CLIENT_ADDITIONAL_TARGETINGS)
            .where(CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID.`in`(clientIds))
            .execute()
}
