package ru.yandex.direct.core.entity.adgeneration;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.adgeneration.model.SitelinkSuggest;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants.NOT_ALLOW_BANNER_LETTERS_RE;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstants.ALLOW_SITELINK_LETTERS;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstants.SITELINKS_MAX_LENGTH;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstants.SITELINKS_PER_BLOCK;
import static ru.yandex.direct.utils.TextConstants.SPACE_CHARS;

public class GenerationUtils {

    private static final String WWW_PREFIX = "www.";
    private static final String HTTP_PREFIX = "http://";
    private static final String HTTPS_PREFIX = "https://";

    public static <R> Result<R> successResult(R result) {
        return Result.successful(result);
    }

    public static <R> Result<R> successResult(R result, @Nullable DefectId d) {
        return d == null
                ? Result.successful(result)
                : successResult(result, List.of(d));
    }

    public static <R> Result<R> successResult(R result, Collection<DefectId> defectIds) {
        return Result.successful(
                result,
                new ValidationResult<>(
                        null,
                        Collections.emptyList(),
                        StreamEx.of(defectIds).map(d -> new Defect(d)).toList()
                ));
    }

    public static <R> Result<R> errorResult(DefectId d) {
        return errorResult(List.of(d));
    }

    public static <R> Result<R> errorResult(Collection<DefectId> defectIds) {
        return Result.broken(new ValidationResult<>(
                null,
                StreamEx.of(defectIds).map(d -> new Defect(d)).toList(),
                Collections.emptyList()
        ));
    }

    public static <R> Result<R> getSuccessResultOrDefaultWithWarnings(Result<R> result, R defaultValue) {
        return result.isSuccessful() ? result : Result.successful(defaultValue, moveErrorsToWarnings(result.getValidationResult()));
    }

    public static <T, D> ValidationResult<T, D> moveErrorsToWarnings(ValidationResult<T, D>... validationResults) {
        List<D> warnings = new ArrayList<>();
        StreamEx.of(validationResults).forEach(r -> {
            if (r.getErrors() != null) {
                warnings.addAll(r.getErrors());
            }
            if (r.getWarnings() != null) {
                warnings.addAll(r.getWarnings());
            }
        });
        return new ValidationResult<>(
                null,
                Collections.emptyList(),
                warnings
        );
    }

    public static String getDomainWithPath(String url) {
        URL uri = getURL(url.toLowerCase());
        String domain = removeWWWFromDomain(uri.getHost());
        String result = domain + uri.getPath();
        return result.endsWith("/") ? result.substring(0, result.length() - 1) : result;
    }

    static String getDomainLowerCaseWithoutWWW(String url) {
        String domain = getDomain(url.toLowerCase());
        return removeWWWFromDomain(domain);
    }

    private static String removeWWWFromDomain(String domain){
        return domain.startsWith(WWW_PREFIX) ? domain.substring(WWW_PREFIX.length()) : domain;
    }

    public static String getDomain(String url) {
        return getURL(url).getHost();
    }

    private static URL getURL(String url) {
        try {
            return new URL(appendProtocol(url));
        } catch (MalformedURLException ex) {
            try {
                return new URL("http:/");
            } catch (MalformedURLException e) { // should never happen :)
                throw new RuntimeException(e);
            }
        }
    }

    private static String appendProtocol(String url) {
        return url.startsWith("http") ? url : HTTP_PREFIX + url;
    }

    public static boolean compareHostAndPath(String request, String found) {
        request = appendProtocol(request);
        found = appendProtocol(found);
        URL left;
        URL right;
        try {
            left = new URL(request);
            right = new URL(found);
        } catch (MalformedURLException e) {
            return true;
        }
        return compareHost(left.getHost(), right.getHost()) &&
                comparePath(left.getPath(), right.getPath());
    }

    private static boolean compareHost(String left, String right) {
        left = left.toLowerCase();
        right = right.toLowerCase();
        left = left.startsWith("www.") ? left : "www." + left;
        right = right.startsWith("www.") ? right : "www." + right;
        return StringUtils.equals(left, right);
    }

    private static boolean comparePath(String left, String right) {
        left = left.toLowerCase();
        right = right.toLowerCase();
        left = left.endsWith("/") ? left : left + "/";
        right = right.endsWith("/") ? right : right + "/";
        return StringUtils.equals(left, right);
    }

    public static String removeMinusWords(String phrase) {
        final String separator = " ";
        return StringUtils.join(
                StreamEx.of(phrase.split(separator))
                        .filter(w -> !w.startsWith("-"))
                        .toList(),
                separator
        );
    }

    public static String setHttpsToUrl(String input) {
        if (input == null || !input.startsWith(HTTP_PREFIX)) {
            return input;
        }
        return HTTPS_PREFIX + input.substring(HTTP_PREFIX.length());
    }

    public static String updateSitelinkText(String input) {
        return correctRegister(clearInvalidSymbols(input, "[^" + Pattern.quote(ALLOW_SITELINK_LETTERS) + "]"));
    }

    public static String updateAdText(String input) {
        return correctRegister(clearInvalidSymbols(input, NOT_ALLOW_BANNER_LETTERS_RE));
    }

    public static List<String> updateAdTexts(List<String> input) {
        return input == null
                ? emptyList()
                : StreamEx.of(input)
                .map(GenerationUtils::updateAdText)
                .toList();
    }

    /**
     *
     * @param input - исходный текст
     * @param notAllowedSymbolsRegexp - регулярка для поиска запрещенных символов
     * @return
     */
    public static String clearInvalidSymbols(String input, String notAllowedSymbolsRegexp) {
        return input == null ? null : input
                .replaceAll(notAllowedSymbolsRegexp, " ")
                .replaceAll("[" + Pattern.quote(SPACE_CHARS) + "]{2,}", " ")
                .trim();
    }

    private final static Set<String> PARTICLES = Set.of("или", "либо", "нибудь", "иль");

    private final static int INDEX_OF_START = 0;
    private final static int INDEX_OF_END = 1;
    private final static int IS_FIRST_IN_SENTENCE = 2;
    private final static int UPPER_LETTER_COUNT = 3;
    private final static int IS_UPPER_WORD = 4;
    private final static int NEED_UPDATE_CASE = 5;

    public static String correctRegister(String input) {
        // Для каждого слова храним
        // [0] - индекс начала
        // [1] - индекс конца
        // [2] - флаг, что слово первое в предложении
        // [3] - количество заглавных букв
        // [4] - флаг, что слово заглавное
        // [5] - флаг, что нужно править регистр слова
        LinkedList<int[]> words = new LinkedList<>();

        // Слова, в которых присутствуют буквы из текущего множества подряд идущих букв.
        Set<int[]> wordsWithUpperCase = new HashSet<>();

        if (input == null) {
            return null;
        }
        String lower = input.toLowerCase();
        String upper = input.toUpperCase();
        int upperLetterCount = 0; //счетчик заглавных букв подряд (без строчных)
        int letterCountInWord = 0; //счетчик букв в слове
        boolean isFirstInSentence = true;
        for (int index=0; index<input.length(); index++) {
            char l = lower.charAt(index);
            char u = upper.charAt(index);
            if (l == u && !Character.isDigit(u)) {
                isFirstInSentence |= l == '.';
                if (isFirstInSentence) {
                    upperLetterCount = 0;
                }
                letterCountInWord = 0;
                continue;
            }
            letterCountInWord++;
            if (letterCountInWord == 1) {
                words.add(new int[]{index, index, isFirstInSentence ? 1 : 0, 0, 0, 0});
                isFirstInSentence = false;
            }
            int[] word = words.getLast();
            word[INDEX_OF_END] = index;
            char i = input.charAt(index);
            if (Character.isDigit(i)) {
                // идем дальше
            } else if (l == i) {
                upperLetterCount = 0;
                wordsWithUpperCase.clear();
            } else {
                word[UPPER_LETTER_COUNT]++;
                upperLetterCount++;
                if (upperLetterCount > 6) { // более 6 заглавных букв подряд, поэтому нужно править регистр.
                    for (int[] w : wordsWithUpperCase) {
                        w[NEED_UPDATE_CASE] = 1;
                    }
                    wordsWithUpperCase.clear();
                    word[NEED_UPDATE_CASE] = 1;
                } else {
                    wordsWithUpperCase.add(word);
                }
            }
            if (word[UPPER_LETTER_COUNT] >= Math.min(3, letterCountInWord)) {
                word[IS_UPPER_WORD] = 1; // слово заглавное, если все его буквы заглавные или хотя бы 3 его быквы заглавные
            } else {
                word[IS_UPPER_WORD] = 0;
            }
        }

        int upperWords = 0; // счетчик заглавных слов подряд
        int wordCountInSentence = 0; // счетчик слов в предложении
        // Слова, принадлежащие текущему множеству заглавных слов подряд.
        wordsWithUpperCase = new HashSet<>();
        for (int[] word : words) {
            if (word[IS_FIRST_IN_SENTENCE] > 0) {
                if (upperWords == wordCountInSentence) {
                    for (int[] w : wordsWithUpperCase) {
                        w[NEED_UPDATE_CASE] = 1;
                    }
                    wordsWithUpperCase.clear();
                }
                upperWords = 0;
                wordsWithUpperCase.clear();
                wordCountInSentence = 0;
            }
            wordCountInSentence++;
            if (word[IS_UPPER_WORD] > 0) {
                upperWords++;
                if (upperWords > 2) { // более 2 заглавных слов подряд или все слова заглавные, поэтому нужно править в них регистр регистр.
                    for (int[] w : wordsWithUpperCase) {
                        w[NEED_UPDATE_CASE] = 1;
                    }
                    wordsWithUpperCase.clear();
                    word[NEED_UPDATE_CASE] = 1;
                } else {
                    wordsWithUpperCase.add(word);
                }
            } else {
                upperWords = 0;
                wordsWithUpperCase.clear();
            }
        }
        if (upperWords == wordCountInSentence) {
            for (int[] w : wordsWithUpperCase) {
                w[NEED_UPDATE_CASE] = 1;
            }
            wordsWithUpperCase.clear();
        }
        StringBuilder result = new StringBuilder(input);
        for (int[] word : words) {
            if (word[NEED_UPDATE_CASE] == 0) {
                continue;
            }
            int start = word[INDEX_OF_START];
            int end = word[INDEX_OF_END];
            String replace = upper.charAt(start) + (start == end ? "" : lower.substring(start + 1, end + 1));
            if (word[IS_FIRST_IN_SENTENCE] == 0 &&
                    (replace.length() < 3 || PARTICLES.contains(replace.toLowerCase()))) {
                replace = replace.toLowerCase();
            }
            result.replace(start, end + 1, replace);
        }
        return result.toString();
    }

    public static List<SitelinkSuggest> sortAndSubListSitelinks(List<SitelinkSuggest> sitelinks) {
        //Сортируем по длине заголовков
        List<SitelinkSuggest> sortedList = StreamEx.of(sitelinks)
                .mapToEntry(SitelinkSuggest::getTitle, Function.identity())
                .sortedByInt(entry -> entry.getKey().length())
                .values()
                .toList();
        int length = 0;
        int sitelinksLimit = 0;
        for (SitelinkSuggest sitelink : sortedList) {
            length += sitelink.getTitle().length();
            if (length > SITELINKS_MAX_LENGTH) {
                break; // Лимит, значит нужно остановиться на предыдущем элементе
            } else {
                sitelinksLimit++; // Лимит не сработал, можно принять текущий элемент.
            }
            if (sitelinksLimit == SITELINKS_PER_BLOCK) {
                length = 0; // Блок закончился, нужно считать длину с нуля.
            }
        }
        if (sitelinksLimit < sortedList.size()) {
            sortedList = sortedList.subList(0, sitelinksLimit);
        }
        return sortedList;
    }
}
