package ru.yandex.direct.core.entity.adgeneration;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.validator.routines.UrlValidator;
import org.apache.curator.shaded.com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgeneration.model.ImageSuggest;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.imagesearch.ImageSearchClient;
import ru.yandex.direct.imagesearch.ImageSearchClientException;
import ru.yandex.direct.imagesearch.model.AbstractImage;
import ru.yandex.direct.imagesearch.model.ImageDoc;
import ru.yandex.direct.imagesearch.model.Thumb;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.richcontent.RichContentClient;
import ru.yandex.direct.richcontent.RichContentClientException;
import ru.yandex.direct.richcontent.model.Image;
import ru.yandex.direct.richcontent.model.UrlInfo;
import ru.yandex.direct.validation.result.DefectId;

import static ru.yandex.direct.common.db.PpcPropertyNames.DOMAIN_BLACKLIST_FOR_IMAGE_SEARCH;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.errorResult;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.getDomain;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.getSuccessResultOrDefaultWithWarnings;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.successResult;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.AD_GROUP_WITHOUT_KEYWORDS;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.BAD_CAMPAIGN_HREF;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.CAMPAIGN_WITHOUT_HREF;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.DOMAIN_DISABLED;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.EMPTY_IMAGE_SEARCH_API_RESPONSE;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.IMAGE_SEARCH_API_ERROR;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.IMAGE_SEARCH_API_WITHOUT_QUERY;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.RICH_CONTENT_API_ERROR;
import static ru.yandex.direct.core.entity.domain.DomainUtils.isSocialNetworkDomain;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_REGULAR_IMAGE_MIN_SIZE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Service
@ParametersAreNonnullByDefault
public class ImageGenerationService {

    private static final int MAX_IMAGES_NUMBER = 20;
    public static final int REQUEST_IMAGES_LIMIT = 100;
    private static final int MAX_KEYWORDS_NUMBER_FOR_SEARCH = 5;
    private static final int MIN_IMAGE_WIDTH = BANNER_REGULAR_IMAGE_MIN_SIZE;
    private static final int MIN_IMAGE_HEIGHT = BANNER_REGULAR_IMAGE_MIN_SIZE;

    private static final Logger logger = LoggerFactory.getLogger(ImageGenerationService.class);
    private static final UrlValidator URL_VALIDATOR = new UrlValidator(new String[]{"http", "https"});

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final ImageSearchClient imageSearchClient;
    private final KeywordService keywordService;
    private final RichContentClient richContentClient;
    private final PpcProperty<List<String>> domainBlacklistProperty;

    @Autowired
    public ImageGenerationService(
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            ImageSearchClient imageSearchClient,
            KeywordService keywordService,
            RichContentClient richContentClient,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.imageSearchClient = imageSearchClient;
        this.keywordService = keywordService;
        this.richContentClient = richContentClient;
        domainBlacklistProperty = ppcPropertiesSupport.get(DOMAIN_BLACKLIST_FOR_IMAGE_SEARCH);
    }

    public Result<Collection<ImageSuggest>> generateImages(@Nonnull ClientId clientId,
                                                           @Nullable String url,
                                                           @Nullable String text,
                                                           @Nullable Long campaignId,
                                                           @Nullable Long adGroupId,
                                                           Map<String, Object> additionalInfo) {
        if (url == null && campaignId != null) {
            int shard = shardHelper.getShardByCampaignId(campaignId);
            url = campaignRepository.getHrefCampAdditionalData(shard, List.of(campaignId)).get(campaignId);
        }

        Set<String> additionalTexts = Collections.emptySet();
        if (adGroupId != null) {
            List<Keyword> keywords = keywordService.getKeywordsByAdGroupIds(clientId, List.of(adGroupId))
                    .getOrDefault(adGroupId, Collections.emptyList());
            additionalTexts = StreamEx.of(keywords)
                    .sortedByLong(k -> -k.getShowsForecast())
                    .filter(k -> !k.getIsAutotargeting())
                    .map(Keyword::getPhrase)
                    .filter(Objects::nonNull)
                    .map(GenerationUtils::removeMinusWords)
                    .filter(phrase -> !phrase.equals(text))
                    .limit(MAX_KEYWORDS_NUMBER_FOR_SEARCH)
                    .toSet();
        }

        additionalInfo.put("url", url);
        additionalInfo.put("text", text);
        if (campaignId != null) {
            additionalInfo.put("cid", campaignId);
        }
        if (adGroupId != null) {
            additionalInfo.put("pid", adGroupId);
        }
        if (!additionalTexts.isEmpty()) {
            additionalInfo.put("keywords", new ArrayList<>(additionalTexts));
        }

        Result<Collection<ImageSuggest>> result = generateImages(url, text, additionalTexts);
        return getSuccessResultOrDefaultWithWarnings(result, Collections.emptyList());
    }

    protected Result<Collection<ImageSuggest>> generateImages(@Nullable String url,
            @Nullable String mainText, Set<String> additionalTexts) {
        if (url == null) {
            return errorResult(CAMPAIGN_WITHOUT_HREF);
        }
        if (!URL_VALIDATOR.isValid(url)) {
            return errorResult(BAD_CAMPAIGN_HREF);
        }
        List<DefectId> warningDefectIds = new ArrayList<>();
        if (mainText == null && additionalTexts.isEmpty()) {
            warningDefectIds.add(AD_GROUP_WITHOUT_KEYWORDS);
        }

        String domain = getDomain(url);
        domain = ifNotNull(domain, String::toLowerCase);
        final boolean isDomainDisabledForSearchByDomain = isDomainDisabledForSearchByDomain(domain);
        Map<String, List<ImageDoc>> result;
        try {
            Set<String> texts = Sets.union(
                    additionalTexts,
                    mainText == null ? Collections.emptySet() : Set.of(mainText));
            if (isDomainDisabledForSearchByDomain) {
                warningDefectIds.add(DOMAIN_DISABLED);
                result = imageSearchClient.getImagesByTextsAndUrl(texts, url, REQUEST_IMAGES_LIMIT);
            } else {
                result = imageSearchClient.getImagesByTextsAndDomain(texts, domain, REQUEST_IMAGES_LIMIT);
            }
        } catch (ImageSearchClientException e) {
            logger.error("Failed to search image", e);
            warningDefectIds.add(IMAGE_SEARCH_API_ERROR);
            result = Collections.emptyMap();
        }

        Map<String, ImageSuggest> suggestByImageId = new HashMap<>();
        Map<String, Integer> mainPriorityByImageId = new HashMap<>();
        Map<String, Integer> additionalPriorityByImageId = new HashMap<>();

        Map<String, Map<String, Integer>> indexByImageIdByText = EntryStream.of(result)
                .mapToValue((text, images) -> {
                    Map<String, Integer> priorityByImageId = text.equals(mainText) ?
                            mainPriorityByImageId : additionalPriorityByImageId;
                    return EntryStream.of(images) // Entry<order, image>
                            .mapValues(image -> handleImageDoc(image, suggestByImageId, priorityByImageId)) // Entry<index, imageId>
                            .nonNullValues() //только валидные картинки (для невалидных вернутся imageId=null
                            .invert() // Entry<imageId, index>
                            .toMap();
                }).toMap();

        Map<String, Integer> priorityByImageId = EntryStream.of(indexByImageIdByText)
                .map(entry -> {
                    boolean isMain = entry.getKey().equals(mainText);
                    Map<String, Integer> indexByImageId;
                    if (isMain) {
                        indexByImageId = new HashMap<>(entry.getValue().size());
                        StreamEx.ofKeys(suggestByImageId).forEach(imageId -> {
                            indexByImageId.put(
                                    imageId,
                                    // Результат с поиском по заголоску с весом 5 (MAX_KEYWORDS_NUMBER_FOR_SEARCH)
                                    MAX_KEYWORDS_NUMBER_FOR_SEARCH * entry.getValue()
                                            // если картинки нет, то считаем будто она на 101 позиции
                                            .getOrDefault(imageId, REQUEST_IMAGES_LIMIT + 1));
                        });
                    } else {
                        indexByImageId = new HashMap<>(entry.getValue());
                        StreamEx.ofKeys(suggestByImageId).forEach(imageId -> {
                            // если картинки нет, то считаем будто она на 101 позиции (REQUEST_IMAGES_LIMIT + 1)
                            indexByImageId.putIfAbsent(imageId, REQUEST_IMAGES_LIMIT + 1);
                        });
                    }
                    return indexByImageId;
                }) // Map<imageId, f(index)>
                .flatCollection(Map::entrySet) // Entry<imageId, index>
                .groupingBy(
                        Map.Entry::getKey,
                        Collectors.summingInt(Map.Entry::getValue)
                );

        List<String> imageIds = StreamEx.of(suggestByImageId.keySet())
                .sortedByInt(priorityByImageId::get)
                .toList();

        List<ImageSuggest> suggests = new ArrayList<>();

        suggests.addAll(
                StreamEx.of(imageIds)
                        .map(u -> suggestByImageId.get(u)
                                .withSourceInfo(
                                        mainPriorityByImageId.getOrDefault(u, 0),
                                        additionalPriorityByImageId.getOrDefault(u, 0)
                                ))
                        .toList()
        );

        if (suggests.isEmpty()) {
            List<ImageDoc> images;
            try {
                if (isDomainDisabledForSearchByDomain) {
                    images = imageSearchClient.getImagesByUrl(url, REQUEST_IMAGES_LIMIT);
                } else {
                    images = imageSearchClient.getImagesByDomain(domain, REQUEST_IMAGES_LIMIT);
                }
            } catch (ImageSearchClientException e) {
                logger.error("Failed to search image", e);
                images = Collections.emptyList();
            }
            if (images != null) {
                StreamEx.of(images)
                        .map(ImageDoc::getBigThumb)
                        .filter(ImageGenerationService::isValidImage)
                        .map(image -> new ImageSuggest(image)
                                .withSourceInfo(1, 1))
                        .forEach(suggests::add);
            }
            if (!suggests.isEmpty()) {
                warningDefectIds.add(IMAGE_SEARCH_API_WITHOUT_QUERY);
            }
        }

        if (suggests.isEmpty()) {
            warningDefectIds.add(EMPTY_IMAGE_SEARCH_API_RESPONSE);
            Image imageByUrl = null;
            Image imageByDomain = null;
            try {
                imageByUrl = ifNotNull(richContentClient.getUrlInfo(url), UrlInfo::getImage);
                imageByDomain = isDomainDisabledForSearchByDomain ? null :
                        ifNotNull(richContentClient.getUrlInfo(domain), UrlInfo::getImage);
            } catch (RichContentClientException ex) {
                logger.error("Failed to search image", ex);
                warningDefectIds.add(RICH_CONTENT_API_ERROR);
            }
            suggests.addAll(mergeImagesByRCA(imageByUrl, imageByDomain));
        }

        if (suggests.size() > MAX_IMAGES_NUMBER) {
            suggests = suggests.subList(0, MAX_IMAGES_NUMBER);
        }

        return successResult(suggests, warningDefectIds);
    }

    private boolean isDomainDisabledForSearchByDomain(@Nullable String domain) {
        if (isSocialNetworkDomain(domain)) {
            return true;
        }
        if (domainBlacklistProperty.getOrDefault(Collections.emptyList()).contains(domain)) {
            return true;
        }
        return false;
    }

    private static List<ImageSuggest> mergeImagesByRCA(Image imageByUrl, Image imageByDomain) {
        boolean isValidImageByUrl = isValidImage(imageByUrl);
        boolean isValidImageByDomain = isValidImage(imageByDomain);
        if (!isValidImageByUrl && !isValidImageByDomain) {
            return Collections.emptyList();
        }
        Image source;
        boolean byUrl = false;
        boolean byDomain = false;
        if (!isValidImageByUrl) {
            source = imageByDomain;
            byDomain = true;
        } else if (isValidImageByDomain && imageByUrl.getUrl().equals(imageByDomain.getUrl())) {
            source = imageByUrl;
            byUrl = true;
            byDomain = true;
        } else {
            source = imageByUrl;
            byUrl = true;
        }
        return List.of(new ImageSuggest(source).withSourceInfo(byUrl, byDomain));
    }

    private static boolean isValidImage(Image image) {
        return image != null && isValidImage(image.getUrl(), image.getWidth(), image.getHeight());
    }

    private static boolean isValidImage(AbstractImage image) {
        return image != null && isValidImage(image.getUrl(), image.getWidth(), image.getHeight());
    }

    private static boolean isValidImage(String url, Integer width, Integer height) {
        return StringUtils.isNotEmpty(url) &&
                width != null && width >= MIN_IMAGE_WIDTH &&
                height != null && height >= MIN_IMAGE_HEIGHT;
    }

    /**
     *
     * @param doc документ с картинкой
     * @return null для невалидных изображений и id для валидных
     */
    private static String handleImageDoc(
            ImageDoc doc,
            Map<String, ImageSuggest> suggestByImageId,
            Map<String, Integer> priorityByImageId
    ) {
        Thumb image = doc.getBigThumb();
        if (!isValidImage(image)) {
            return null;
        }
        suggestByImageId.put(image.getId(), new ImageSuggest(image));
        int priority = priorityByImageId.getOrDefault(image.getId(), 0);
        priority += 1;
        priorityByImageId.put(image.getId(), priority);
        return image.getId();
    }
}
