package ru.yandex.direct.core.entity.adgeneration;

import java.io.IOException;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import NAdvMachine.Searchqueryrec;
import com.fasterxml.jackson.core.type.TypeReference;
import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bangenproxy.client.BanGenProxyClient;
import ru.yandex.direct.bangenproxy.client.BanGenProxyClientException;
import ru.yandex.direct.bangenproxy.client.model.TextInfoCombinatorics;
import ru.yandex.direct.bangenproxy.client.model.TextInfoUc;
import ru.yandex.direct.bangenproxy.client.model.UrlTextInfo;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.container.AdGroupInfoForKeywordAdd;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordsAddValidationService;
import ru.yandex.direct.core.entity.uac.model.MediaType;
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.richcontent.RichContentClient;
import ru.yandex.direct.richcontent.RichContentClientException;
import ru.yandex.direct.richcontent.model.UrlInfo;
import ru.yandex.direct.searchqueryrecommendation.SearchQueryRecommendationApiUtil;
import ru.yandex.direct.searchqueryrecommendation.SearchQueryRecommendationClient;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.common.db.PpcPropertyNames.CORRECT_REGISTER_FOR_SUGGESTS;
import static ru.yandex.direct.common.db.PpcPropertyNames.DOMAIN_BLACKLIST_FOR_TEXT_RCA;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.errorResult;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.getDomainLowerCaseWithoutWWW;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.getSuccessResultOrDefaultWithWarnings;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.moveErrorsToWarnings;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.successResult;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.updateAdText;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.updateAdTexts;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.BANGEN_PROXY_API_ERROR;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.CAMPAIGN_WITHOUT_HREF;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.DOMAIN_DISABLED;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.EMPTY_SEARCH_QUERY_API_RESPONSE;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.RICH_CONTENT_API_ERROR;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.RICH_CONTENT_API_USE_ZORA;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.SEARCH_QUERY_API_ERROR;
import static ru.yandex.direct.libs.keywordutils.parser.KeywordParser.parseWithMinuses;
import static ru.yandex.direct.utils.CommonUtils.isValidId;

@Lazy
@Service
@ParametersAreNonnullByDefault
public class KeywordGenerationService {

    private static final String TITLE_SOURCE_KEY = "TitleSource";
    private static final String TEXT_SOURCE_KEY = "TextSource";
    private static final String URL_SOURCE_KEY = "UrlSource";
    private static final String SOURCE_URL = "Url";
    private static final String SOURCE_CAMPAIGN = "Campaign";

    private static final String URL_KEY = "Url";
    private static final String BANNER_URL_KEY = "BannerURL";
    private static final String TITLE_KEY = "BannerTitle";
    private static final String TEXT_KEY = "BannerText";
    private static final String REGION_KEY = "RegionIds";
    private static final String MINUS_REGION_KEY = "MinusRegionIds";

    private static final Logger logger = LoggerFactory.getLogger(KeywordGenerationService.class);

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final RichContentClient richContentClient;
    private final BanGenProxyClient banGenProxyClient;
    private final SearchQueryRecommendationClient searchQueryRecommendationClient;
    private final KeywordsAddValidationService keywordsAddValidationService;
    private final UacCampaignServiceHolder uacCampaignServiceHolder;
    private final PpcProperty<Boolean> correctRegisterProperty;
    private final PpcProperty<List<String>> domainBlacklistProperty;

    @SuppressWarnings("checkstyle:ParameterNumber")
    @Autowired
    public KeywordGenerationService(ShardHelper shardHelper,
                                    PpcPropertiesSupport ppcPropertiesSupport,
                                    CampaignRepository campaignRepository,
                                    RichContentClient richContentClient,
                                    BanGenProxyClient banGenProxyClient,
                                    SearchQueryRecommendationClient searchQueryRecommendationClient,
                                    KeywordsAddValidationService keywordsAddValidationService,
                                    UacCampaignServiceHolder uacCampaignServiceHolder) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.richContentClient = richContentClient;
        this.banGenProxyClient = banGenProxyClient;
        this.searchQueryRecommendationClient = searchQueryRecommendationClient;
        this.keywordsAddValidationService = keywordsAddValidationService;
        this.uacCampaignServiceHolder = uacCampaignServiceHolder;
        correctRegisterProperty = ppcPropertiesSupport.get(CORRECT_REGISTER_FOR_SUGGESTS, Duration.ofMinutes(5));
        domainBlacklistProperty = ppcPropertiesSupport.get(DOMAIN_BLACKLIST_FOR_TEXT_RCA, Duration.ofMinutes(5));
    }

    private String getUrlByCampaignId(@Nullable Long campaignId) {
        if (!isValidId(campaignId)) {
            return null;
        }
        int shard = shardHelper.getShardByCampaignId(campaignId);
        if (shard == ShardSupport.NO_SHARD) {
            return null;
        }
        return campaignRepository.getHrefCampAdditionalData(shard, List.of(campaignId)).get(campaignId);
    }

    public Result<UrlInfo> getUrlInfo(String url,
                                      Long campaignId,
                                      Map<String, Object> additionalInfo) {
        String campaignUrl = StringUtils.isNotBlank(url) ? url : getUrlByCampaignId(campaignId);
        if (StringUtils.isEmpty(campaignUrl)) {
            return successResult(null, CAMPAIGN_WITHOUT_HREF);
        }
        additionalInfo.put(URL_KEY, campaignUrl);
        Result<UrlInfo> urlInfoResult = getUrlInfoInternal(campaignUrl);
        if (urlInfoResult.getResult() != null && correctRegisterProperty.getOrDefault(false)) {
            UrlInfo urlInfo = urlInfoResult.getResult();
            urlInfo.setTitle(updateAdText(urlInfo.getTitle()));
            urlInfo.setDescription(updateAdText(urlInfo.getDescription()));
        }
        return getSuccessResultOrDefaultWithWarnings(urlInfoResult, null);
    }

    public Result<TextInfoCombinatorics> getMultipleTextSuggestionsByUrl(String url,
                                                                         @Nullable Long campaignId,
                                                                         @Nullable String userText,
                                                                         Map<String, Object> additionalInfo) {
        var textInfoResult = getTextSuggestionsByUrl(url, campaignId, userText, additionalInfo, false);
        TextInfoCombinatorics textInfo = (TextInfoCombinatorics) textInfoResult.getResult();
        Result<TextInfoCombinatorics> finalResult =
                new Result<>(textInfo, textInfoResult.getValidationResult(), textInfoResult.getState());
        if (textInfo != null && correctRegisterProperty.getOrDefault(false)) {
            textInfo.setTitle(updateAdTexts(textInfo.getTitle()));
            textInfo.setBody(updateAdTexts(textInfo.getBody()));
        }
        return getSuccessResultOrDefaultWithWarnings(finalResult, null);
    }

    public Result<TextInfoUc> getSingleTextSuggestionsByUrl(String url,
                                                            Long campaignId,
                                                            Map<String, Object> additionalInfo) {
        Result<UrlTextInfo> textInfoResult = getTextSuggestionsByUrl(url, campaignId, null, additionalInfo, true);
        TextInfoUc textInfo = (TextInfoUc) textInfoResult.getResult();
        Result<TextInfoUc> finalResult =
                new Result<>(textInfo, textInfoResult.getValidationResult(), textInfoResult.getState());
        if (textInfo != null && correctRegisterProperty.getOrDefault(false)) {
            textInfo.setTitle(updateAdText(textInfo.getTitle()));
            textInfo.setBody(updateAdText(textInfo.getBody()));
        }
        return getSuccessResultOrDefaultWithWarnings(finalResult, null);
    }

    private Result<UrlTextInfo> getTextSuggestionsByUrl(String url,
                                                        @Nullable Long campaignId,
                                                        @Nullable String userText,
                                                        Map<String, Object> additionalInfo,
                                                        boolean isSingle) {
        String campaignUrl = StringUtils.isNotBlank(url) ? url : getUrlByCampaignId(campaignId);
        if (StringUtils.isEmpty(campaignUrl)) {
            return successResult(null, CAMPAIGN_WITHOUT_HREF);
        }
        additionalInfo.put(URL_KEY, campaignUrl);
        var existingTitlesAndBodies = getExistingTitlesAndBodies(campaignId);
        List<String> existingTitles = existingTitlesAndBodies.getOrDefault(MediaType.TITLE, emptyList());
        List<String> existingBodies = existingTitlesAndBodies.getOrDefault(MediaType.TEXT, emptyList());

        return getTextSuggestionsByUrlInternal(campaignUrl, existingTitles, existingBodies, userText, isSingle);
    }

    private Result<UrlTextInfo> getTextSuggestionsByUrlInternal(String url,
                                                                List<String> existingTitles,
                                                                List<String> existingBodies,
                                                                @Nullable String userText,
                                                                boolean isSingle) {
        try {
            return Result.successful(isSingle
                    ? banGenProxyClient.getUrlInfoForUc(url, existingTitles, existingBodies)
                    : banGenProxyClient.getUrlInfoForCombinatorics(url, userText, existingTitles, existingBodies));
        } catch (BanGenProxyClientException e) {
            logger.error("Failed to generate title and snippet suggestions", e);
            return errorResult(BANGEN_PROXY_API_ERROR);
        }
    }

    //возвращает все заголовки и тесты данной кампании, кроме удаленных
    private Map<MediaType, List<String>> getExistingTitlesAndBodies(@Nullable Long directCampaignId) {
        if (!isValidId(directCampaignId)) {
            return emptyMap();
        }

        return uacCampaignServiceHolder.getUacCampaignServiceForDirectCampaignId(directCampaignId)
                .getTextContentsByMediaType(directCampaignId);
    }

    private Result<UrlInfo> getUrlInfoInternal(String url) {
        if (isDomainDisabled(url)) {
            return errorResult(DOMAIN_DISABLED);
        }

        UrlInfo result = null;
        Set<DefectId> defects = new HashSet<DefectId>();
        for (boolean crawl : List.of(false, true)) { // Первый запрос быстрый, второй - медленный
            UrlInfo iterResult = null;
            try {
                iterResult = richContentClient.getUrlInfo(url, crawl);
            } catch (RichContentClientException e) {
                logger.error("Failed to generate title and snippet", e);
                defects.add(RICH_CONTENT_API_ERROR);
            }
            if (urlInfoContainTitleAndSnippet(iterResult)) {
                return successResult(iterResult, defects); // Если ответ содержит тайтл и текст, то можно не продолжать
            } else {
                defects.add(RICH_CONTENT_API_USE_ZORA); // Флаг про использование второго запроса для сбора статистики
            }
            if (result == null) {
                result = iterResult; // Если предыдущего результата нет, то перезатираем его.
            } else if (iterResult != null) { // Иначе начинаем их мержить с приоритетом на медленный запрос
                if (iterResult.getTitle() == null) {
                    iterResult.setTitle(result.getTitle());
                }
                if (iterResult.getDescription() == null) {
                    iterResult.setDescription(result.getDescription());
                }
                result = iterResult;
            }
        }
        return result == null ? errorResult(defects) : successResult(result, defects);
    }

    boolean isDomainDisabled(String url) {
        String domain = getDomainLowerCaseWithoutWWW(url);
        return domainBlacklistProperty.getOrDefault(Collections.emptyList()).contains(domain);
    }

    private boolean urlInfoContainTitleAndSnippet(UrlInfo urlInfo) {
        return urlInfo != null &&
                urlInfo.getTitle() != null &&
                urlInfo.getDescription() != null;
    }

    public Result<Collection<Keyword>> generateKeywords(
            @Nullable String url, @Nullable String title,
            @Nullable String body, List<Long> regionIds) {
        logger.info("Try generate keywords by regions=[{}] ; url='{}' ; title='{}' ; body='{}'",
                StreamEx.of(regionIds).joining(", "), url, title, body);

        try {
            var request = SearchQueryRecommendationApiUtil.buildRequest(url, title, body, regionIds);
            var response = searchQueryRecommendationClient.getSearchQueryRecommendations(request);

            List<Keyword> keywords = StreamEx.of(response.getCandidatesList())
                    .flatMap(this::flattenKeywords)
                    .map(phrase -> new Keyword()
                            .withAutobudgetPriority(5)
                            .withPhrase(phrase))
                    .toList();
            keywords = filterInvalidKeywords(keywords);
            return successResult(keywords, keywords.isEmpty() ? EMPTY_SEARCH_QUERY_API_RESPONSE : null);
        } catch (IOException ex) {
            logger.error("Failed call Search Query Recommendation API", ex);
            return errorResult(SEARCH_QUERY_API_ERROR);
        }
    }

    public Result<Collection<Keyword>> generateKeywords(Searchqueryrec.TSearchQueryRecRequest request,
                                                        Map<String, Object> additionalInfo) {
        logger.info("Try generate keywords by request={}", request.toString());
        if (StringUtils.isNotBlank(request.getBannerURL()) &&
                (StringUtils.isBlank(request.getBannerTitle()) || StringUtils.isBlank(request.getBannerText()))) {
            Result<UrlInfo> urlInfoResult = getUrlInfoInternal(request.getBannerURL());
            if (urlInfoResult.isSuccessful()) {
                var  reqBuilder = request.toBuilder();
                UrlInfo urlInfo = urlInfoResult.getResult();
                if (StringUtils.isNotBlank(urlInfo.getTitle()) && StringUtils.isBlank(request.getBannerTitle())) {
                    reqBuilder.setBannerTitle(urlInfo.getTitle());
                    additionalInfo.put(TITLE_SOURCE_KEY, SOURCE_URL);
                }
                if (StringUtils.isNotBlank(urlInfo.getDescription()) && StringUtils.isBlank(request.getBannerText())) {
                    reqBuilder.setBannerText(urlInfo.getDescription());
                    additionalInfo.put(TEXT_SOURCE_KEY, SOURCE_URL);
                }
                request = reqBuilder.build();
            } else if (StringUtils.isBlank(request.getBannerTitle()) && StringUtils.isBlank(request.getBannerText())) {
                return Result.successful(Collections.emptyList(), moveErrorsToWarnings(urlInfoResult.getValidationResult()));
            }
        } else if (StringUtils.isBlank(request.getBannerTitle()) && StringUtils.isBlank(request.getBannerText())) {
            return successResult(Collections.emptyList(), CAMPAIGN_WITHOUT_HREF);
        }

        try {
            additionalInfo.putAll(JsonUtils.fromJson(JsonUtils.PROTO_PRINTER.print(request),
                    new TypeReference<Map<String, Object>>() {
                    })
            );
        } catch (Exception e) {
            logger.error("cant log to additional data " + e.getMessage());
        }

        Result<Collection<Keyword>> keywordsResult;
        try {
            var response = searchQueryRecommendationClient.getSearchQueryRecommendations(request);

            List<Keyword> keywords = StreamEx.of(response.getCandidatesList())
                    .flatMap(this::flattenKeywords)
                    .map(phrase -> new Keyword()
                            .withAutobudgetPriority(5)
                            .withPhrase(phrase))
                    .toList();
            keywords = filterInvalidKeywords(keywords);
            keywordsResult = successResult(keywords, keywords.isEmpty() ? EMPTY_SEARCH_QUERY_API_RESPONSE : null);
        } catch (IOException ex) {
            logger.error("Failed call Search Query Recommendation API", ex);
            keywordsResult = errorResult(SEARCH_QUERY_API_ERROR);
        }
        return getSuccessResultOrDefaultWithWarnings(keywordsResult, Collections.emptyList());
    }

    private Stream<String> flattenKeywords(Searchqueryrec.TSearchQueryRecResponse.TScoredSearchQueryProfile profile) {
        return Stream.concat(
                Stream.of(profile.getSearchQueryProfile().getSearchQuery()),
                profile.getSubCandidatesList().stream().flatMap(this::flattenKeywords)
        );
    }

    private List<Keyword> filterInvalidKeywords(List<Keyword> keywords) {
        ValidationResult<List<Keyword>, Defect> vr = null;
        boolean needPrevalidate = true;
        while (needPrevalidate) {
            vr = keywordsAddValidationService.preValidate(keywords, false);
            needPrevalidate = removeKeywordsWithErrors(keywords, vr);
        }

        var internalKeywords = IntStreamEx.range(0, keywords.size())
                .mapToEntry(i -> i, keywords::get)
                .mapValues(k -> new InternalKeyword(k, parseWithMinuses(k.getPhrase())))
                .toMap();
        var adGroupInfo = IntStreamEx.range(0, keywords.size())
                .mapToEntry(i -> i, keywords::get)
                .mapToValue((i, k) -> new AdGroupInfoForKeywordAdd(i, -1L, AdGroupType.BASE))
                .toMap();
        keywordsAddValidationService
                .validateWithNonexistentAdGroups(vr, adGroupInfo, emptyMap(), internalKeywords, false);
        removeKeywordsWithErrors(keywords, vr);

        return keywords;
    }

    /**
     *
     * @param keywords фразы
     * @param validationResult результат валидации
     * @return true если хотя бы одна фраза была удалена
     */
    private static boolean removeKeywordsWithErrors(List<Keyword> keywords, ValidationResult<List<Keyword>, Defect> validationResult) {
        if (validationResult != null && validationResult.hasAnyErrors()) {
            EntryStream.of(validationResult.getSubResults())
                    .filterValues(ValidationResult::hasAnyErrors)
                    .keys()
                    .filter(pathNode -> pathNode instanceof PathNode.Index)
                    .map(pathNode -> ((PathNode.Index) pathNode).getIndex())
                    .reverseSorted()
                    .forEach(index -> keywords.remove(index.intValue()));
            return true;
        }
        return false;
    }
}
