package ru.yandex.direct.core.entity.adgeneration;

import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgeneration.model.RegionSuggest;
import ru.yandex.direct.core.entity.adgeneration.region.AbstractRegionSource;
import ru.yandex.direct.core.entity.adgeneration.region.InputContainer;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.DefectId;

import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.common.db.PpcPropertyNames.SUGGEST_REGION_SOURCES;
import static ru.yandex.direct.common.db.PpcPropertyNames.SUGGEST_REGION_TYPES;
import static ru.yandex.direct.common.db.PpcPropertyNames.SUGGEST_REGION_WEIGHT_MULTIPLIER_BY_SOURCE;
import static ru.yandex.direct.common.db.PpcPropertyNames.SUGGEST_REGION_WEIGHT_MULTIPLIER_BY_TYPE_AND_SOURCE;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.errorResult;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.successResult;
import static ru.yandex.direct.core.entity.adgeneration.region.RegionByDefaultSource.DEFAULT_SOURCE;
import static ru.yandex.direct.core.entity.adgeneration.region.RegionByOperatorSource.OPERATOR_SOURCE;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;

@Service
@ParametersAreNonnullByDefault
public class RegionGenerationService {

    private static final Logger logger = LoggerFactory.getLogger(RegionGenerationService.class);

    private static final List<String> DEFAULT_SUGGEST_SOURCES = List.of(OPERATOR_SOURCE, DEFAULT_SOURCE);

    private static final List<Integer> DEFAULT_SUGGEST_REGION_TYPES = List.of(
            Region.REGION_TYPE_COUNTRY,
            Region.REGION_TYPE_DISTRICT,
            Region.REGION_TYPE_PROVINCE,
            Region.REGION_TYPE_TOWN
    );

    private static final double DEFAULT_MULTIPLIER = 0.1;
    public static final int MAX_SUGGEST_NUMBER = 20;

    private final ShardHelper shardHelper;
    private Map<String, AbstractRegionSource> regionSources;
    private final PpcProperty<List<String>> enableSourcesProperty;
    private final PpcProperty<List<Integer>> regionTypesProperty;
    private final PpcProperty<Map<String, Double>> multiplierBySource;
    private final PpcProperty<Map<Integer, Map<String, Double>>> multiplierByRegionTypeAndSource;

    @Autowired
    public RegionGenerationService(
            ShardHelper shardHelper,
            PpcPropertiesSupport ppcPropertiesSupport,
            List<? extends AbstractRegionSource> regionSources) {
        this.shardHelper = shardHelper;
        enableSourcesProperty = ppcPropertiesSupport.get(SUGGEST_REGION_SOURCES, Duration.ofMinutes(5));
        regionTypesProperty = ppcPropertiesSupport.get(SUGGEST_REGION_TYPES, Duration.ofMinutes(5));
        multiplierBySource = ppcPropertiesSupport.get(SUGGEST_REGION_WEIGHT_MULTIPLIER_BY_SOURCE, Duration.ofMinutes(5));
        multiplierByRegionTypeAndSource = ppcPropertiesSupport.get(SUGGEST_REGION_WEIGHT_MULTIPLIER_BY_TYPE_AND_SOURCE, Duration.ofMinutes(5));
        this.regionSources = StreamEx.of(regionSources)
                .toMap(AbstractRegionSource::getRegionSourceName, identity());
    }

    /**
     * Если не указан sourceNames, то используются все разрешенные ресурсы и все они вносят вклад в конечный результат.
     * Если указан sourceNames, то рекомендация составляется по первому ресурсу списка, вернувшего непустой результат.
     * @param input - контейнер с входными данными
     * @param additionalInfo - хранилище для сохранения полезных для дебага и логирования
     * @param sourceNames - отсортированный по приоритету список ресурсов
     * @return
     */
    public Result<Collection<RegionSuggest>> generateRegions(
            ClientId clientId, InputContainer input, Map<String, Object> additionalInfo, List<String> sourceNames) {
        Long campaignId = input.get(InputContainer.CAMPAIGN_ID);
        if (campaignId != null) {
            int shard = shardHelper.getShardByCampaignId(campaignId);
            input.put(InputContainer.SHARD, shard);
        }
        Map<Long, RegionSuggest> results = new HashMap<>();
        Set<DefectId> defects = new HashSet<>();
        Map<String, Map<Long, Double>> sourceToRegToWeight = new HashMap<>();
        final boolean useAllEnabledSources = isEmpty(sourceNames);
        if (useAllEnabledSources) {
            sourceNames = enableSourcesProperty.getOrDefault(DEFAULT_SUGGEST_SOURCES);
        }
        for (String sourceName : sourceNames) {
            AbstractRegionSource source = regionSources.get(sourceName);
            if (source == null) {
                logger.warn("There is no region source for name '" + sourceName + "'.");
                continue;
            }
            Result<Collection<RegionSuggest>> res = source
                    .generateRegions(clientId, input, regionTypesProperty.getOrDefault(DEFAULT_SUGGEST_REGION_TYPES));
            defects.addAll(StreamEx.of(res.getWarnings())
                    .append(res.getErrors())
                    .map(d -> d.getDefect().defectId())
                    .toList());
            if (res.isSuccessful()) {
                Map<Long, Double> regToWeight = new HashMap<>();
                StreamEx.of(res.getResult())
                        .forEach(region -> {
                            regToWeight.put(region.getRegionId(), region.getWeight());
                            applyMultiplier(region);
                            RegionSuggest current = results.get(region.getRegionId());
                            if (current == null) {
                                results.put(region.getRegionId(), region);
                            } else {
                                current.merge(region);
                            }
                        });
                sourceToRegToWeight.put(sourceName, regToWeight);
                if (!useAllEnabledSources) {
                    break;
                }
            }
        }
        additionalInfo.put("weights", sourceToRegToWeight);
        return results.isEmpty() && !defects.isEmpty() ?
                errorResult(defects) :
                successResult(sortSuggests(results.values()), defects);
    }

    /**
     * Для каждого типа региона, с самого маленького (город, область, округ, страна), выбираются от 1 до 3 наилучших
     * Менее 3 может быть выбрано, если третий уступает по качеству первому своего типа или следующего типа вдвое
     * Далее все остальные регионы идут вперемешку отсортированно по качеству рекомендации до 20 в сумме
     * @param suggests список рекомендаций
     * @return рекомендации в нужном порядке
     */
    private List<RegionSuggest> sortSuggests(Collection<RegionSuggest> suggests) {
        List<RegionSuggest> sortedSuggests = StreamEx.of(suggests)
                .sortedByDouble(regionSuggest -> -regionSuggest.getWeight())
                .toList();
        List<Integer> sortedTypes = StreamEx.of(regionTypesProperty.getOrDefault(DEFAULT_SUGGEST_REGION_TYPES))
                .reverseSorted()
                .toList();
        Map<Integer, Integer> typeToNextType = new HashMap<>();
        {
            Integer prev = null;
            for (Integer type : sortedTypes) {
                if (prev != null) {
                    typeToNextType.put(prev, type);
                }
                prev = type;
            }
        }
        Map<Integer, List<RegionSuggest>> typeToSortedSuggests = StreamEx.of(sortedTypes)
                .toMap(type -> StreamEx.of(sortedSuggests)
                        .filter(suggest -> suggest.getRegionType() == type)
                        .toList()
                );
        Map<Integer, Double> typeToBestWeight = EntryStream.of(typeToSortedSuggests)
                .mapValues(list -> list.isEmpty() ? 0 : list.iterator().next().getWeight())
                .toMap();
        List<RegionSuggest> topSuggests = EntryStream.of(typeToSortedSuggests)
                .mapToValue((type, list) -> {
                    Integer nextType = typeToNextType.get(type);
                    double limit1 = typeToBestWeight.get(type);
                    double limit2 = nextType == null ? 0 : typeToBestWeight.get(nextType) / 2;
                    double limit = Math.min(limit1, Math.max(limit1 / 2, limit2));
                    return StreamEx.of(list)
                            .takeWhile(suggest -> suggest.getWeight() >= limit)
                            .limit(3) // не более 3 регионов одного типа вне очереди
                            .toList();
                })
                .values()
                .flatMap(list -> StreamEx.of(list))
                .sortedByDouble(suggest -> -suggest.getRegionType() - suggest.getWeight())
                .toList();
        sortedSuggests.removeAll(topSuggests);
        return StreamEx.of(topSuggests)
                .append(sortedSuggests)
                .limit(MAX_SUGGEST_NUMBER)
                .toList();
    }

    private void applyMultiplier(RegionSuggest regionSuggest) {
        int type = regionSuggest.getRegionType();
        String source = regionSuggest.getSources().iterator().next();
        Double multiplier = multiplierByRegionTypeAndSource
                .getOrDefault(emptyMap())
                .getOrDefault(type, emptyMap())
                .get(source);
        if (multiplier == null) {
            multiplier = multiplierBySource.getOrDefault(emptyMap()).getOrDefault(source, DEFAULT_MULTIPLIER);
        }
        regionSuggest.multiWeight(multiplier);
    }
}
