package ru.yandex.direct.core.entity.adgeneration;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record3;
import org.jooq.SelectConditionStep;
import org.jooq.types.ULong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgeneration.model.SitelinkSuggest;
import ru.yandex.direct.grid.schema.yt.tables.ImportFromBnoSitelinks;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytcomponents.service.SitelinksImportDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static ru.yandex.direct.common.db.PpcPropertyNames.DOMAIN_BLACKLIST_FOR_SITELINK_GENERATION;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.compareHostAndPath;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.getDomainLowerCaseWithoutWWW;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.getDomainWithPath;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.setHttpsToUrl;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.sortAndSubListSitelinks;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.successResult;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.updateSitelinkText;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.DOMAIN_DISABLED;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.SITELINKS_FOUND_FOR_PARENT_PAGE;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.SITELINKS_FOUND_ONLY_FOR_COMMON_PAGE;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.SITELINKS_NOT_FOUND;
import static ru.yandex.direct.grid.schema.yt.Tables.IMPORT_FROM_BNO_SITELINKS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;

@Service
@ParametersAreNonnullByDefault
public class SitelinkGenerationService {

    private static final Logger logger = LoggerFactory.getLogger(SitelinkGenerationService.class);

    // Если у подобранной страницы подстраниц более чем LIMIT, то не предлагаем быстрые ссылки этой страница
    private static final int SUB_PAGES_LIMIT = 30;

    private static final ImportFromBnoSitelinks IMPORT_SITELINKS = IMPORT_FROM_BNO_SITELINKS.as("SITELINKS");
    private static final Field<String> URL = aliased(IMPORT_SITELINKS.URL);
    private static final Field<ULong> SUP_PAGES = aliased(IMPORT_SITELINKS.SUB_PAGES);
    private static final Field<String> SITELINKS = aliased(IMPORT_SITELINKS.SITELINKS);

    private final SitelinksImportDynContextProvider sitelinksImportDynContextProvider;
    private final PpcProperty<List<String>> domainBlacklistProperty;

    @Autowired
    public SitelinkGenerationService(
            SitelinksImportDynContextProvider sitelinksImportDynContextProvider,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.sitelinksImportDynContextProvider = sitelinksImportDynContextProvider;
        domainBlacklistProperty = ppcPropertiesSupport.get(DOMAIN_BLACKLIST_FOR_SITELINK_GENERATION, Duration.ofMinutes(5));
    }

    public Result<Collection<SitelinkSuggest>> generateSitelinks(String url, Map<String, Object> additionalInfo) {
        additionalInfo.put("requestUrl", url);

        String domain = getDomainLowerCaseWithoutWWW(url);
        if (domainBlacklistProperty.getOrDefault(Collections.emptyList()).contains(domain)) {
            return successResult(Collections.emptyList(), DOMAIN_DISABLED);
        }
        SelectConditionStep<Record3<String, ULong, String>> query = YtDSL.ytContext()
                .select(URL, SUP_PAGES, SITELINKS)
                .from(IMPORT_SITELINKS)
                .where(IMPORT_SITELINKS.DOMAIN.eq(domain));

        UnversionedRowset rows =
                sitelinksImportDynContextProvider.getContext().executeTimeoutSafeSelect(query);
        TableSchema schema = rows.getSchema();

        String formatUrl = getDomainWithPath(url);
        Optional<SitelinksData> result = StreamEx.of(rows.getRows())
                .map(row -> new SitelinksData(row, schema))
                .filter(data -> {
                    String formatCandidateUrl = getDomainWithPath(data.url);
                    return formatUrl.equals(formatCandidateUrl) ||
                            formatUrl.startsWith(formatCandidateUrl) &&
                                    formatUrl.charAt(formatCandidateUrl.length()) == '/';
                })
                .sortedByInt(data -> -getDomainWithPath(data.url).length())
                .findFirst();
        if (result.isEmpty()) {
            return successResult(Collections.emptyList(), SITELINKS_NOT_FOUND);
        }
        SitelinksData sitelinksData = result.get();
        additionalInfo.put("subPages", sitelinksData.subPages);
        additionalInfo.put("foundUrl", sitelinksData.url);
        boolean sitelinksByParentPage = !compareHostAndPath(url, sitelinksData.url);
        if (sitelinksByParentPage && sitelinksData.subPages > SUB_PAGES_LIMIT) {
            return successResult(Collections.emptyList(), SITELINKS_FOUND_ONLY_FOR_COMMON_PAGE);
        }
        return successResult(sortAndSubListSitelinks(sitelinksData.getSitelinks(url.startsWith("https://"))),
                sitelinksByParentPage ? SITELINKS_FOUND_FOR_PARENT_PAGE : null);
    }

    private static class SitelinksData {
        public static final TypeReference<List<List<String>>> SITELINKS_DATA_TYPE = new TypeReference<>() {
        };

        private final String url;
        private final String sitelinks;
        private final Long subPages;

        private SitelinksData(UnversionedRow row, TableSchema schema) {
            url = stringValueGetter(schema, URL).apply(row);
            subPages = longValueGetter(schema, SUP_PAGES).apply(row);
            sitelinks = stringValueGetter(schema, SITELINKS).apply(row);
        }

        /**
         * Возвращает быстрые ссылки с откорректированными текстами и ссылками
         * @param useHttps флаг, что нужно подменять http на https.
         * @return Список быстрых ссылок
         */
        private List<SitelinkSuggest> getSitelinks(boolean useHttps) {
            try {
                List<List<String>> list = JsonUtils.fromJson(sitelinks, SITELINKS_DATA_TYPE);
                return StreamEx.of(list)
                        .map(l -> new SitelinkSuggest(
                                useHttps ? setHttpsToUrl(l.get(1)) : l.get(1),
                                updateSitelinkText(l.get(0))))
                        .toList();
            } catch (RuntimeException ex) {
                logger.error("Can't parse sitelinks " + sitelinks, ex);
                throw ex;
            }
        }
    }
}
