package ru.yandex.direct.core.entity.adgeneration;

import java.time.ZoneId;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;

import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.DEFAULT_TIMEZONE;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;

@Service
@ParametersAreNonnullByDefault
public class TimezoneGenerationService {
    private static final Logger logger = LoggerFactory.getLogger(TimezoneGenerationService.class);

    private final GeoBaseHelper geoBaseHelper;
    private final GeoTimezoneRepository geoTimezoneRepository;
    private final LoadingCache<Long, Long> regionToTimezone;

    public TimezoneGenerationService(GeoBaseHelper geoBaseHelper,
                                     GeoTimezoneRepository geoTimezoneRepository) {
        this.geoBaseHelper = geoBaseHelper;
        this.geoTimezoneRepository = geoTimezoneRepository;
        this.regionToTimezone = CacheBuilder.newBuilder()
                        .expireAfterWrite(1, TimeUnit.DAYS)
                        .build(CacheLoader.from(this::getTimezoneIdByRegionId));
    }

    public Long generateTimezoneId(@Nullable Long regionId) {
        if (regionId == null) {
            return DEFAULT_TIMEZONE;
        }
        return regionToTimezone.getUnchecked(regionId);
    }

    private Long getTimezoneIdByRegionId(Long regionId) {
        String timezone = geoBaseHelper.getTimezoneByRegionId(regionId);
        if (StringUtils.isNotEmpty(timezone)) {
            GeoTimezone geoTimezone = getGeoTimezone(regionId, timezone);
            if (geoTimezone != null) {
                Long timezoneId = geoTimezone.getTimezoneId();
                logger.info("TimezoneId for regionId={} and timezone={}: {}", regionId, timezone, timezoneId);
                return timezoneId;
            }
        }

        long chiefRegionId = geoBaseHelper.getChiefRegionId(regionId);
        if (chiefRegionId != GLOBAL_REGION_ID) {
            timezone = geoBaseHelper.getTimezoneByRegionId(chiefRegionId);
            if (StringUtils.isNotEmpty(timezone)) {
                GeoTimezone geoTimezone = getGeoTimezone(chiefRegionId, timezone);
                if (geoTimezone != null) {
                    Long timezoneId = geoTimezone.getTimezoneId();
                    logger.info("TimezoneId for regionId={}, chiefRegionId={} and timezone={}: {}",
                            regionId, chiefRegionId, timezone, timezoneId);
                    return timezoneId;
                }
            }
        }
        logger.warn("Timezone not found for regionId={}, chiefRegionId={} and timezone={}",
                regionId, chiefRegionId, timezone);
        return DEFAULT_TIMEZONE;
    }

    @Nullable
    private GeoTimezone getGeoTimezone(Long regionId, String timezone) {
        int countryId = geoBaseHelper.getCountryId(regionId);
        return geoTimezoneRepository.getByCountryIdAndTimezone((long) countryId, ZoneId.of(timezone));
    }
}
