package ru.yandex.direct.core.entity.adgeneration;

import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jsoup.Jsoup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bangenproxy.client.BanGenProxyClient;
import ru.yandex.direct.bangenproxy.client.BanGenProxyClientException;
import ru.yandex.direct.bangenproxy.client.model.BannerInfo;
import ru.yandex.direct.bangenproxy.client.model.TextInfoCombinatorics;
import ru.yandex.direct.bangenproxy.client.zenmeta.ZenMetaInfoClient;
import ru.yandex.direct.bangenproxy.client.zenmeta.ZenMetaInfoClientException;
import ru.yandex.direct.bangenproxy.client.zenmeta.model.ZenMetaInfo;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgeneration.model.TextSuggest;
import ru.yandex.direct.core.entity.adgeneration.model.ZenCounter;
import ru.yandex.direct.core.entity.adgeneration.model.ZenSuggest;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.common.db.PpcPropertyNames.CORRECT_REGISTER_FOR_SUGGESTS;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.updateAdTexts;

@Lazy
@Service
@ParametersAreNonnullByDefault
public final class ZenMetaInfoService {
    private static final Logger logger = LoggerFactory.getLogger(ZenMetaInfoService.class);

    private static final int IMAGES_LIMIT = 10;
    private static final int TITLES_LIMIT = 5;
    private static final int BODIES_LIMIT = 3;
    private static final int TITLE_LENGTH_LIMIT = 56;
    private static final int BODY_LENGTH_LIMIT = 81;

    private final BanGenProxyClient banGenProxyClient;
    private final ZenMetaInfoClient zenMetaInfoClient;
    private final MetrikaGoalsService metrikaGoalsService;
    private final PpcProperty<Boolean> correctRegisterProperty;

    @Autowired
    public ZenMetaInfoService(PpcPropertiesSupport ppcPropertiesSupport,
                              BanGenProxyClient banGenProxyClient,
                              ZenMetaInfoClient zenMetaInfoClient,
                              MetrikaGoalsService metrikaGoalsService) {
        this.banGenProxyClient = banGenProxyClient;
        this.zenMetaInfoClient = zenMetaInfoClient;
        this.metrikaGoalsService = metrikaGoalsService;
        correctRegisterProperty = ppcPropertiesSupport.get(CORRECT_REGISTER_FOR_SUGGESTS, Duration.ofMinutes(5));
    }

    public Optional<ZenSuggest> getZenMetaInfoWithGeneratedTextByUrl(String url) {
        if (!ZenDomain.isZenDomain(url)) {
            logger.error(String.format("Url %s not pass a zen-url validation", url));
            return Optional.empty();
        }
        try {
            final var zenMetaInfo = zenMetaInfoClient.getZenMetaInfoByUrl(url);
            TextInfoCombinatorics generatedText = new TextInfoCombinatorics();
            List<String> images = List.of();
            ZenCounter counter = new ZenCounter();
            if (Objects.nonNull(zenMetaInfo.getTitle())
                    && (Objects.nonNull(zenMetaInfo.getContent()) || Objects.nonNull(zenMetaInfo.getSnippet()))) {
                final var generationResult = generateTextByZenMeta(zenMetaInfo);
                if (generationResult.isPresent()) {
                    generatedText = generationResult.get();
                }
            }
            if (Objects.nonNull(zenMetaInfo.getImageUrls())) {
                images = applyImagesLimit(zenMetaInfo.getImageUrls());
            }
            if (Objects.nonNull(zenMetaInfo.getPublisherVacuumCounterId())) {
                counter = getZenCounterByCounterId(zenMetaInfo.getPublisherVacuumCounterId());
            }
            return Optional.of(new ZenSuggest(zenMetaInfo.getPublisherId(), zenMetaInfo.getPublisherItemId(),
                    TextSuggest.fromTextInfoCombinatorics(generatedText), images, counter));
        } catch (ZenMetaInfoClientException e) {
            logger.error("Failed to get zen meta info", e);
            return Optional.empty();
        }
    }

    public Optional<ZenCounter> getZenCounterByUrl(String url) {
        if (!ZenDomain.isZenDomain(url)) {
            logger.error(String.format("Url %s not pass a zen-url validation", url));
            return Optional.empty();
        }
        try {
            final var zenMetaInfo = zenMetaInfoClient.getZenMetaInfoByUrl(url);
            if (Objects.nonNull(zenMetaInfo.getPublisherVacuumCounterId())) {
                return Optional.of(getZenCounterByCounterId(zenMetaInfo.getPublisherVacuumCounterId()));
            } else {
                logger.info(String.format("Url %s dont has a zen counter", url));
                return Optional.empty();
            }
        } catch (ZenMetaInfoClientException e) {
            logger.error("Failed to get zen meta info", e);
            return Optional.empty();
        }
    }

    @Nullable
    public String getPublisherItemIdByUrl(String url) {
        if (!ZenDomain.isZenDomain(url)) {
            return null;
        }
        try {
            return zenMetaInfoClient.getZenMetaInfoByUrl(url).getPublisherItemId();
        } catch (ZenMetaInfoClientException e) {
            logger.error("Failed to get zen meta info", e);
            return null;
        }
    }

    private Optional<TextInfoCombinatorics> generateTextByZenMeta(ZenMetaInfo zenMetaInfo) {
        try {
            final var textInfo = banGenProxyClient.getUrlInfoForCustom(
                    zenMetaInfo.getTitle(),
                    !isBlank(zenMetaInfo.getContent())
                            ? parseHtmlContent(zenMetaInfo.getContent())
                            : zenMetaInfo.getSnippet());
            List<BannerInfo> banners = textInfo.getBanners();
            List<String> titles = applyTitlesLimit(
                    banners.stream().map(BannerInfo::getTitle).collect(Collectors.toList()));
            List<String> bodies = applyBodiesLimit(
                    banners.stream().map(BannerInfo::getBody).collect(Collectors.toList()));
            if (!titles.isEmpty() && !bodies.isEmpty() &&
                    correctRegisterProperty.getOrDefault(false)) {
                titles = updateAdTexts(titles);
                bodies = updateAdTexts(bodies);
            }
            return Optional.of(new TextInfoCombinatorics(textInfo.getAlgVersion(), textInfo.getRcaTitle(),
                    textInfo.getRcaSnippet(), titles, bodies));
        } catch (BanGenProxyClientException e) {
            logger.error("Failed to generate zen title and snippet suggestions", e);
            return Optional.empty();
        }
    }

    private ZenCounter getZenCounterByCounterId(Long counterId) {
        final var goals = metrikaGoalsService.getGoalsByCounterId(counterId.intValue());
        return new ZenCounter(counterId, StreamEx.of(goals).collect(Collectors.toSet()));
    }

    private List<String> applyImagesLimit(List<String> images) {
        return StreamEx.ofReversed(images).limit(IMAGES_LIMIT).toList();
    }

    private List<String> applyTitlesLimit(List<String> titles) {
        return StreamEx.of(titles).distinct().filter((t) -> t.length() <= TITLE_LENGTH_LIMIT)
                .limit(TITLES_LIMIT).toList();
    }

    private List<String> applyBodiesLimit(List<String> bodies) {
        return StreamEx.of(bodies).distinct().filter((t) -> t.length() <= BODY_LENGTH_LIMIT)
                .limit(BODIES_LIMIT).toList();
    }

    private String parseHtmlContent(String content) {
        var doc = Jsoup.parse(content);
        return doc.text();
    }
}
