package ru.yandex.direct.core.entity.adgeneration.region;

import java.util.Collection;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgeneration.model.RegionSuggest;
import ru.yandex.direct.core.entity.region.repository.RegionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;

abstract public class AbstractRegionSource {

    private final RegionRepository regionRepository;
    private final GeoTreeFactory geoTreeFactory;
    private final GeoBaseHelper geoBaseHelper;

    public AbstractRegionSource(
            RegionRepository regionRepository,
            GeoTreeFactory geoTreeFactory,
            GeoBaseHelper geoBaseHelper
    ) {
        this.regionRepository = regionRepository;
        this.geoTreeFactory = geoTreeFactory;
        this.geoBaseHelper = geoBaseHelper;
    }

    /**
     *
     * @param regionId - идентификатор региона
     * @return ближайший по иерархии регион, хранящийся в GeoTree.
     */
    protected Region findRegion(Long regionId) {
        regionId = geoBaseHelper.convertToDirectRegionId(regionId).orElse(null);
        if (regionId == null) {
            return null;
        }
        Region region = getRegion(regionId);

        while (region == null && regionId != null) {
            regionId = regionRepository.getParent(regionId);
            region = getRegion(regionId);
        }
        return region;
    }

    protected Region getRegion(Long regionId) {
        return geoTreeFactory.getGlobalGeoTree().getRegion(regionId);
    }

    public Result<Collection<RegionSuggest>> generateRegions(
            ClientId clientId, InputContainer input, Collection<Integer> suggestedRegionTypes) {
        updateInputContainer(clientId, input);
        Result<Collection<RegionSuggest>> result = generateRegionsInternal(input);
        if (!result.isSuccessful()) {
            return result;
        }
        return Result.successful(
                StreamEx.of(result.getResult())
                        .filter(element -> suggestedRegionTypes.contains(element.getRegionType()))
                        .map(element -> element.addSource(getRegionSourceName()))
                        .toList(),
                result.getValidationResult()
        );
    }

    protected void updateInputContainer(ClientId clientId, InputContainer input) {}

    abstract protected Result<Collection<RegionSuggest>> generateRegionsInternal(InputContainer input);

    abstract public String getRegionSourceName();
}
