package ru.yandex.direct.core.entity.adgeneration.region;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgeneration.model.RegionSuggest;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.metrika.model.MetrikaCounterByDomain;
import ru.yandex.direct.core.entity.region.repository.RegionRepository;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCounter;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCountersAndGoals;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.common.db.PpcPropertyNames.METRIKA_COUNTER_USE_WEAK_RESTRICTIONS_FOR_SUGGESTION;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.errorResult;
import static ru.yandex.direct.core.entity.adgeneration.RegionGenerationService.MAX_SUGGEST_NUMBER;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.COUNTERS_NOT_FOUND;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.METRIKA_API_ERROR;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.toUnicodeDomain;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class RegionByMetrika extends AbstractRegionSource {

    public static final String METRIKA_SOURCE = "metrika";
    public static final long MIN_TRAFFIC = 100L;

    private final MetrikaClient metrikaClient;
    private final TurboLandingService turboLandingService;
    private final BannersUrlHelper bannersUrlHelper;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final PpcProperty<Boolean> metrikaCounterUseWeakRestrictions;

    @Autowired
    public RegionByMetrika(
            MetrikaClient metrikaClient,
            TurboLandingService turboLandingService,
            BannersUrlHelper bannersUrlHelper,
            CampMetrikaCountersService campMetrikaCountersService,
            RegionRepository regionRepository,
            GeoTreeFactory geoTreeFactory,
            GeoBaseHelper geoBaseHelper,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(regionRepository, geoTreeFactory, geoBaseHelper);
        this.metrikaClient = metrikaClient;
        this.turboLandingService = turboLandingService;
        this.bannersUrlHelper = bannersUrlHelper;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.metrikaCounterUseWeakRestrictions =
                ppcPropertiesSupport.get(METRIKA_COUNTER_USE_WEAK_RESTRICTIONS_FOR_SUGGESTION, Duration.ofMinutes(5));
    }

    @Override
    public String getRegionSourceName() {
        return METRIKA_SOURCE;
    }

    @Override
    protected Result<Collection<RegionSuggest>> generateRegionsInternal(InputContainer input) {
        Set<Long> counters =  input.get(InputContainer.COUNTER_IDS);
        if (CollectionUtils.isEmpty(counters)) {
            return errorResult(COUNTERS_NOT_FOUND);
        }
        Map<Region, Long> usersByRegion;
        try {
            usersByRegion =
                    EntryStream.of(metrikaClient
                            .getUserNumberByRegions(counters, MAX_SUGGEST_NUMBER + 1, 7))
                            .mapKeys(this::getRegion)
                            .nonNullKeys()
                            .toMap();
        } catch (MetrikaClientException ex) {
            return errorResult(METRIKA_API_ERROR);
        }
        Collection<RegionSuggest> suggests = StreamEx.ofKeys(usersByRegion)
                .map(Region::getType)
                .distinct()
                .map(
                        type -> EntryStream.of(usersByRegion)
                                .filterKeys(region -> region.getType() == type.intValue())
                                .toMap()
                ).filter(map -> !map.isEmpty())
                .map(RegionByMetrika::createRegionSuggest)
                .toFlatList(Function.identity());

        return Result.successful(suggests);
    }

    private static List<RegionSuggest> createRegionSuggest(Map<Region, Long> usersByRegion) {
        List<Region> sortedList = EntryStream.of(usersByRegion)
                .sortedByLong(entry -> -entry.getValue())
                .keys()
                .toList();
        List<RegionSuggest> result = new ArrayList<>(usersByRegion.size());
        Region prev = null;
        long prevUsers = 0;
        long sumUsers = 0;
        for (Region current : sortedList) {
            long users = usersByRegion.get(current);
            sumUsers += users;
            if (prev != null) {
                result.add(new RegionSuggest(prev).multiWeight(1.0 * prevUsers / Math.max(MIN_TRAFFIC, sumUsers)));
            }
            prev = current;
            prevUsers = users;
        }
        result.add(new RegionSuggest(prev).multiWeight(1.0 * prevUsers / Math.max(MIN_TRAFFIC, sumUsers)));
        return result;
    }

    @Override
    protected void updateInputContainer(ClientId clientId, InputContainer input) {
        if (!input.has(InputContainer.COUNTER_IDS)) {
            String url = input.get(InputContainer.URL);
            if (url != null) {
                Set<Long> counterIds;
                if (turboLandingService.isTurboLandingUrl(url)) {
                    counterIds = getTurboLandingMetrikaCounters(url);
                } else {
                    String domain = ifNotNull(bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(url),
                            d -> toUnicodeDomain(d.toLowerCase()));
                    List<MetrikaCounterByDomain> counters =
                            campMetrikaCountersService.getMetrikaCountersByDomain(clientId, domain,
                                    metrikaCounterUseWeakRestrictions.getOrDefault(false));
                    counterIds = listToSet(counters, MetrikaCounterByDomain::getCounterId);
                }
                input.put(InputContainer.COUNTER_IDS, counterIds);
            }
        }
    }

    private Set<Long> getTurboLandingMetrikaCounters(String url) {
        TurboLanding turbo = turboLandingService.externalFindTurboLandingByUrl(url);
        if (turbo == null) {
            return emptySet();
        }
        return StreamEx.of(new TurboLandingMetrikaCountersAndGoals(turbo.getMetrikaCounters()).getCountersWithGoals())
                .map(TurboLandingMetrikaCounter::getId)
                .toSet();
    }
}
