package ru.yandex.direct.core.entity.adgeneration.region;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgeneration.model.RegionSuggest;
import ru.yandex.direct.core.entity.organizations.OrganizationsClientTranslatableException;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.region.repository.RegionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.errorResult;
import static ru.yandex.direct.core.entity.adgeneration.GenerationUtils.successResult;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.CAMPAIGN_WITHOUT_PERMALINK;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.PERMALINK_WITHOUT_GEO;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.PERMALINK_WITHOUT_SUITABLE_GEO;
import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.SPRAV_API_ERROR;

@Service
@ParametersAreNonnullByDefault
public class RegionByOrganizationSource extends AbstractRegionSource {

    public static final String SPRAV_SOURCE = "sprav";

    private final OrganizationRepository organizationRepository;
    private final OrganizationService organizationService;

    @Autowired
    public RegionByOrganizationSource(
            OrganizationRepository organizationRepository,
            OrganizationService organizationService,
            RegionRepository regionRepository,
            GeoTreeFactory geoTreeFactory,
            GeoBaseHelper geoBaseHelper
    ) {
        super(regionRepository, geoTreeFactory, geoBaseHelper);
        this.organizationRepository = organizationRepository;
        this.organizationService = organizationService;
    }

    public String getRegionSourceName() {
        return SPRAV_SOURCE;
    }

    @Override
    protected void updateInputContainer(ClientId clientId, InputContainer input) {
        if (!input.has(InputContainer.PERMALINK_ID)) {
            Integer shard = input.get(InputContainer.SHARD);
            Long campaignId = input.get(InputContainer.CAMPAIGN_ID);
            if (campaignId != null && shard != null) {
                Long permalinkId = organizationRepository
                        .getDefaultPermalinkIdsByCampaignId(shard, List.of(campaignId))
                        .get(campaignId);
                input.put(InputContainer.PERMALINK_ID, permalinkId);
            }
        }
    }

    @Override
    protected Result<Collection<RegionSuggest>> generateRegionsInternal(InputContainer input) {
        Long permalinkId = input.get(InputContainer.PERMALINK_ID);
        if (permalinkId == null) {
            return errorResult(CAMPAIGN_WITHOUT_PERMALINK);
        }
        Long regionId;
        try {
            regionId = organizationService.getRegion(permalinkId);
        } catch (OrganizationsClientTranslatableException ex) {
            return errorResult(SPRAV_API_ERROR);
        }

        Region region = findRegion(regionId);
        if (region == null) {
            return errorResult(PERMALINK_WITHOUT_GEO);
        }

        Collection<RegionSuggest> suggests = new ArrayList<>();
        while (region != null && region.getId() != Region.GLOBAL_REGION_ID) {
            suggests.add(new RegionSuggest(region));
            region = region.getParent();
        }
        return suggests.isEmpty() ? successResult(suggests, PERMALINK_WITHOUT_SUITABLE_GEO) : successResult(suggests);
    }
}
