package ru.yandex.direct.core.entity.adgroup.generation

import NAdvMachine.Searchqueryrec
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService
import ru.yandex.direct.core.entity.banner.model.Banner
import ru.yandex.direct.core.entity.banner.model.BannerWithBody
import ru.yandex.direct.core.entity.banner.model.BannerWithHref
import ru.yandex.direct.core.entity.banner.model.BannerWithTitle
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerAdGroupIdFilter
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMinusKeywords
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.keyword.service.KeywordService
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseValidator
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory
import ru.yandex.direct.core.entity.stopword.service.StopWordService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses
import ru.yandex.direct.libs.keywordutils.parser.KeywordParser
import ru.yandex.direct.result.Result
import ru.yandex.direct.searchqueryrecommendation.ParallelBatchedSearchQueryRecommendationClient
import ru.yandex.direct.searchqueryrecommendation.SearchQuery
import ru.yandex.direct.searchqueryrecommendation.SearchQueryRecommendationApiUtil.buildRequest
import ru.yandex.direct.searchqueryrecommendation.SearchQueryRecommendationApiUtil.queries
import ru.yandex.direct.searchqueryrecommendation.SearchQueryRecommendationClient
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import javax.annotation.ParametersAreNonnullByDefault


@Service
@ParametersAreNonnullByDefault
class AdGroupKeywordRecommendationService @Autowired constructor(searchQueryRecommendationClient: SearchQueryRecommendationClient,
                                                                 val adGroupService: AdGroupService,
                                                                 val keywordService: KeywordService,
                                                                 val bannerRepository: BannerTypedRepository,
                                                                 val stopWordService: StopWordService,
                                                                 val keywordFactory: KeywordWithLemmasFactory,
                                                                 val campaignTypedRepository: CampaignTypedRepository,
                                                                 val shardHelper: ShardHelper,
                                                                 val ppcPropertiesSupport: PpcPropertiesSupport) {

    private val client = ParallelBatchedSearchQueryRecommendationClient(searchQueryRecommendationClient)

    fun recommendedKeywords(clientId: ClientId, input: AdGroupKeywordRecommendationInput): Result<Map<RelevanceMatchCategory, List<String>>> {
        val shard = shardHelper.getShardByClientId(clientId)
        val adGroupId = input.adGroupId
        val adGroup = adGroupService.getAdGroups(clientId, listOf(adGroupId))
            .firstOrNull()
            ?: return Result.broken(ValidationResult.failed(adGroupId, CommonDefects.objectNotFound()))
        val requests = requests(clientId, shard, adGroup, input)

        val keywordsByCategory = client.getSearchQueryRecommendations(requests)
            .mapNotNull { it.getOrNull() }
            .flatMap { queries(it) }
            .groupBy { relevanceMatchCategory(it.category) }
            .mapValues { it.value.sortedByDescending { v -> v.score } }
            .mapValues { it.value.distinct().map { v -> v.query } }

        return Result.successful(keywordsByCategory)
    }

    private fun requests(clientId: ClientId,
                         shard: Int,
                         adGroup: AdGroup,
                         input: AdGroupKeywordRecommendationInput): List<Searchqueryrec.TSearchQueryRecRequest> {

        val bannersContents = bannersContents(shard, input.adGroupId) + bannerContentFromInput(input)
        val keywordsWithMinuses = keywordsWithMinuses(clientId, input)
        val adGroupMinusKeywords = minusKeywords(adGroup, input)
        val campMinusKeywords = campMinusKeywords(shard, adGroup)

        val allMinusKeywords =
            campMinusKeywords +
                adGroupMinusKeywords +
                keywordsWithMinuses.flatMap { kw -> kw.minusKeywords.map { it.toString() } }

        return bannersContents
            .filterNotNull()
            .map {
                buildRequest(
                    it,
                    keywordsWithMinuses.map { it.keyword.toString() },
                    allMinusKeywords,
                    geo(adGroup, input),
                    minusGeo(adGroup, input)
                )
            }
    }

    private fun selectRandomBanners(banners: List<Banner>, limit: Int): List<Banner> = banners.shuffled().take(limit)

    private fun keywordsWithMinuses(clientId: ClientId, input: AdGroupKeywordRecommendationInput) =
        if (input.info != null && input.info is AdGroupInfo) {
            input.info.keywords.map { KeywordParser.parseWithMinuses(it) }
        } else {
            keywordService.getKeywordsByAdGroupIds(clientId, listOf(input.adGroupId))
                .getOrDefault(input.adGroupId, emptyList())
                .map { KeywordParser.parseWithMinuses(it.phrase) }
        }


    private fun geo(adGroup: AdGroup, input: AdGroupKeywordRecommendationInput) =
        ((input.info as? AdGroupInfo)?.regionIds ?: adGroup.geo).filter { it >= 0 }

    private fun minusGeo(adGroup: AdGroup, input: AdGroupKeywordRecommendationInput) =
        ((input.info as? AdGroupInfo)?.regionIds ?: adGroup.geo).filter { it < 0 }.map(Math::abs)

    private fun minusKeywords(adGroup: AdGroup, input: AdGroupKeywordRecommendationInput) =
        (input.info as? AdGroupInfo)?.minusKeywords ?: adGroup.minusKeywords


    private fun buildRequest(bannerContent: BannerContent,
                             keywords: List<String>,
                             minusKeywords: List<String>,
                             geo: List<Long>,
                             minusGeo: List<Long>): Searchqueryrec.TSearchQueryRecRequest {
        val tPhrasesBuilder = Searchqueryrec.TSearchQueryRecRequest.TPhrases.newBuilder()

        if (passKeywords()) tPhrasesBuilder.addAllPlus(keywords)
        if (passMinusWords()) tPhrasesBuilder.addAllMinus(minusKeywords)

        return buildRequest(
            bannerContent.href,
            bannerContent.title,
            bannerContent.body,
            tPhrasesBuilder.build(),
            geo,
            minusGeo
        )
    }

    private fun campMinusKeywords(shard: Int, adGroup: AdGroup): List<String> {
        val campaignWithMinusWords =
            campaignTypedRepository.getSafely(shard, listOf(adGroup.campaignId), CampaignWithMinusKeywords::class.java)
        return campaignWithMinusWords.flatMap { it.minusKeywords }
    }

    private fun bannersContents(shard: Int, adGroupId: Long): List<BannerContent> {
        val banners = bannerRepository.getSafely(
            shard,
            bannerAdGroupIdFilter(listOf(adGroupId)),
            listOf(BannerWithBody::class.java, BannerWithTitle::class.java, BannerWithHref::class.java)
        )
        return selectRandomBanners(banners, 5)
            .map {
                BannerContent(
                    (it as? BannerWithHref)?.href,
                    (it as? BannerWithTitle)?.title,
                    (it as? BannerWithBody)?.body
                )
            }
    }

    private fun bannerContentFromInput(input: AdGroupKeywordRecommendationInput): BannerContent? =
        (input.info as? BannerInfo)?.let {
            BannerContent(null, it.title, it.body)
        }

    private fun isInvalid(keywordsWithMinuses: List<KeywordWithMinuses>, query: String): Boolean =
        keywordsWithMinuses.any {
            val validator = PhraseValidator(stopWordService, keywordFactory, it)
            validator.apply(query).hasAnyErrors()
        }

    private fun passKeywords(): Boolean = ppcPropertiesSupport.get(PpcPropertyNames.KEYWORD_RECOMMENDATION_SERVICE_USE_KEYWORDS).getOrDefault(false)
    private fun passMinusWords(): Boolean = ppcPropertiesSupport.get(PpcPropertyNames.KEYWORD_RECOMMENDATION_SERVICE_USE_MINUS_WORDS).getOrDefault(false)

    companion object {

        private fun relevanceMatchCategory(category: SearchQuery.Companion.Category): RelevanceMatchCategory =
            when (category) {
                is SearchQuery.Companion.Accessory -> RelevanceMatchCategory.accessory_mark
                is SearchQuery.Companion.Alternative -> RelevanceMatchCategory.alternative_mark
                is SearchQuery.Companion.Broader -> RelevanceMatchCategory.broader_mark
                is SearchQuery.Companion.Competitor -> RelevanceMatchCategory.competitor_mark
                is SearchQuery.Companion.Exact -> RelevanceMatchCategory.exact_mark
            }

        data class AdGroupKeywordRecommendationInput(val adGroupId: Long, val info: AdditionalInfo?)

        sealed interface AdditionalInfo

        data class AdGroupInfo(val regionIds: List<Long>, val minusKeywords: List<String>, val keywords: List<String>) : AdditionalInfo

        data class BannerInfo(val title: String?, val body: String?) : AdditionalInfo

        data class BannerContent(val href: String?, val title: String?, val body: String?)
    }
}
