package ru.yandex.direct.core.entity.adgroup.repository;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.SelectConditionStep;
import org.jooq.SelectJoinStep;
import org.jooq.SelectQuery;
import org.jooq.TableField;
import org.jooq.TableLike;
import org.jooq.UpdateConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.ServingStatus;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupAppIconStatus;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupShowsForecast;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStates;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStatus;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithTypeAndGeo;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithUsersSegments;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.adgroup.repository.internal.AdGroupBsTagsRepository;
import ru.yandex.direct.core.entity.adgroup.repository.internal.AdGroupTagsRepository;
import ru.yandex.direct.core.entity.adgroup.repository.typesupport.AdGroupTypeSupportDispatcher;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils;
import ru.yandex.direct.core.entity.adgroup.service.ContentCategoriesTargetingConverter;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AuditoriumGeoSegmentsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ContentCategoriesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YandexUidsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.AdGroupAdditionalTargetingRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bids.container.BidDynamicOpt;
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository;
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupAdditionalTargetingsTargetingType;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsDynamicStatusblgenerated;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsPerformanceStatusblgenerated;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsTextStatusblgenerated;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentOsType;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentStatusiconmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusautobudgetshow;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusshowsforecast;
import ru.yandex.direct.dbschema.ppc.tables.AdgroupBsTags;
import ru.yandex.direct.dbschema.ppc.tables.AdgroupProjectParams;
import ru.yandex.direct.dbschema.ppc.tables.GroupParams;
import ru.yandex.direct.dbschema.ppc.tables.Phrases;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupBsTagsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupProjectParamsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.GroupParamsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.PhrasesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.flatMapping;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.jooq.impl.DSL.count;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.sum;
import static ru.yandex.bolts.collection.Tuple2.tuple;
import static ru.yandex.direct.core.entity.ServingStatus.ELIGIBLE;
import static ru.yandex.direct.core.entity.ServingStatus.RARELY_SERVED;
import static ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings.geoFromDb;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.ContentPromotionAdGroupSupport.MAPPER_FOR_ADGROUPS_CONTENT_PROMOTION;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmBannerAdGroupSupport.ADGROUP_MAPPER_FOR_CPM_BANNER_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmIndoorAdGroupSupport.ADGROUP_MAPPER_FOR_CPM_INDOOR_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmOutdoorAdGroupSupport.ADGROUP_MAPPER_FOR_CPM_OUTDOOR_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmOutdoorAdGroupSupport.pageBlocksFromDb;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmVideoAdGroupSupport.MAPPER_FOR_ADGROUPS_CPM_BANNER;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmVideoAdGroupSupport.MAPPER_FOR_ADGROUPS_CPM_VIDEO;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.CpmYndxFrontpageAdGroupSupport.ADGROUP_MAPPER_FOR_PRICE_SALES_ADGROUP_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.DynamicFeedAdGroupSupport.ADGROUP_MAPPER_FOR_DYNAMIC_FEED_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.DynamicTextAdGroupSupport.ADGROUP_MAPPER_FOR_DYNAMIC_DOMAIN_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.InternalAdGroupSupport.ADGROUP_MAPPER_FOR_INTERNAL_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.MobileContentAdGroupSupport.ADGROUP_MAPPER_FOR_MOBILE_CONTENT_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.PerformanceAdGroupSupport.ADGROUP_MAPPER_FOR_PERFORMANCE_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.TextAdGroupSupport.MAPPER_FOR_TEXT_ADGROUPS_WITH_FEED;
import static ru.yandex.direct.core.entity.mapping.CommonMappings.longListFromDbJsonFormat;
import static ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository.MAPPER_FOR_MOBILE_CONTENT_FIELDS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CONTENT_PROMOTION;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CPM_BANNER;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CPM_VIDEO;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_HYPERGEO_RETARGETINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MINUS_WORDS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_TEXT;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_BS_TAGS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PAGE_TARGETS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PRIORITY;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PROJECT_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.FEEDS;
import static ru.yandex.direct.dbschema.ppc.Tables.GROUP_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.MINUS_WORDS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupPromoactions.ADGROUP_PROMOACTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbutil.SqlUtils.findInSet;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class AdGroupRepository {

    private static final Logger logger = LoggerFactory.getLogger(AdGroupRepository.class);

    private static final Collection<TableField<?, ?>> ALL_TEXT_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_COMMON_FIELDS.getFieldsToRead();
    private static final TableField[] ADGROUP_WITH_TYPE_FIELDS_TO_READ = new TableField[]
            {PHRASES.PID, PHRASES.CID, PHRASES.ADGROUP_TYPE};
    private static final TableField[] SIMPLE_ADGROUP_FIELDS_TO_READ = new TableField[]
            {PHRASES.PID, PHRASES.CID, PHRASES.ADGROUP_TYPE, PHRASES.STATUS_MODERATE, PHRASES.GEO};
    private static final TableField[] ADGROUP_NAME_FIELDS_TO_READ = new TableField[]
            {PHRASES.PID, PHRASES.CID, PHRASES.GROUP_NAME};
    private static final TableField[] SIMPLE_ADGROUP_FIELDS_FOR_BANNER_OPERATION = new TableField[]
            {PHRASES.PID, PHRASES.CID, PHRASES.STATUS_MODERATE, PHRASES.GEO, PHRASES.ADGROUP_TYPE,
                    GROUP_PARAMS.HAS_PHRASEID_HREF};
    private static final Set<Field<?>> TEXT_ADGROUP_WITH_FEED_FIELDS_TO_READ =
            MAPPER_FOR_TEXT_ADGROUPS_WITH_FEED.getFieldsToRead();
    private static final Collection<TableField<?, ?>> DYNAMIC_DOMAIN_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_DYNAMIC_DOMAIN_FIELDS.getFieldsToRead();
    private static final Collection<TableField<?, ?>> DYNAMIC_FEED_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_DYNAMIC_FEED_FIELDS.getFieldsToRead();
    private static final Collection<TableField<?, ?>> PERFORMANCE_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_PERFORMANCE_FIELDS.getFieldsToRead();
    private static final Collection<TableField<?, ?>> MOBILE_CONTENT_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_MOBILE_CONTENT_FIELDS.getFieldsToRead();
    private static final Collection<Field<?>> MOBILE_CONTENT_FIELDS_TO_READ =
            MAPPER_FOR_MOBILE_CONTENT_FIELDS.getFieldsToRead();
    private static final Collection<TableField<?, ?>> CPM_BANNER_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_CPM_BANNER_FIELDS.getFieldsToRead();
    private static final Set<Field<?>> CPM_VIDEO_ADGROUP_FIELDS_TO_READ =
            StreamEx.of(MAPPER_FOR_ADGROUPS_CPM_BANNER.getFieldsToRead())
                    .append(MAPPER_FOR_ADGROUPS_CPM_VIDEO.getFieldsToRead())
                    .toSet();
    private static final Collection<TableField<?, ?>> CPM_OUTDOOR_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_CPM_OUTDOOR_FIELDS.getFieldsToRead();
    private static final Collection<TableField<?, ?>> CPM_INDOOR_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_CPM_INDOOR_FIELDS.getFieldsToRead();
    private static final Collection<Field<?>> CPM_YNDX_FRONTPAGE_PRIORITY_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_PRICE_SALES_ADGROUP_FIELDS.getFieldsToRead();
    private static final Set<Field<?>> CONTENT_PROMOTION_ADGROUP_FIELDS_TO_READ =
            MAPPER_FOR_ADGROUPS_CONTENT_PROMOTION.getFieldsToRead();
    private static final Collection<Field<?>> INTERNAL_ADGROUP_FIELDS_TO_READ =
            ADGROUP_MAPPER_FOR_INTERNAL_FIELDS.getFieldsToRead();

    private static final Set<PhrasesAdgroupType> ADGROUP_TYPES_WITHOUT_MODERATION =
            Set.of(PhrasesAdgroupType.performance, PhrasesAdgroupType.internal);

    private final DslContextProvider ppcDslContextProvider;
    private final AdGroupTagsRepository adGroupTagsRepository;
    private final AdGroupBsTagsRepository adGroupBsTagsRepository;
    private final BidModifierRepository bidModifierRepository;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final UsersSegmentRepository usersSegmentRepository;
    private final AdGroupAdditionalTargetingRepository adGroupAdditionalTargetingRepository;
    private final HyperGeoRepository hyperGeoRepository;
    private final HyperGeoService hyperGeoService;
    private final AdGroupTypeSupportDispatcher adGroupTypeSupportDispatcher;
    private final ShardHelper shardHelper;

    @Autowired
    public AdGroupRepository(DslContextProvider ppcDslContextProvider,
                             AdGroupTagsRepository adGroupTagsRepository,
                             AdGroupBsTagsRepository adGroupBsTagsRepository,
                             BidModifierRepository bidModifierRepository,
                             MinusKeywordsPackRepository minusKeywordsPackRepository,
                             UsersSegmentRepository usersSegmentRepository,
                             AdGroupAdditionalTargetingRepository adGroupAdditionalTargetingRepository,
                             HyperGeoRepository hyperGeoRepository,
                             HyperGeoService hyperGeoService,
                             AdGroupTypeSupportDispatcher adGroupTypeSupportDispatcher,
                             ShardHelper shardHelper) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.adGroupTagsRepository = adGroupTagsRepository;
        this.adGroupBsTagsRepository = adGroupBsTagsRepository;
        this.bidModifierRepository = bidModifierRepository;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.usersSegmentRepository = usersSegmentRepository;
        this.adGroupAdditionalTargetingRepository = adGroupAdditionalTargetingRepository;
        this.hyperGeoRepository = hyperGeoRepository;
        this.hyperGeoService = hyperGeoService;
        this.adGroupTypeSupportDispatcher = adGroupTypeSupportDispatcher;
        this.shardHelper = shardHelper;
    }

    /**
     * Конвертирует набор статусов группы в соответствующее sql-условие для jooq-а
     *
     * @param statuses набор статусов группы
     * @return sql-условие для jooq-а
     */
    private static Condition convertStatusesToCondition(Set<AdGroupStatus> statuses) {
        checkArgument(!statuses.isEmpty(), "Set of adgroup statuses cannot be empty");

        List<Condition> conditions = new ArrayList<>();

        for (AdGroupStatus status : statuses) {
            switch (status) {
                case ACCEPTED:
                    conditions.add(PHRASES.STATUS_MODERATE.eq(PhrasesStatusmoderate.Yes)
                            .and(PHRASES.STATUS_POST_MODERATE.eq(PhrasesStatuspostmoderate.Yes)));
                    break;
                case DRAFT:
                    conditions.add(PHRASES.STATUS_MODERATE.eq(PhrasesStatusmoderate.New));
                    break;
                case MODERATION:
                    conditions.add(PHRASES.STATUS_MODERATE
                            .in(PhrasesStatusmoderate.Sent, PhrasesStatusmoderate.Sending, PhrasesStatusmoderate.Ready)
                            .and(PHRASES.STATUS_POST_MODERATE
                                    .in(PhrasesStatuspostmoderate.New, PhrasesStatuspostmoderate.No,
                                            PhrasesStatuspostmoderate.Ready, PhrasesStatuspostmoderate.Sent,
                                            PhrasesStatuspostmoderate.Rejected)));
                    break;
                case PREACCEPTED:
                    conditions.add(PHRASES.STATUS_MODERATE
                            .in(PhrasesStatusmoderate.Sent, PhrasesStatusmoderate.Sending, PhrasesStatusmoderate.Ready)
                            .and(PHRASES.STATUS_POST_MODERATE.eq(PhrasesStatuspostmoderate.Yes)));
                    break;
                case REJECTED:
                    conditions.add(PHRASES.STATUS_MODERATE.eq(PhrasesStatusmoderate.No));
                    break;
                default:
                    throw new IllegalArgumentException("Not supported adgroup status: " + status);
            }
        }

        return conditions.stream().reduce(Condition::or).orElseThrow(
                () -> new IllegalArgumentException("Can not build conditions for selection adgroups by statuses!"));
    }

    /**
     * Конвертирует набор статусов возможности показов группы в соответствующее sql-условие для jooq-а
     *
     * @param statuses набор статусов возможности показов группы
     * @return sql-условие для jooq-а
     */
    private static Condition convertServingStatusToCondition(Set<ServingStatus> statuses) {
        checkArgument(!statuses.isEmpty(), "Set of adgroup serving statuses cannot be empty");

        // если запрашивают группы со всеми статусами возможности показов, то не усложняем
        // запрос дополнительными условиями отбора групп
        if (statuses.size() == ServingStatus.values().length) {
            return null;
        }

        List<Condition> conditions = new ArrayList<>();

        for (ServingStatus status : statuses) {
            switch (status) {
                case RARELY_SERVED:
                    conditions.add(PHRASES.IS_BS_RARELY_LOADED.eq(RARELY_SERVED.dbValue())
                            .and(CAMPAIGNS.ARCHIVED.ne(CampaignsArchived.Yes).and(PHRASES.ADGROUP_TYPE
                                    .notIn(PhrasesAdgroupType.dynamic, PhrasesAdgroupType.performance))));
                    break;
                case ELIGIBLE:
                    // https://st.yandex-team.ru/DIRECT-62527
                    conditions.add(PHRASES.IS_BS_RARELY_LOADED.eq(ELIGIBLE.dbValue())
                            .or(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes).or(PHRASES.ADGROUP_TYPE
                                    .in(PhrasesAdgroupType.dynamic, PhrasesAdgroupType.performance))));
                    break;
                default:
                    throw new IllegalArgumentException("Not supported adgroup serving status: " + status);
            }
        }

        return conditions.stream().reduce(Condition::or).orElseThrow(
                () -> new IllegalArgumentException(
                        "Can not build conditions for selection adgroups by serving statuses!"));
    }

    /**
     * Конвертирует набор статусов модерации иконки приложения в соответствующее sql-условие для jooq-а
     *
     * @param statuses набор статусов модерации иконки приложения
     * @return sql-условие для jooq-а
     */
    private static Condition convertAppIconStatusesToCondition(Set<AdGroupAppIconStatus> statuses) {
        checkArgument(!statuses.isEmpty(), "Set of app icon statuses cannot be empty");

        Set<MobileContentStatusiconmoderate> appIconStatuses = EnumSet.noneOf(MobileContentStatusiconmoderate.class);

        for (AdGroupAppIconStatus status : statuses) {
            switch (status) {
                case ACCEPTED:
                    appIconStatuses.add(MobileContentStatusiconmoderate.Yes);
                    break;
                case MODERATION:
                    appIconStatuses.addAll(
                            asList(MobileContentStatusiconmoderate.Sent, MobileContentStatusiconmoderate.Sending,
                                    MobileContentStatusiconmoderate.Ready));
                    break;
                case REJECTED:
                    appIconStatuses.add(MobileContentStatusiconmoderate.No);
                    break;
                default:
                    throw new IllegalStateException("Not supported app icon status: " + status);
            }
        }

        return MOBILE_CONTENT.ICON_HASH.isNotNull().and(MOBILE_CONTENT.STATUS_ICON_MODERATE.in(appIconStatuses));
    }

    /**
     * Сгенерировать условие на {@link Phrases#PID} и {@link Phrases#GEO}
     * по {@link AdGroupShowsForecast#getId()} и соответствующим значениям {@link AdGroupShowsForecast#getGeo()}).
     * Значения гео равные {@code null} подставляются как пустая строка
     *
     * @param adGroupsForecasts коллекция прогнозов показов по которой нужно сгенерировать условие
     * @return Условие на вхождение {@code pid} и равенство соответствующих ему значений {@code geo}
     */
    private static Condition getAdgroupIdAndGeoCondition(Collection<AdGroupShowsForecast> adGroupsForecasts) {
        Map<Long, String> geoByAdgroupId = StreamEx.of(adGroupsForecasts)
                .toMap(AdGroupShowsForecast::getId, a -> nvl(a.getGeo(), ""));

        Field<String> geoCase = JooqMapperUtils.makeCaseStatement(PHRASES.PID, PHRASES.GEO, geoByAdgroupId);

        return PHRASES.PID.in(geoByAdgroupId.keySet()).and(PHRASES.GEO.eq(geoCase));
    }

    public void getLockOnAdGroups(Configuration conf, Collection<Long> adGroupIds) {
        conf.dsl()
                .selectZero()
                .from(PHRASES)
                .where(PHRASES.PID.in(adGroupIds))
                .forUpdate()
                .fetch();
    }

    /**
     * Сохраняет список групп для одного клиента.
     * <p>
     * ВАЖНО(!): все группы должны относиться к указанному клиенту!
     * <p>
     * В переданных объектах групп также выставляются
     * сгенерированные id и id минус-фраз (id минус-фраз
     * могут быть как новые, так и уже существующие,
     * так как хранятся уникально).
     *
     * @param config   jooq конфиг
     * @param clientId клиент
     * @param adGroups список групп
     * @return id сохраненных групп
     */
    public List<Long> addAdGroups(Configuration config, ClientId clientId, List<AdGroup> adGroups) {
        generateAdGroupIds(clientId, adGroups);
        addTags(config, adGroups);
        addLibraryMinusKeywords(config, clientId, adGroups);
        addHyperGeos(config, adGroups);
        addHyperGeoAdditionalTargetings(config, clientId, adGroups);
        addContentCategoriesAdditionalTargetings(config, clientId, adGroups);
        addAdGroupsToDatabaseTables(config, clientId, adGroups);
        return mapList(adGroups, AdGroup::getId);
    }

    /**
     * Получает список групп по списку их id, отсортированный в порядке возрастания id
     *
     * @param shard      шард
     * @param adGroupIds список id извлекаемых групп
     * @return список групп
     */
    public List<AdGroup> getAdGroups(int shard, Collection<Long> adGroupIds) {
        SelectQuery<Record> query = ppcDslContextProvider.ppc(shard).selectQuery();

        query.addSelect(ALL_TEXT_ADGROUP_FIELDS_TO_READ);

        query.addFrom(PHRASES);
        query.addJoin(MINUS_WORDS, JoinType.LEFT_OUTER_JOIN, PHRASES.MW_ID.eq(MINUS_WORDS.MW_ID));
        query.addJoin(GROUP_PARAMS, JoinType.LEFT_OUTER_JOIN, GROUP_PARAMS.PID.eq(PHRASES.PID));
        query.addJoin(ADGROUP_BS_TAGS, JoinType.LEFT_OUTER_JOIN, ADGROUP_BS_TAGS.PID.eq(PHRASES.PID));
        query.addJoin(ADGROUP_PROJECT_PARAMS, JoinType.LEFT_OUTER_JOIN, ADGROUP_PROJECT_PARAMS.PID.eq(PHRASES.PID));
        query.addJoin(ADGROUPS_HYPERGEO_RETARGETINGS, JoinType.LEFT_OUTER_JOIN,
                ADGROUPS_HYPERGEO_RETARGETINGS.PID.eq(PHRASES.PID));

        query.addSelect(DYNAMIC_DOMAIN_ADGROUP_FIELDS_TO_READ);
        query.addSelect(DYNAMIC_FEED_ADGROUP_FIELDS_TO_READ);
        query.addSelect(PERFORMANCE_ADGROUP_FIELDS_TO_READ);
        query.addJoin(ADGROUPS_DYNAMIC, JoinType.LEFT_OUTER_JOIN, ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID));
        query.addJoin(ADGROUPS_PERFORMANCE, JoinType.LEFT_OUTER_JOIN, ADGROUPS_PERFORMANCE.PID.eq(PHRASES.PID));
        query.addJoin(DOMAINS, JoinType.LEFT_OUTER_JOIN, DOMAINS.DOMAIN_ID.eq(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID));
        query.addJoin(FEEDS, JoinType.LEFT_OUTER_JOIN, FEEDS.FEED_ID.eq(ADGROUPS_DYNAMIC.FEED_ID));

        query.addSelect(MOBILE_CONTENT_ADGROUP_FIELDS_TO_READ);
        query.addSelect(MOBILE_CONTENT_FIELDS_TO_READ);
        query.addJoin(ADGROUPS_MOBILE_CONTENT, JoinType.LEFT_OUTER_JOIN,
                ADGROUPS_MOBILE_CONTENT.PID.eq(PHRASES.PID));
        query.addJoin(MOBILE_CONTENT, JoinType.LEFT_OUTER_JOIN,
                MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID));

        query.addSelect(CPM_BANNER_ADGROUP_FIELDS_TO_READ);
        query.addSelect(CPM_VIDEO_ADGROUP_FIELDS_TO_READ);
        query.addJoin(ADGROUPS_CPM_BANNER, JoinType.LEFT_OUTER_JOIN, ADGROUPS_CPM_BANNER.PID.eq(PHRASES.PID));
        query.addJoin(ADGROUPS_CPM_VIDEO, JoinType.LEFT_OUTER_JOIN, ADGROUPS_CPM_VIDEO.PID.eq(PHRASES.PID));

        query.addSelect(CONTENT_PROMOTION_ADGROUP_FIELDS_TO_READ);
        query.addJoin(ADGROUPS_CONTENT_PROMOTION,
                JoinType.LEFT_OUTER_JOIN, ADGROUPS_CONTENT_PROMOTION.PID.eq(PHRASES.PID));

        query.addSelect(CPM_OUTDOOR_ADGROUP_FIELDS_TO_READ);
        query.addSelect(CPM_INDOOR_ADGROUP_FIELDS_TO_READ);
        query.addJoin(ADGROUP_PAGE_TARGETS, JoinType.LEFT_OUTER_JOIN, ADGROUP_PAGE_TARGETS.PID.eq(PHRASES.PID));

        query.addSelect(CPM_YNDX_FRONTPAGE_PRIORITY_FIELDS_TO_READ);
        query.addJoin(ADGROUP_PRIORITY, JoinType.LEFT_OUTER_JOIN, ADGROUP_PRIORITY.PID.eq(PHRASES.PID));

        query.addSelect(INTERNAL_ADGROUP_FIELDS_TO_READ);
        query.addJoin(ADGROUPS_INTERNAL, JoinType.LEFT_OUTER_JOIN, ADGROUPS_INTERNAL.PID.eq(PHRASES.PID));

        query.addSelect(TEXT_ADGROUP_WITH_FEED_FIELDS_TO_READ);
        query.addJoin(ADGROUPS_TEXT, JoinType.LEFT_OUTER_JOIN, ADGROUPS_TEXT.PID.eq(PHRASES.PID));

        // ограничиваем типы групп только поддерживаемыми в репозитории
        query.addConditions(PHRASES.ADGROUP_TYPE.in(adGroupTypeSupportDispatcher.allSupportedDbTypes()));

        query.addConditions(PHRASES.PID.in(adGroupIds));

        query.addOrderBy(PHRASES.PID);

        List<AdGroup> adGroups = query.fetch().map(adGroupTypeSupportDispatcher::constructInstanceFromDb);

        Map<Long, List<Long>> adGroupsTags = adGroupTagsRepository.getAdGroupsTags(shard, adGroupIds);
        adGroups.forEach(adGroup -> adGroup.setTags(adGroupsTags.get(adGroup.getId())));

        Map<Long, List<Long>> adGroupsMinusKeywordsPacks =
                minusKeywordsPackRepository.getAdGroupsLibraryMinusKeywordsPacks(shard, adGroupIds);
        adGroups.forEach(adGroup -> adGroup.setLibraryMinusKeywordsIds(adGroupsMinusKeywordsPacks.get(adGroup.getId())));

        List<AdGroupWithUsersSegments> adGroupWithUsersSegments
                = filterAndMapList(adGroups, g -> g instanceof AdGroupWithUsersSegments, g -> (AdGroupWithUsersSegments) g);
        if (!adGroupWithUsersSegments.isEmpty()) {
            Map<Long, List<UsersSegment>> segmentsByAdGroupIds =
                    usersSegmentRepository.getSegmentsAsMap(shard, mapList(adGroupWithUsersSegments,
                            AdGroupWithUsersSegments::getId));
            adGroupWithUsersSegments.forEach(adGroup -> adGroup.setUsersSegments(
                    defaultIfNull(segmentsByAdGroupIds.get(adGroup.getId()), emptyList())));
        }

        var adGroupAdditionalTargetings = adGroupAdditionalTargetingRepository.getByAdGroupIds(shard, adGroupIds);
        var contentCategoriesToGroupIds =
                adGroupAdditionalTargetings.stream()
                        .filter(item -> ContentCategoriesAdGroupAdditionalTargeting.class.isAssignableFrom(item.getClass()))
                        .map(targeting -> (ContentCategoriesAdGroupAdditionalTargeting) targeting)
                        .collect(groupingBy(AdGroupAdditionalTargeting::getAdGroupId));
        var yandexUidsToGroupIds =
                adGroupAdditionalTargetings.stream()
                        .filter(item -> YandexUidsAdGroupAdditionalTargeting.class.isAssignableFrom(item.getClass()))
                        .collect(groupingBy(AdGroupAdditionalTargeting::getAdGroupId,
                                flatMapping(t -> ((YandexUidsAdGroupAdditionalTargeting) t).getValue().stream(),
                                        toList())));
        adGroups.forEach(adGroup ->
                adGroup.withYandexUids(yandexUidsToGroupIds.get(adGroup.getId()))
                        .withContentCategoriesRetargetingConditionRules(
                                mapList(contentCategoriesToGroupIds.get(adGroup.getId()),
                                        ContentCategoriesTargetingConverter::toRule)));

        return adGroups;
    }

    public Map<Long, List<PageBlock>> getAdGroupsPageTargetByAdGroupId(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard).dsl()
                .select(ADGROUP_PAGE_TARGETS.PID, ADGROUP_PAGE_TARGETS.PAGE_BLOCKS)
                .from(ADGROUP_PAGE_TARGETS)
                .where(ADGROUP_PAGE_TARGETS.PID.in(adGroupIds))
                .fetchMap(ADGROUP_PAGE_TARGETS.PID,
                        record -> pageBlocksFromDb(record.get(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS)));
    }

    public Map<Long, Tuple2<List<PageBlock>, List<Long>>> getAdGroupsOutdoorPageTarget(Configuration configuration,
                                                                                       Collection<Long> bids) {
        SelectConditionStep<Record> query = DSL.using(configuration)
                .select(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS)
                .select(BANNERS.BID)
                .select(PHRASES.GEO)
                .from(ADGROUP_PAGE_TARGETS)
                .join(BANNERS).on(BANNERS.PID.eq(ADGROUP_PAGE_TARGETS.PID))
                .join(PHRASES).on(BANNERS.PID.eq(PHRASES.PID))
                .where(PHRASES.ADGROUP_TYPE.eq(PhrasesAdgroupType.cpm_outdoor)
                        .and(BANNERS.BID.in(bids)));

        Map<Long, Tuple2<List<PageBlock>, List<Long>>> res = new HashMap<>();

        query.fetch().forEach(r -> res.put(
                r.get(BANNERS.BID),
                tuple(pageBlocksFromDb(r.get(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS)), geoFromDb(r.get(PHRASES.GEO)))));

        return res;
    }

    /**
     * Получает список id групп, отсортированный в порядке возрастания, по условиям отбора
     *
     * @param shard             шард
     * @param selectionCriteria условия отбора групп
     * @param limitOffset       ограничения на список результатов
     * @return список групп, отсортированный в порядке возрастания id
     */
    public List<Long> getAdGroupIdsBySelectionCriteria(int shard, AdGroupsSelectionCriteria selectionCriteria,
                                                       LimitOffset limitOffset) {
        checkArgument(!selectionCriteria.getAdGroupIds().isEmpty() || !selectionCriteria.getCampaignIds().isEmpty(),
                "AdGroupIds or CampaignIds must be specified!");

        Map<TableLike<?>, Condition> tablesForJoin = new LinkedHashMap<>(); // the order of tables is important
        List<Condition> conditions = new ArrayList<>();

        if (!selectionCriteria.getAdGroupIds().isEmpty()) {
            conditions.add(PHRASES.PID.in(selectionCriteria.getAdGroupIds()));
        }

        if (!selectionCriteria.getCampaignIds().isEmpty()) {
            conditions.add(PHRASES.CID.in(selectionCriteria.getCampaignIds()));
        }

        if (!selectionCriteria.getAdGroupTypes().isEmpty()) {
            conditions.add(PHRASES.ADGROUP_TYPE
                    .in(selectionCriteria.getAdGroupTypes().stream().map(AdGroupType::toSource).collect(toList())));
        }

        if (!selectionCriteria.getAdGroupStatuses().isEmpty()) {
            conditions.add(convertStatusesToCondition(selectionCriteria.getAdGroupStatuses()));
        }

        if (!selectionCriteria.getServingStatuses().isEmpty()) {
            tablesForJoin.put(CAMPAIGNS, CAMPAIGNS.CID.eq(PHRASES.CID));

            Condition condition = convertServingStatusToCondition(selectionCriteria.getServingStatuses());
            if (condition != null) {
                conditions.add(condition);
            }
        }

        if (!selectionCriteria.getAdGroupAppIconStatuses().isEmpty()) {
            tablesForJoin.put(ADGROUPS_MOBILE_CONTENT, ADGROUPS_MOBILE_CONTENT.PID.eq(PHRASES.PID));
            tablesForJoin.put(MOBILE_CONTENT,
                    MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID));
            conditions.add(convertAppIconStatusesToCondition(selectionCriteria.getAdGroupAppIconStatuses()));
        }

        if (!selectionCriteria.getNegativeKeywordSharedSetIds().isEmpty()) {
            tablesForJoin.put(ADGROUPS_MINUS_WORDS, ADGROUPS_MINUS_WORDS.PID.eq(PHRASES.PID));
            conditions.add(ADGROUPS_MINUS_WORDS.MW_ID.in(selectionCriteria.getNegativeKeywordSharedSetIds()));
        }

        if (!selectionCriteria.getContentPromotionAdgroupTypes().isEmpty()) {
            tablesForJoin.put(ADGROUPS_CONTENT_PROMOTION, ADGROUPS_CONTENT_PROMOTION.PID.eq(PHRASES.PID));
            conditions.add(ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE.in(
                    selectionCriteria.getContentPromotionAdgroupTypes().stream()
                            .map(ContentPromotionAdgroupType::toSource)
                            .collect(toSet())));
        }

        SelectJoinStep<Record1<Long>> joinStep =
                ppcDslContextProvider.ppc(shard).select(PHRASES.PID).from(PHRASES);

        for (Map.Entry<TableLike<?>, Condition> entry : tablesForJoin.entrySet()) {
            joinStep = joinStep.join(entry.getKey()).on(entry.getValue());
        }

        return joinStep.where(conditions).orderBy(PHRASES.PID).offset(limitOffset.offset()).limit(limitOffset.limit())
                .fetch().map(e -> e.get(PHRASES.PID));
    }

    public void updateAdGroups(
            int shard, ClientId clientId, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        updateAdGroups(ppcDslContextProvider.ppc(shard).configuration(), clientId, appliedChanges);
    }

    /**
     * Определить язык группы объявлений через кампанию
     *
     * @return Отображение идентификатора группы объявления в язык определенный через кампанию
     */
    public Map<Long, String> getAdGroupsLangFromCampaign(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID, CAMP_OPTIONS.CONTENT_LANG)
                .from(PHRASES)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(PHRASES.CID))
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(PHRASES.PID, CAMP_OPTIONS.CONTENT_LANG);
    }

    /**
     * Обновление групп объявлений для одного клиента.
     * <p>
     * ВАЖНО(!): все группы должны относиться к указанному клиенту!
     *
     * @param config         конфигурация подключени подключение к базе данных шарда клиента
     * @param clientId       id клиента
     * @param appliedChanges коллекция изменений, применных к моделям
     */
    public void updateAdGroups(
            Configuration config, ClientId clientId, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        doUpdateAdGroups(config, clientId, appliedChanges);
        updateAdGroupsTags(config, appliedChanges);
        updateAdGroupsLibraryMinusWords(config, clientId, appliedChanges);
        updateHyperGeo(config, clientId, appliedChanges);
        updateHyperGeoAdditionalTargetings(config, clientId, appliedChanges);
        updateContentCategoriesAdditionalTargetings(config, clientId, appliedChanges);
        updateYandexUidAdditionalTargeting(config, clientId, appliedChanges);
        updatePageGroupAndTargetTags(config, appliedChanges);
        updateProjectParamConditions(config, appliedChanges);
    }

    /**
     * Обновление базовых полей групп.
     * <p>
     * Предназначен для балкового обновления групп в джобах/ваншотах.
     */
    public void updateAdGroupsCommonFields(int shard, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        updateCommonFields(ppcDslContextProvider.ppc(shard).configuration(), appliedChanges);
    }

    private void doUpdateAdGroups(
            Configuration config, ClientId clientId, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        updateCommonFields(config, appliedChanges);

        updateTrackingParams(config, appliedChanges);

        adGroupTypeSupportDispatcher.updateAdGroups(config.dsl(), clientId, appliedChanges);
    }

    private void updateCommonFields(Configuration config, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        JooqUpdateBuilder<PhrasesRecord, AdGroup> updateBuilder =
                new JooqUpdateBuilder<>(PHRASES.PID, appliedChanges);

        updateBuilder.processProperty(AdGroup.NAME, PHRASES.GROUP_NAME);
        updateBuilder.processProperty(AdGroup.GEO, PHRASES.GEO, AdGroupMappings::geoToDb);
        updateBuilder.processProperty(AdGroup.STATUS_BS_SYNCED, PHRASES.STATUS_BS_SYNCED,
                AdGroupMappings::statusBsSyncedToDb);
        updateBuilder.processProperty(AdGroup.STATUS_MODERATE, PHRASES.STATUS_MODERATE,
                StatusModerate::toSource);
        updateBuilder.processProperty(AdGroup.STATUS_POST_MODERATE, PHRASES.STATUS_POST_MODERATE,
                StatusPostModerate::toSource);
        updateBuilder.processProperty(AdGroup.MINUS_KEYWORDS_ID, PHRASES.MW_ID);
        updateBuilder.processProperty(AdGroup.STATUS_AUTOBUDGET_SHOW, PHRASES.STATUS_AUTOBUDGET_SHOW,
                AdGroupMappings::statusAutobudgetShowToDb);
        updateBuilder.processProperty(AdGroup.STATUS_SHOWS_FORECAST, PHRASES.STATUS_SHOWS_FORECAST,
                StatusShowsForecast::toSource);
        updateBuilder.processProperty(AdGroup.FORECAST_DATE, PHRASES.FORECAST_DATE);

        updateBuilder.processProperty(AdGroup.LAST_CHANGE, PHRASES.LAST_CHANGE);

        config.dsl().update(PHRASES)
                .set(updateBuilder.getValues())
                .where(PHRASES.PID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    private void updateTrackingParams(Configuration config, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        List<AdGroup> groupParams = appliedChanges.stream()
                .filter(ch -> ch.changed(AdGroup.TRACKING_PARAMS))
                .map(AppliedChanges::getModel)
                .collect(toList());

        InsertHelper<GroupParamsRecord> helper = new InsertHelper<>(config.dsl(), GroupParams.GROUP_PARAMS);
        helper.addAll(ADGROUP_MAPPER_FOR_COMMON_FIELDS, groupParams);
        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(GROUP_PARAMS.HREF_PARAMS, MySQLDSL.values(GROUP_PARAMS.HREF_PARAMS));
        }
        helper.executeIfRecordsAdded();
    }

    /**
     * Изменить список тегов для таргетинга группы.
     * При обоих пустых полях page_group_tags_json и target_tags_json удаляется запись из adgroup_bs_tags
     */
    private void updatePageGroupAndTargetTags(Configuration config,
                                              Collection<AppliedChanges<AdGroup>> appliedChanges) {
        List<AdGroup> changedAdGroups = appliedChanges.stream()
                .filter(ch -> ch.changed(AdGroup.PAGE_GROUP_TAGS) || ch.changed(AdGroup.TARGET_TAGS))
                .map(AppliedChanges::getModel)
                .collect(toList());

        List<AdGroup> adGroupBsTagsForUpdate = changedAdGroups.stream()
                .filter(adGroup -> adGroup.getPageGroupTags() != null && !adGroup.getPageGroupTags().isEmpty() ||
                        adGroup.getTargetTags() != null && !adGroup.getTargetTags().isEmpty())
                .collect(toList());

        InsertHelper<AdgroupBsTagsRecord> helper = new InsertHelper<>(config.dsl(), AdgroupBsTags.ADGROUP_BS_TAGS);
        helper.addAll(ADGROUP_MAPPER_FOR_COMMON_FIELDS, adGroupBsTagsForUpdate);
        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON, MySQLDSL.values(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON))
                    .set(ADGROUP_BS_TAGS.TARGET_TAGS_JSON, MySQLDSL.values(ADGROUP_BS_TAGS.TARGET_TAGS_JSON));
        }
        helper.executeIfRecordsAdded();

        // Удаляем запись из adgroup_bs_tags в которой оба поля page_group_tags_json и target_tags_json пустые
        List<Long> adGroupBsTagsForDeleteIds = changedAdGroups.stream()
                .filter(adGroup -> (adGroup.getPageGroupTags() == null || adGroup.getPageGroupTags().isEmpty()) &&
                        (adGroup.getTargetTags() == null || adGroup.getTargetTags().isEmpty()))
                .map(AdGroup::getId)
                .collect(toList());

        if (!adGroupBsTagsForDeleteIds.isEmpty()) {
            adGroupBsTagsRepository.deleteAllAdGroupBsTags(config.dsl(), adGroupBsTagsForDeleteIds);
        }
    }

    /**
     * Изменить список формул таргетинга на параметры проектов.
     * При пустом поле project_param_conditions удаляется запись из adgroup_project_params
     */
    private void updateProjectParamConditions(Configuration config,
                                              Collection<AppliedChanges<AdGroup>> appliedChanges) {
        List<AdGroup> changedAdGroups = appliedChanges.stream()
                .filter(ch -> ch.changed(AdGroup.PROJECT_PARAM_CONDITIONS))
                .map(AppliedChanges::getModel)
                .collect(toList());

        List<AdGroup> adGroupProjectParamsConditionsForUpdate = changedAdGroups.stream()
                .filter(adGroup -> adGroup.getProjectParamConditions() != null
                        && !adGroup.getProjectParamConditions().isEmpty())
                .collect(toList());

        InsertHelper<AdgroupProjectParamsRecord> helper = new InsertHelper<>(config.dsl(),
                AdgroupProjectParams.ADGROUP_PROJECT_PARAMS);
        helper.addAll(ADGROUP_MAPPER_FOR_COMMON_FIELDS, adGroupProjectParamsConditionsForUpdate);
        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(ADGROUP_PROJECT_PARAMS.PROJECT_PARAM_CONDITIONS,
                            MySQLDSL.values(ADGROUP_PROJECT_PARAMS.PROJECT_PARAM_CONDITIONS));
        }
        helper.executeIfRecordsAdded();

        // Удаляем запись из adgroup_project_params в которой project_param_conditions пустое
        List<Long> adGroupProjectParamsForDeleteIds = changedAdGroups.stream()
                .filter(adGroup -> adGroup.getProjectParamConditions() == null || adGroup.getProjectParamConditions().isEmpty())
                .map(AdGroup::getId)
                .collect(toList());

        if (!adGroupProjectParamsForDeleteIds.isEmpty()) {
            config.dsl()
                    .deleteFrom(ADGROUP_PROJECT_PARAMS)
                    .where(ADGROUP_PROJECT_PARAMS.PID.in(adGroupProjectParamsForDeleteIds))
                    .execute();
        }
    }

    public Map<Long, List<Long>> getAdGroupsProjectParamConditions(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard).configuration().dsl()
                .select(ADGROUP_PROJECT_PARAMS.PID, ADGROUP_PROJECT_PARAMS.PROJECT_PARAM_CONDITIONS)
                .from(ADGROUP_PROJECT_PARAMS)
                .where(ADGROUP_PROJECT_PARAMS.PID.in(adGroupIds))
                .fetchMap(ADGROUP_PROJECT_PARAMS.PID,
                        r -> longListFromDbJsonFormat(r.get(ADGROUP_PROJECT_PARAMS.PROJECT_PARAM_CONDITIONS)));
    }

    /**
     * Обновляет метки:
     * - метки, которые есть в новом списке, но не было в старом - добавляются;
     * - метки, которых нет в новом списке, но были в старом - удаляются
     */
    private void updateAdGroupsTags(Configuration config, Collection<AppliedChanges<AdGroup>> appliedChanges) {
        Map<Long, List<Long>> adGroupTags = appliedChanges.stream()
                .filter(isChanged(AdGroup.TAGS))
                .map(AppliedChanges::getModel)
                .collect(toMap(AdGroup::getId, ag -> Optional.ofNullable(ag.getTags()).orElse(new ArrayList<>())));
        if (adGroupTags.isEmpty()) {
            return;
        }
        Map<Long, List<Long>> existingTagsMap = adGroupTagsRepository.getAdGroupsTags(config, adGroupTags.keySet());
        Map<Long, Set<Long>> adGroupTagIdsForAdd = new HashMap<>();
        Map<Long, Set<Long>> adGroupTagIdsForDelete = new HashMap<>();
        adGroupTags.forEach((adGroupId, newTagsList) -> {
            Set<Long> newTags = Optional.ofNullable(newTagsList)
                    .map(HashSet::new).orElse(new HashSet<>());
            Set<Long> existingTags = Optional.ofNullable(existingTagsMap.get(adGroupId))
                    .map(HashSet::new).orElse(new HashSet<>());

            Set<Long> tagsForAdd = new HashSet<>(newTags);
            tagsForAdd.removeAll(existingTags);
            adGroupTagIdsForAdd.put(adGroupId, tagsForAdd);

            Set<Long> tagsForDelete = new HashSet<>(existingTags);
            tagsForDelete.removeAll(newTags);
            adGroupTagIdsForDelete.put(adGroupId, tagsForDelete);
        });
        adGroupTagsRepository.addAdGroupTags(config, adGroupTagIdsForAdd);
        adGroupTagsRepository.deleteFromAdGroup(config, adGroupTagIdsForDelete);
    }

    /**
     * Изменить список привязанных библиотечных наборов минус-фраз для группы.
     * <p>
     * Удаляем все текущие привязки и добавляем новые.
     * Важно: в методе берется лок на группы и на минус-фразы
     */
    private void updateAdGroupsLibraryMinusWords(Configuration conf, ClientId clientId,
                                                 Collection<AppliedChanges<AdGroup>> appliedChanges) {
        List<AdGroup> adGroups = appliedChanges.stream()
                .filter(isChanged(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS))
                .map(AppliedChanges::getModel)
                .collect(toList());

        List<Long> adGroupIds = mapList(adGroups, AdGroup::getId);

        //Синхронизируемся на группе, чтобы при возникновении гонок сохранялся набор привязок одного
        // из конкурирующих запросов, но не их смесь.
        getLockOnAdGroups(conf, adGroupIds);

        minusKeywordsPackRepository.deleteAdGroupToPackLinks(conf, adGroupIds);
        addLibraryMinusKeywords(conf, clientId, adGroups);
    }

    private void updateContentCategoriesAdditionalTargetings(
            Configuration config, ClientId clientId, Collection<AppliedChanges<AdGroup>> appliedChanges) {

        var changedAdGroups = StreamEx.of(appliedChanges)
                .filter(adGroupAppliedChanges -> adGroupAppliedChanges.changed(AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES))
                .map(AppliedChanges::getModel)
                .toList();

        var adGroupIds = mapList(changedAdGroups, AdGroup::getId);

        var additionalTargetings = StreamEx.of(changedAdGroups)
                .filter(adGroup -> Objects.nonNull(adGroup.getContentCategoriesRetargetingConditionRules()))
                .map(adGroup -> mapList(
                        adGroup.getContentCategoriesRetargetingConditionRules(),
                        category -> ContentCategoriesTargetingConverter.toTargeting(category).withAdGroupId(adGroup.getId())))
                .flatMap(List::stream)
                .map(targeting -> (AdGroupAdditionalTargeting) targeting)
                .toList();

        adGroupAdditionalTargetingRepository.deleteByAdGroupIds(config, adGroupIds);
        adGroupAdditionalTargetingRepository.add(config, clientId, additionalTargetings);
    }

    /**
     * Добавляет привязки геосегмента
     */
    private void updateHyperGeoAdditionalTargetings(Configuration config, ClientId clientId,
                                                    Collection<AppliedChanges<AdGroup>> adGroupsAppliedChanges) {
        List<AdGroup> changedAdGroups = StreamEx.of(adGroupsAppliedChanges)
                .filter(adGroupAppliedChanges -> adGroupAppliedChanges.changed(AdGroup.HYPER_GEO_ID))
                .map(AppliedChanges::getModel)
                .toList();

        List<Long> adGroupIds = mapList(changedAdGroups, AdGroup::getId);

        List<AdGroupAdditionalTargeting> additionalTargetings = StreamEx.of(changedAdGroups)
                .filter(adGroup -> Objects.nonNull(adGroup.getHyperGeoId()))
                .map(adGroup ->
                        (AdGroupAdditionalTargeting) new AuditoriumGeoSegmentsAdGroupAdditionalTargeting()
                                .withAdGroupId(adGroup.getId())
                                .withJoinType(AdGroupAdditionalTargetingJoinType.ANY)
                                .withTargetingMode(AdGroupAdditionalTargetingMode.TARGETING)
                                .withValue(listToSet(adGroup.getHyperGeoSegmentIds(), identity())))
                .toList();

        adGroupAdditionalTargetingRepository.deleteByAdGroupIds(config, adGroupIds);
        adGroupAdditionalTargetingRepository.add(config, clientId, additionalTargetings);
    }

    /**
     * Добавляет привязки yandex uid-ов
     */
    public void updateYandexUidAdditionalTargeting(Configuration config, ClientId clientId,
                                                   Collection<AppliedChanges<AdGroup>> adGroupsAppliedChanges) {
        List<AdGroup> changedAdGroups = StreamEx.of(adGroupsAppliedChanges)
                .filter(adGroupAppliedChanges -> adGroupAppliedChanges.changed(AdGroup.YANDEX_UIDS))
                .map(AppliedChanges::getModel)
                .toList();

        List<Long> adGroupIds = mapList(changedAdGroups, AdGroup::getId);

        List<AdGroupAdditionalTargeting> additionalTargetings = StreamEx.of(changedAdGroups)
                .filter(adGroup -> Objects.nonNull(adGroup.getYandexUids()))
                .map(adGroup ->
                        (AdGroupAdditionalTargeting) new YandexUidsAdGroupAdditionalTargeting()
                                .withAdGroupId(adGroup.getId())
                                .withJoinType(AdGroupAdditionalTargetingJoinType.ANY)
                                .withTargetingMode(AdGroupAdditionalTargetingMode.TARGETING)
                                .withValue(adGroup.getYandexUids()))
                .toList();

        List<AdGroupAdditionalTargeting> existingTargeting =
                adGroupAdditionalTargetingRepository.getByAdGroupIdsAndType(config.dsl(), adGroupIds,
                        AdgroupAdditionalTargetingsTargetingType.yandexuids);
        adGroupAdditionalTargetingRepository.deleteByIds(config.dsl(), mapList(existingTargeting, at -> at.getId()));
        adGroupAdditionalTargetingRepository.add(config, clientId, additionalTargetings);
    }

    private void updateHyperGeo(Configuration config, ClientId clientId,
                                Collection<AppliedChanges<AdGroup>> adGroupsAppliedChanges) {
        List<AdGroup> changedAdGroups = StreamEx.of(adGroupsAppliedChanges)
                .filter(adGroupAppliedChanges -> adGroupAppliedChanges.changed(AdGroup.HYPER_GEO_ID))
                .map(AppliedChanges::getModel)
                .toList();

        List<Long> adGroupIds = mapList(changedAdGroups, AdGroup::getId);

        var probablyUnlinkedHyperGeoIds = hyperGeoRepository.getHyperGeoIds(config, adGroupIds);
        hyperGeoRepository.unlinkHyperGeosFromAdGroups(config, adGroupIds);
        hyperGeoRepository.linkHyperGeosToAdGroups(config,
                StreamEx.of(changedAdGroups)
                        .filter(adGroup -> Objects.nonNull(adGroup.getHyperGeoId()))
                        .toMap(AdGroup::getId, AdGroup::getHyperGeoId));

        // удалятся только те гипер гео, которые теперь не привязаны ни к одной группе
        hyperGeoService.deleteHyperGeos(config, clientId, new ArrayList<>(probablyUnlinkedHyperGeoIds));
    }

    public Map<Long, AdGroupSimple> getAdGroupSimple(DSLContext dslContext, @Nullable ClientId clientId,
                                                     Collection<Long> adGroupIds) {
        Map<Long, AdGroup> adGroupsCommon =
                getCommonAdGroups(dslContext, clientId, adGroupIds, SIMPLE_ADGROUP_FIELDS_TO_READ);
        return EntryStream.of(adGroupsCommon)
                .mapValues(AdGroupSimple.class::cast)
                .toMap();
    }

    /**
     * Возвращает {@link Map}, где ключами являются ID найденных adGroup,
     * а значения &ndash; заполненные {@link AdGroupSimple}
     *
     * @param shard      шард
     * @param clientId   опциональный параметр для ограничения выборки групп по клиенту
     * @param adGroupIds набор ID по которым необходимо выполнить поиск
     * @return {@link Map} с {@link AdGroupSimple}
     */
    public Map<Long, AdGroupSimple> getAdGroupSimple(int shard, @Nullable ClientId clientId,
                                                     Collection<Long> adGroupIds) {
        return getAdGroupSimple(ppcDslContextProvider.ppc(shard), clientId, adGroupIds);
    }

    public Map<Long, AdGroupWithType> getAdGroupsWithType(int shard, @Nullable ClientId clientId,
                                                          Collection<Long> adGroupIds) {
        Map<Long, AdGroup> adGroupsCommon =
                getCommonAdGroups(shard, clientId, adGroupIds, ADGROUP_WITH_TYPE_FIELDS_TO_READ);
        return EntryStream.of(adGroupsCommon)
                .mapValues(adGroup -> (AdGroupWithType) adGroup)
                .toMap();
    }

    public Map<Long, AdGroupWithTypeAndGeo> getAdGroupsWithTypeAndGeo(
            int shard,
            @Nullable ClientId clientId,
            Collection<Long> adGroupIds
    ) {
        var joinCond = ppcDslContextProvider.ppc(shard).select(
                        PHRASES.PID,
                        PHRASES.CID,
                        PHRASES.ADGROUP_TYPE,
                        PHRASES.GEO,
                        ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID
                )
                .from(PHRASES)
                .leftJoin(ADGROUPS_HYPERGEO_RETARGETINGS)
                .on(PHRASES.PID.eq(ADGROUPS_HYPERGEO_RETARGETINGS.PID));

        var whereCond = PHRASES.PID.in(adGroupIds);

        if (clientId != null) {
            joinCond = joinCond.join(CAMPAIGNS).on(PHRASES.CID.eq(CAMPAIGNS.CID));
            whereCond = whereCond.and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()));
        }

        final var adGroupsCommon = joinCond
                .where(whereCond)
                .fetchMap(PHRASES.PID, record -> ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(new UntypedAdGroup(), record));

        return EntryStream.of(adGroupsCommon)
                .mapValues(adGroup -> (AdGroupWithTypeAndGeo) adGroup)
                .toMap();
    }

    /**
     * Возвращает {@link Map}, где ключами являются ID найденных adGroup,
     * а значения &ndash; заполненные {@link AdGroupName}
     *
     * @param shard      шард
     * @param clientId   опциональный параметр для ограничения выборки групп по клиенту
     * @param adGroupIds набор id, по которым необходимо выполнить поиск
     * @return {@link Map} с {@link AdGroupSimple}
     */
    public Map<Long, AdGroupName> getAdGroupNames(int shard, @Nullable ClientId clientId, Collection<Long> adGroupIds) {
        Map<Long, AdGroup> adGroupsCommon =
                getCommonAdGroups(shard, clientId, adGroupIds, ADGROUP_NAME_FIELDS_TO_READ);
        return EntryStream.of(adGroupsCommon)
                .mapValues(adGroup -> (AdGroupName) adGroup)
                .toMap();
    }

    private Map<Long, AdGroup> getCommonAdGroups(int shard, @Nullable ClientId clientId, Collection<Long> adGroupIds,
                                                 TableField[] fieldsToRead) {
        return getCommonAdGroups(ppcDslContextProvider.ppc(shard), clientId, adGroupIds, fieldsToRead);
    }

    /**
     * Метод для получения объектов AdGroup (общие поля всех типов групп).
     *
     * @param dslContext   контекст
     * @param clientId     id клиента (опционально)
     * @param adGroupIds   id групп
     * @param fieldsToRead список полей для чтения
     */
    private Map<Long, AdGroup> getCommonAdGroups(DSLContext dslContext, @Nullable ClientId clientId,
                                                 Collection<Long> adGroupIds, TableField[] fieldsToRead) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        SelectJoinStep<Record> joinStep = dslContext
                .select(fieldsToRead)
                .from(PHRASES);

        if (clientId != null) {
            joinStep.join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID));
        }

        SelectConditionStep<Record> conditionStep = joinStep.where(PHRASES.PID.in(adGroupIds));

        if (clientId != null) {
            conditionStep.and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()));
        }

        return conditionStep
                .fetchMap(PHRASES.PID, record -> ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(new UntypedAdGroup(), record));
    }

    /**
     * Получение списка существующих групп объявлений указанного клиента,
     * принадлежащих указанному списку.
     *
     * @param shard      шард
     * @param clientId   id клиента
     * @param adGroupIds коллекция id групп объявлений, среди которых осуществляется поиск
     * @return список существующих групп объявлений указанного клиента, являющийся подмножеством указанных
     */
    public Set<Long> getClientExistingAdGroupIds(int shard, ClientId clientId, Collection<Long> adGroupIds) {
        return getClientExistingAdGroupIdsInner(shard, clientId, adGroupIds);
    }

    /**
     * Получение списка существующих групп объявлений,
     * принадлежащих указанному списку.
     *
     * @param shard      шард
     * @param adGroupIds коллекция id групп объявлений, среди которых осуществляется поиск
     * @return список существующих групп объявлений, являющийся подмножеством указанных
     */
    public Set<Long> getExistingAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return getClientExistingAdGroupIdsInner(shard, null, adGroupIds);
    }

    private Set<Long> getClientExistingAdGroupIdsInner(int shard, @Nullable ClientId clientId,
                                                       Collection<Long> adGroupIds) {
        var clientIdCondition = clientId == null
                ? DSL.noCondition()
                : CAMPAIGNS.CLIENT_ID.eq(clientId.asLong());

        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(clientIdCondition)
                .and(PHRASES.PID.in(adGroupIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .fetchSet(PHRASES.PID);
    }

    public List<Long> getAdGroupIdsByClientIds(int shard, List<Long> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(CAMPAIGNS.CLIENT_ID.in(clientIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .fetch(PHRASES.PID);
    }

    public Map<Long, ClientId> getClientIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID, CAMPAIGNS.CLIENT_ID)
                .from(PHRASES)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(PHRASES.PID, rec -> ClientId.fromLong(rec.get(CAMPAIGNS.CLIENT_ID)));
    }

    /**
     * Получить множество id групп, принадлежащих к архивным кампаниям,
     * среди тех, которые переданы в качестве параметра.
     *
     * @param shard      шард
     * @param adGroupIds коллекция id групп, среди которых
     *                   необходимо найти принадлежащие архивным кампаниям
     * @return множество id групп, принадлежащих к архивным кампаниям из тех,
     * которые переданы в качестве параметра
     */
    public Set<Long> getAdGroupIdsInArchivedCampaigns(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES, CAMPAIGNS)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.CID.eq(CAMPAIGNS.CID))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes))
                .fetchSet(PHRASES.PID);
    }

    /**
     * Получить множество id архивных групп, среди тех, которые переданы в качестве параметра.
     * Группа считается архивной, если все баннеры в ней архивные.
     * Группа без баннеров архивной не считается.
     *
     * @param shard      шард
     * @param adGroupIds коллекция id групп, среди которых необходимо найти архивные
     * @return множество id архивных групп
     */
    public Set<Long> getArchivedAdGroupIds(int shard, Collection<Long> adGroupIds) {
        Map<Long, AdGroupStates> adGroupStatuses = getAdGroupStatuses(shard, adGroupIds);

        return EntryStream.of(adGroupStatuses)
                .filterValues(AdGroupStates::getArchived)
                .keys()
                .toSet();
    }

    /**
     * Получить информацию о статусах групп, вычисляемых по их баннерам. Если баннеров в группе нет, не возвращает
     * для этой группы вообще ничего.
     *
     * @param shard      шард
     * @param adGroupIds коллекция id групп
     */
    public Map<Long, AdGroupStates> getAdGroupStatuses(int shard, Collection<Long> adGroupIds) {
        Field<BigDecimal> notArchivedNum = sum(DSL.when(BANNERS.STATUS_ARCH.in(BannersStatusarch.No), 1)
                .otherwise(0)).as("nonArchivedNum");
        Field<BigDecimal> showingNum = sum(DSL.when(BANNERS.STATUS_SHOW.in(BannersStatusshow.Yes), 1)
                .otherwise(0)).as("showingNum");
        Field<BigDecimal> activeNum = sum(DSL.when(BANNERS.STATUS_ACTIVE.in(BannersStatusactive.Yes), 1)
                .otherwise(0)).as("activeNum");
        Field<BigDecimal> sentToBsNum = sum(DSL.when(BANNERS.BANNER_ID.gt(0L), 1)
                .otherwise(0)).as("sentToBsNum");
        Field<BigDecimal> draftNum = sum(DSL.when(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.New), 1)
                .otherwise(0)).as("draftNum");
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.PID, notArchivedNum, showingNum, activeNum, sentToBsNum, draftNum)
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .groupBy(BANNERS.PID)
                .fetchMap(BANNERS.PID, r -> new AdGroupStates()
                        .withActive(r.get(activeNum).compareTo(BigDecimal.ZERO) > 0)
                        .withShowing(r.get(showingNum).compareTo(BigDecimal.ZERO) > 0)
                        .withArchived(r.get(notArchivedNum).compareTo(BigDecimal.ZERO) == 0)
                        .withBsEverSynced(r.get(sentToBsNum).compareTo(BigDecimal.ZERO) > 0)
                        .withHasDraftAds(r.get(draftNum).compareTo(BigDecimal.ZERO) > 0)
                );
    }

    /**
     * @param shard       шард
     * @param campaignIds коллекция id кампаний
     * @return мапа (id кампании -> список id групп)
     */
    public Map<Long, List<Long>> getAdGroupIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        List<Record> records = ppcDslContextProvider.ppc(shard)
                .select(asList(PHRASES.PID, PHRASES.CID))
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .fetch();
        return StreamEx.of(records)
                .mapToEntry(r -> r.get(PHRASES.CID), r -> r.get(PHRASES.PID))
                .grouping();
    }

    /**
     * @param shard        шард
     * @param campaignIds  коллекция id кампаний
     * @param adgroupTypes список типов групп, для которых надо достать id
     * @return мапа (id кампании -> список id групп)
     */
    public Map<Long, List<Long>> getAdGroupIdsByCampaignIdsWithTypes(int shard, Collection<Long> campaignIds,
                                                                     EnumSet<PhrasesAdgroupType> adgroupTypes) {
        List<Record> records = ppcDslContextProvider.ppc(shard)
                .select(asList(PHRASES.PID, PHRASES.CID))
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds).and(PHRASES.ADGROUP_TYPE.in(adgroupTypes)))
                .fetch();
        return StreamEx.of(records)
                .mapToEntry(r -> r.get(PHRASES.CID), r -> r.get(PHRASES.PID))
                .grouping();
    }

    /**
     * По id группы получит id eё кампании.
     *
     * @param shard     шард
     * @param adGroupId id группы
     * @return id кампании или null, если id группы некорректно
     */
    public Long getCampaignIdByAdGroupId(int shard, Long adGroupId) {
        return getCampaignIdsByAdGroupIds(shard, List.of(adGroupId)).get(adGroupId);
    }

    /**
     * @param shard      шард
     * @param adGroupIds коллекция id групп
     * @return мапа (id группы -> id кампании)
     */
    public Map<Long, Long> getCampaignIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(asList(PHRASES.PID, PHRASES.CID))
                .from(PHRASES)
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(r -> r.getValue(PHRASES.PID), r -> r.getValue(PHRASES.CID));
    }

    /**
     * @param shard      шард
     * @param adGroupIds коллекция id групп
     * @return мапа (id группы -> id кампании)
     */
    public Map<Long, Long> getCampaignIdsByAdGroupIds(int shard, ClientId clientId, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(asList(PHRASES.PID, PHRASES.CID))
                .from(PHRASES)
                .join(CAMPAIGNS).on(PHRASES.CID.eq(CAMPAIGNS.CID))
                .where(PHRASES.PID.in(adGroupIds)).and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .fetchMap(r -> r.getValue(PHRASES.PID), r -> r.getValue(PHRASES.CID));
    }

    /**
     * @param shard       шард
     * @param campaignIds коллекция id кампаний
     * @return мапа (id кампании -> кол-во групп)
     */
    public Map<Long, Long> getAdGroupCountByCampaignIds(int shard, Collection<Long> campaignIds) {
        String adGroupIdCounterAlias = "PID_COUNTER";
        return ppcDslContextProvider.ppc(shard)
                .select(asList(PHRASES.CID, count(PHRASES.PID).as(adGroupIdCounterAlias)))
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .groupBy(PHRASES.CID)
                .fetchMap(r -> r.getValue(PHRASES.CID), r -> r.getValue(adGroupIdCounterAlias, Long.class));
    }

    /**
     * По списку ID кампаний получить список ID групп, в которых были изменения с указанного времени.
     *
     * @param shard        шард
     * @param campaignIds  список id кампаний извлекаемых баннеров
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения в группах
     */
    public List<Long> getChangedAdGroupIdsByCampaignIds(int shard, Set<Long> campaignIds, LocalDateTime fromDateTime) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .and(PHRASES.LAST_CHANGE.ge(fromDateTime))
                .fetch(PHRASES.PID);
    }

    /**
     * По списку ID кампаний получить мапу ID кампании -> список ID групп, где в группах были изменения с указанного
     * времени.
     *
     * @param shard        шард
     * @param campaignIds  список id кампаний извлекаемых баннеров
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения в группах
     * @param limit        максимальное количество данных в ответе
     */
    public Map<Long, List<Long>> getChangedAdGroupIdsByCampaignIds(int shard, Set<Long> campaignIds,
                                                                   LocalDateTime fromDateTime, int limit) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.CID, PHRASES.PID)
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .and(PHRASES.LAST_CHANGE.ge(fromDateTime))
                .orderBy(PHRASES.CID)
                .limit(limit)
                .fetchGroups(PHRASES.CID, PHRASES.PID);
    }

    /**
     * Выбирать из списка кампаний такие, в которых произошли изменения в группах, начиная с
     * указанного момента времени.
     *
     * @param shard        шард
     * @param campaignIds  список id кампаний извлекаемых баннеров
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения в группах
     */
    public Set<Long> getChangedCampaignIds(int shard, Set<Long> campaignIds, LocalDateTime fromDateTime) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.CID)
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .and(PHRASES.LAST_CHANGE.ge(fromDateTime))
                .fetchSet(PHRASES.CID);
    }

    /**
     * Актуализация AdGroup, например, при изменении Retargeting'ов.
     * <p>
     * Обновление adGroup.lastChange для всех затронутых AdGroup.
     * Обновление adGroup.statusBsSync = 'No' для тех, кто не в черновиках (adGroup.statusModerate != 'New').
     *
     * @param shard      shard
     * @param adGroupIds набор Id затронутых AdGroup
     */
    public void actualizeAdGroupsOnChildModification(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        // Если adgroup.status_moderate != 'New', надо обновить statusBsSynced = 'No' на adgroup
        updateStatusBsSyncedExceptNew(shard, adGroupIds, StatusBsSynced.NO);
        // обновить adgroup.last_change надо для всех затронутых AdGroup
        updateLastChange(shard, adGroupIds);
    }

    /**
     * Обновить statusBsSynced у переданных adgroup
     *
     * @param shard      шард
     * @param adgroupIds коллекция id групп, в которых нужно обновить статус синхронизации с БК
     * @param status     новый статус синхронизации с БК
     * @return - количество изменённых строк
     */
    public int updateStatusBsSynced(int shard, Collection<Long> adgroupIds, StatusBsSynced status) {
        Configuration conf = ppcDslContextProvider.ppc(shard).configuration();
        return updateStatusBsSynced(conf, adgroupIds, status, false);
    }

    public int updateStatusBsSynced(Configuration conf, Collection<Long> adgroupIds, StatusBsSynced status) {
        return updateStatusBsSynced(conf, adgroupIds, status, false);
    }

    public int updateStatusBsSyncedByCampaignIds(
            Configuration conf, Collection<Long> campaignIds, StatusBsSynced status
    ) {
        return conf.dsl()
                .update(PHRASES)
                .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.valueOf(status.toDbFormat()))
                .where(PHRASES.CID.in(campaignIds))
                .execute();
    }

    /**
     * Обновить statusBsSynced у переданных adgroup, если группы не имеют статус модерации New
     *
     * @param shard      шард
     * @param adgroupIds коллекция id групп, в которых нужно обновить статус синхронизации с БК
     * @param status     новый статус синхронизации с БК
     * @return - количество изменённых строк
     */
    public int updateStatusBsSyncedExceptNew(int shard, Collection<Long> adgroupIds, StatusBsSynced status) {
        Configuration conf = ppcDslContextProvider.ppc(shard).configuration();
        return updateStatusBsSyncedExceptNew(conf, adgroupIds, status);
    }

    /**
     * Обновить statusBsSynced у переданных adgroup, если группы не имеют статус модерации New
     *
     * @param conf       конфиг
     * @param adgroupIds коллекция id групп, в которых нужно обновить статус синхронизации с БК
     * @param status     новый статус синхронизации с БК
     * @return - количество изменённых строк
     */
    public int updateStatusBsSyncedExceptNew(Configuration conf, Collection<Long> adgroupIds, StatusBsSynced status) {
        return updateStatusBsSynced(conf, adgroupIds, status, true);
    }

    /**
     * Обновить LastChange у переданных adgroup, now() вычисляется на стороне БД
     *
     * @param shard      шард
     * @param adgroupIds коллекция id групп, в которых нужно обновить статус синхронизации с БК
     * @return - количество изменённых строк
     */
    public int updateLastChange(int shard, Collection<Long> adgroupIds) {
        return updateLastChange(ppcDslContextProvider.ppc(shard).configuration(), adgroupIds);
    }

    public int updateLastChange(Configuration conf, Collection<Long> adgroupIds) {
        if (adgroupIds.isEmpty()) {
            return 0;
        }
        return conf.dsl()
                .update(PHRASES)
                .set(PHRASES.LAST_CHANGE, DSL.currentLocalDateTime())
                .where(PHRASES.PID.in(adgroupIds))
                .execute();
    }

    public void updateStatusBsSyncedWithLastChange(Configuration conf, Collection<Long> adgroupIds,
                                                   StatusBsSynced status) {
        conf.dsl()
                .update(PHRASES)
                .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.valueOf(status.toDbFormat()))
                .set(PHRASES.LAST_CHANGE, DSL.currentLocalDateTime())
                .where(PHRASES.PID.in(adgroupIds))
                .execute();
    }

    /**
     * Обновляет флаг status_autobudget_show для группы
     *
     * @param shard      шард
     * @param adgroupIds список id групп
     * @param status     новый статус
     */
    public void updateStatusAutoBudgetShow(int shard, Collection<Long> adgroupIds, StatusAutobudgetShow status) {
        Configuration conf = ppcDslContextProvider.ppc(shard).configuration();
        if (adgroupIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_AUTOBUDGET_SHOW, StatusAutobudgetShow.toSource(status))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adgroupIds))
                .execute();
    }

    /**
     * Обновляет флаг status_autobudget_show для группы по номеру кампании
     *
     * @param shard       шард
     * @param campaignIds список id кампаний
     * @param status      новый статус
     */
    public void updateStatusAutoBudgetShowForCampaign(int shard, Collection<Long> campaignIds,
                                                      StatusAutobudgetShow status) {
        updateStatusAutoBudgetShowForCampaign(ppcDslContextProvider.ppc(shard), campaignIds, status);
    }

    public void updateStatusAutoBudgetShowForCampaign(DSLContext context, Collection<Long> campaignIds,
                                                      StatusAutobudgetShow status) {
        if (campaignIds.isEmpty()) {
            return;
        }

        PhrasesStatusautobudgetshow dbStatus = StatusAutobudgetShow.toSource(status);
        context.update(PHRASES)
                .set(PHRASES.STATUS_AUTOBUDGET_SHOW, dbStatus)
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.CID.in(campaignIds)
                        .and(PHRASES.STATUS_AUTOBUDGET_SHOW.notEqual(dbStatus)))
                .execute();
    }

    /**
     * переотправка групп на модерацию, не являющихся смарт-группами и группами внутренней рекламы
     *
     * @see #ADGROUP_TYPES_WITHOUT_MODERATION
     */
    public void dropStatusModerateExceptTypesWithoutModeration(DSLContext context, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        context
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, StatusModerate.toSource(StatusModerate.READY))
                .set(PHRASES.STATUS_POST_MODERATE, DSL.choose(PHRASES.STATUS_POST_MODERATE)
                        .when(StatusPostModerate.toSource(StatusPostModerate.REJECTED),
                                StatusPostModerate.toSource(StatusPostModerate.REJECTED))
                        .otherwise(StatusPostModerate.toSource(StatusPostModerate.NO)))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds)
                        .and(PHRASES.ADGROUP_TYPE.notIn(ADGROUP_TYPES_WITHOUT_MODERATION)))
                .execute();
    }

    /**
     * переотправка групп на модерацию, не являющиеся черновиками
     */
    public void dropStatusModerateExceptDrafts(Configuration conf, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, StatusModerate.toSource(StatusModerate.READY))
                .set(PHRASES.STATUS_POST_MODERATE, DSL.choose(PHRASES.STATUS_POST_MODERATE)
                        .when(StatusPostModerate.toSource(StatusPostModerate.REJECTED),
                                StatusPostModerate.toSource(StatusPostModerate.REJECTED))
                        .otherwise(StatusPostModerate.toSource(StatusPostModerate.NO)))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.notEqual(StatusModerate.toSource(StatusModerate.NEW)))
                .execute();
    }

    public void dropStatusModerateExceptDraftsAndModerated(Configuration conf, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, StatusModerate.toSource(StatusModerate.READY))
                .set(PHRASES.STATUS_POST_MODERATE, DSL.choose(PHRASES.STATUS_POST_MODERATE)
                        .when(StatusPostModerate.toSource(StatusPostModerate.REJECTED),
                                StatusPostModerate.toSource(StatusPostModerate.REJECTED))
                        .otherwise(StatusPostModerate.toSource(StatusPostModerate.NO)))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.notEqual(StatusModerate.toSource(StatusModerate.NEW)))
                .and(PHRASES.STATUS_MODERATE.notEqual(StatusModerate.toSource(StatusModerate.YES)))
                .execute();
    }

    /**
     * Переводит группы, в состояние, соответствующее значительному изменению охвата
     * (любое изменение охвата кроме сужения).
     */
    public void setSignificantlyChangedCoverageState(Configuration conf, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.No)
                .set(PHRASES.STATUS_SHOWS_FORECAST, PhrasesStatusshowsforecast.New)
                .set(PHRASES.LAST_CHANGE, LocalDateTime.now())
                .where(PHRASES.PID.in(adGroupIds))
                .execute();
    }

    /**
     * Обновить информацию о состоянии прогноза показов в группах.
     * <p>
     * Устанавливает {@code statusShowsForecast} равным {@link PhrasesStatusshowsforecast#Processed},
     * а также соответствующие значения времени {@code forecastDate}.
     * <p>
     * Изменяет данные при выполнении следующих условийg: <ul>
     * <li>совпадении значения геотаргетинга с текущим</li>
     * <li>превышении новым значение времени обновления прогноза - текущего</li>
     * </ul>
     *
     * @param shard             шард
     * @param adGroupsForecasts коллекция групп, для которых нужно обновить состояние
     * @return количество обновленных строк в базе данных
     */
    public int updateForecastDateForAnyStatus(int shard, Collection<AdGroupShowsForecast> adGroupsForecasts) {
        return updateForecastDate(shard, adGroupsForecasts, asList(PhrasesStatusshowsforecast.values()));
    }

    /**
     * Обновить информацию о состоянии прогноза показов в группах.
     * <p>
     * Устанавливает {@code statusShowsForecast} равным {@link PhrasesStatusshowsforecast#Processed},
     * а также соответствующие значения времени {@code forecastDate}.
     * <p>
     * Изменяет данные при выполнении следующих условийg: <ul>
     * <li>совпадении значения геотаргетинга с текущим</li>
     * <li>текущем статусе группы {@link PhrasesStatusshowsforecast#Sending}</li>
     * <li>превышении новым значение времени обновления прогноза - текущего</li>
     * </ul>
     *
     * @param shard             шард
     * @param adGroupsForecasts коллекция групп, для которых нужно проставить дату
     * @return количество обновленных строк в базе данных
     */
    public int updateForecastDateForSendingStatus(int shard, Collection<AdGroupShowsForecast> adGroupsForecasts) {
        return updateForecastDate(shard, adGroupsForecasts, singletonList(PhrasesStatusshowsforecast.Sending));
    }

    /**
     * Обновить значение таймстемпа обновления прогноза показов для фраз в группе (forecastDate)
     *
     * @param shard             шард
     * @param adGroupsForecasts коллекция групп, для которых нужно проставить дату
     * @param expectedStatuses  ожидаемые значения статуса прогноза показов
     * @return количество обновленных строк в базе данных
     */
    private int updateForecastDate(int shard, Collection<AdGroupShowsForecast> adGroupsForecasts,
                                   Collection<PhrasesStatusshowsforecast> expectedStatuses) {
        if (adGroupsForecasts.isEmpty()) {
            return 0;
        }
        Field<LocalDateTime> dateCase = JooqMapperUtils.makeCaseStatement(PHRASES.PID, PHRASES.FORECAST_DATE,
                StreamEx.of(adGroupsForecasts)
                        .toMap(AdGroupShowsForecast::getId, AdGroupShowsForecast::getForecastDate));

        return ppcDslContextProvider.ppc(shard)
                .update(PHRASES)
                .set(PHRASES.FORECAST_DATE, dateCase)
                .set(PHRASES.STATUS_SHOWS_FORECAST, PhrasesStatusshowsforecast.Processed)
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(getAdgroupIdAndGeoCondition(adGroupsForecasts)
                        .and(PHRASES.STATUS_SHOWS_FORECAST.in(expectedStatuses))
                        .and(PHRASES.FORECAST_DATE.lessThan(dateCase)))
                .execute();
    }

    /**
     * Получить из переданных прогнозов показов те номера групп, у которых совпадает значение {@code geo}.
     *
     * @param shard             шард
     * @param adGroupsForecasts коллекция прогнозов показов по которой нужно
     * @return adGroupIds для которых значение {@code geo} в базе совпадает со значением из прогноза
     */
    public Set<Long> getAdGroupIdsWithUnchangedGeo(int shard, Collection<AdGroupShowsForecast> adGroupsForecasts) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES)
                .where(getAdgroupIdAndGeoCondition(adGroupsForecasts))
                .fetchSet(PHRASES.PID);
    }

    /**
     * Возвращает {@link Map}, где ключами являются ID найденных adGroup, а значения &ndash; заполненные
     * {@link AdGroupForBannerOperation}
     *
     * @param shard      шард
     * @param adGroupIds набор ID по которым необходимо выполнить поиск
     * @return {@link Map} с {@link AdGroupForBannerOperation}
     */
    public Map<Long, AdGroupForBannerOperation> getAdGroupsForBannerOperation(int shard, Collection<Long> adGroupIds,
                                                                              @Nullable ClientId clientId) {
        var selectQuery = ppcDslContextProvider.ppc(shard).selectQuery();
        selectQuery.addSelect(SIMPLE_ADGROUP_FIELDS_FOR_BANNER_OPERATION);
        selectQuery.addFrom(PHRASES);
        if (clientId != null) {
            selectQuery.addJoin(CAMPAIGNS, CAMPAIGNS.CID.eq(PHRASES.CID));
        }
        selectQuery.addJoin(GROUP_PARAMS, JoinType.LEFT_OUTER_JOIN, GROUP_PARAMS.PID.eq(PHRASES.PID));
        selectQuery.addConditions(PHRASES.PID.in(adGroupIds));
        if (clientId != null) {
            selectQuery.addConditions(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()));
        }
        return selectQuery.fetchMap(PHRASES.PID,
                record -> ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(new UntypedAdGroup(), record));
    }

    /**
     * @param shard      шард
     * @param adGroupIds коллекция id групп, из которых нужно вернуть только динамические и перфоманс
     * @return id динамических и перфоманс групп из переданной в метод коллекции.
     */
    public Collection<Long> getPerformanceAndDynamicAdGroups(int shard, Collection<Long> adGroupIds) {
        List<PhrasesAdgroupType> adGroupTypes =
                asList(PhrasesAdgroupType.dynamic, PhrasesAdgroupType.performance);
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.ADGROUP_TYPE.in(adGroupTypes))
                .fetchSet(PHRASES.PID);
    }

    /**
     * Получить список id групп, у которых есть условия показа, исключая остановленные условия
     * Применимо только для текстовой группы (а также динамиков и смартов)
     *
     * @param shard      шард
     * @param adGroupIds список id групп
     */
    public Set<Long> getAdGroupIdsWithConditions(int shard, Collection<Long> adGroupIds) {
        return getAdGroupIdsWithConditions(ppcDslContextProvider.ppc(shard), adGroupIds);
    }

    /**
     * Получить список id групп, у которых есть условия показа, исключая остановленные условия
     * Применимо только для текстовой группы (а также динамиков и смартов)
     *
     * @param context    — контекст работы с БД
     * @param adGroupIds — список id групп
     */
    public Set<Long> getAdGroupIdsWithConditions(DSLContext context, Collection<Long> adGroupIds) {
        SelectConditionStep<Record1<Long>> adGroupIdsWithPhrases = context
                .selectDistinct(BIDS.PID)
                .from(BIDS)
                .where(BIDS.PID.in(adGroupIds))
                .and(BIDS.IS_SUSPENDED.eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        SelectConditionStep<Record1<Long>> adGroupIdsWithRetargetings = context
                .selectDistinct(BIDS_RETARGETING.PID)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.PID.in(adGroupIds))
                .and(BIDS_RETARGETING.IS_SUSPENDED.eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        SelectConditionStep<Record1<Long>> adGroupIdsWithRelevanceMatches = context
                .selectDistinct(BIDS_BASE.PID)
                .from(BIDS_BASE)
                .where(BIDS_BASE.PID.in(adGroupIds))
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .and(findInSet(BidBaseOpt.DELETED.getTypedValue(), BIDS_BASE.OPTS)
                        .eq(RepositoryUtils.booleanToLong(Boolean.FALSE)))
                .and(findInSet(BidBaseOpt.SUSPENDED.getTypedValue(), BIDS_BASE.OPTS)
                        .eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        SelectConditionStep<Record1<Long>> dynamicAdGroupIds = context
                .selectDistinct(BIDS_DYNAMIC.PID)
                .from(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.PID.in(adGroupIds))
                .and(findInSet(BidDynamicOpt.SUSPENDED.getTypedValue(), BIDS_DYNAMIC.OPTS)
                        .eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        SelectConditionStep<Record1<Long>> performanceAdGroupIds = context
                .selectDistinct(BIDS_PERFORMANCE.PID)
                .from(BIDS_PERFORMANCE)
                .where(BIDS_PERFORMANCE.PID.in(adGroupIds))
                .and(BIDS_PERFORMANCE.IS_SUSPENDED.eq(RepositoryUtils.booleanToLong(Boolean.FALSE)))
                .and(BIDS_PERFORMANCE.IS_DELETED.eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        return adGroupIdsWithPhrases
                .union(adGroupIdsWithRetargetings)
                .union(adGroupIdsWithRelevanceMatches)
                .union(dynamicAdGroupIds)
                .union(performanceAdGroupIds)
                .fetchSet(BIDS.PID);
    }

    /**
     * Получить список id групп, у которых есть условия показа, включая остановленные условия
     * Применимо только для текстовой группы, а также динамиков и смартов
     *
     * @param shard      — шард
     * @param adGroupIds — список id групп
     */
    public Set<Long> getAdGroupIdsWithAnyConditions(int shard, Collection<Long> adGroupIds) {
        return getAdGroupIdsWithAnyConditions(ppcDslContextProvider.ppc(shard), adGroupIds);
    }

    /**
     * Получить список id групп, у которых есть условия показа, включая остановленные условия
     * Применимо только для текстовой группы, а также динамиков и смартов
     *
     * @param context    — контекст работы с БД
     * @param adGroupIds — список id групп
     */
    public Set<Long> getAdGroupIdsWithAnyConditions(DSLContext context, Collection<Long> adGroupIds) {
        SelectConditionStep<Record1<Long>> adGroupIdsWithPhrases = context
                .selectDistinct(BIDS.PID)
                .from(BIDS)
                .where(BIDS.PID.in(adGroupIds));

        SelectConditionStep<Record1<Long>> adGroupIdsWithRetargetings = context
                .selectDistinct(BIDS_RETARGETING.PID)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.PID.in(adGroupIds));

        SelectConditionStep<Record1<Long>> adGroupIdsWithRelevanceMatches = context
                .selectDistinct(BIDS_BASE.PID)
                .from(BIDS_BASE)
                .where(BIDS_BASE.PID.in(adGroupIds))
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .and(findInSet(BidBaseOpt.DELETED.getTypedValue(), BIDS_BASE.OPTS)
                        .eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        SelectConditionStep<Record1<Long>> dynamicAdGroupIds = context
                .selectDistinct(BIDS_DYNAMIC.PID)
                .from(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.PID.in(adGroupIds));

        SelectConditionStep<Record1<Long>> performanceAdGroupIds = context
                .selectDistinct(BIDS_PERFORMANCE.PID)
                .from(BIDS_PERFORMANCE)
                .where(BIDS_PERFORMANCE.PID.in(adGroupIds))
                .and(BIDS_PERFORMANCE.IS_DELETED.eq(RepositoryUtils.booleanToLong(Boolean.FALSE)));

        return adGroupIdsWithPhrases
                .union(adGroupIdsWithRetargetings)
                .union(adGroupIdsWithRelevanceMatches)
                .union(dynamicAdGroupIds)
                .union(performanceAdGroupIds)
                .fetchSet(BIDS.PID);
    }

    /**
     * Получить список id групп, у которых есть объявления
     *
     * @param shard      шард
     * @param adGroupIds список id групп
     */
    public Set<Long> getAdGroupIdsWithBanners(int shard, Collection<Long> adGroupIds) {
        SelectConditionStep<Record1<Long>> adGroupIdsWithBanners = ppcDslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.PID)
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds));

        return adGroupIdsWithBanners.fetchSet(BANNERS.PID);
    }

    /**
     * Получить список id групп, по id объявлений
     *
     * @param shard     шард
     * @param bannerIds список id объявлений
     */
    public Set<Long> getAdGroupIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        SelectConditionStep<Record1<Long>> adGroupIdsWithBanners = ppcDslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.PID)
                .from(BANNERS)
                .where(BANNERS.BID.in(bannerIds));

        return adGroupIdsWithBanners.fetchSet(BANNERS.PID);
    }

    /**
     * Получить список id групп по id групп креативов
     *
     * @param shard            шард
     * @param creativeGroupIds список id групп креативов
     */
    public Set<Long> getAdGroupIdsByCreativeGroupIds(int shard, Collection<Long> creativeGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.PID)
                .from(PERF_CREATIVES
                        .join(BANNERS_PERFORMANCE).on(PERF_CREATIVES.CREATIVE_ID.eq(BANNERS_PERFORMANCE.CREATIVE_ID))
                        .join(BANNERS).on(BANNERS_PERFORMANCE.BID.eq(BANNERS.BID)))
                .where(PERF_CREATIVES.CREATIVE_GROUP_ID.in(creativeGroupIds))
                .fetchSet(BANNERS.PID);
    }


    /**
     * Отправить группы черновики на модерацию
     * Если группа со статусом  StatusModerate = NEW/NO, то устанавливает StatusModerate = READY
     *
     * @param shard      шард
     * @param adGroupIds список id групп которые хотим отправить на модерацию
     */
    public void sendDraftAdGroupToModerate(int shard, Collection<Long> adGroupIds) {
        sendDraftAdGroupToModerate(ppcDslContextProvider.ppc(shard).dsl(), adGroupIds);
    }

    /**
     * Отправить группы черновики на модерацию
     * Если группа со статусом  StatusModerate = NEW/NO, то устанавливает StatusModerate = READY
     *
     * @param adGroupIds список id групп которые хотим отправить на модерацию
     */
    public void sendDraftAdGroupToModerate(DSLContext context, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        context
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, StatusModerate.toSource(StatusModerate.READY))
                .set(PHRASES.STATUS_POST_MODERATE, StatusPostModerate.toSource(StatusPostModerate.NO))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.in(
                        StatusModerate.toSource(StatusModerate.NEW),
                        StatusModerate.toSource(StatusModerate.NO)))
                .execute();
    }

    private void setAdGroupStatusModerateToYes(int shard, Collection<Long> adGroupIds) {
        setAdGroupStatusModerateToYes(ppcDslContextProvider.ppc(shard), adGroupIds);
    }

    /**
     * Для переданных adGroupIds проставляет statusModerate = YES
     * (используется при добавлении performance баннеров)
     */
    private void setAdGroupStatusModerateToYes(DSLContext context, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        context
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, StatusModerate.toSource(StatusModerate.YES))
                .set(PHRASES.STATUS_POST_MODERATE, StatusPostModerate.toSource(StatusPostModerate.YES))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.in(StatusModerate.toSource(StatusModerate.NEW)))
                .execute();
    }

    /**
     * Проставить группам статусы модерации statusModerate = "Yes", statusPostModerate = "Yes".
     */
    public void markAdGroupsAsModerated(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        ppcDslContextProvider.ppc(shard)
                .update(PHRASES)
                .set(PHRASES.STATUS_MODERATE, PhrasesStatusmoderate.Yes)
                .set(PHRASES.STATUS_POST_MODERATE, PhrasesStatuspostmoderate.Yes)
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .execute();
    }

    public void setStatusBlGeneratedProcessing(DSLContext context, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        context.batch(
                context.update(ADGROUPS_TEXT)
                        .set(ADGROUPS_TEXT.STATUS_BL_GENERATED, AdgroupsTextStatusblgenerated.Processing)
                        .where(ADGROUPS_TEXT.PID.in(adGroupIds)),
                context.update(ADGROUPS_DYNAMIC)
                        .set(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED, AdgroupsDynamicStatusblgenerated.Processing)
                        .where(ADGROUPS_DYNAMIC.PID.in(adGroupIds)),
                context.update(ADGROUPS_PERFORMANCE)
                        .set(ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED, AdgroupsPerformanceStatusblgenerated.Processing)
                        .where(ADGROUPS_PERFORMANCE.PID.in(adGroupIds)))
                .execute();
    }

    public void setStatusBlGeneratedForPerformanceAdGroups(int shard, Collection<Long> adGroupIds) {
        setStatusBlGeneratedForPerformanceAdGroups(ppcDslContextProvider.ppc(shard), adGroupIds);
    }

    public void setStatusBlGeneratedForPerformanceAdGroups(DSLContext context, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        context
                .update(ADGROUPS_PERFORMANCE)
                .set(ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED, AdgroupsPerformanceStatusblgenerated.Processing)
                .where(ADGROUPS_PERFORMANCE.PID.in(adGroupIds))
                .execute();
    }

    public void setStatusBlGeneratedForDynamicAdGroupsToProcessingIfNo(DSLContext context,
                                                                       Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        context
                .update(ADGROUPS_DYNAMIC)
                .set(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED, AdgroupsDynamicStatusblgenerated.Processing)
                .where(ADGROUPS_DYNAMIC.PID.in(adGroupIds)
                        .and(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED.eq(AdgroupsDynamicStatusblgenerated.No)))
                .execute();
    }

    public void markDraftPerformanceAdGroupsAsModerated(int shard, ClientId clientId, Collection<Long> adGroupIds) {
        markDraftPerformanceAdGroupsAsModerated(ppcDslContextProvider.ppc(shard), clientId, adGroupIds);
    }

    public void markDraftPerformanceAdGroupsAsModerated(DSLContext dslContext, ClientId clientId,
                                                        Collection<Long> adGroupIds) {
        Map<Long, AdGroupSimple> adGroupsById = getAdGroupSimple(dslContext, clientId, adGroupIds);

        Set<Long> adGroupIdsToModerate = StreamEx.ofValues(adGroupsById)
                .filter(adGroup -> adGroup.getStatusModerate() == StatusModerate.NEW)
                .map(AdGroupSimple::getId)
                .toSet();

        setAdGroupStatusModerateToYes(dslContext, adGroupIdsToModerate);
        updateStatusBsSynced(dslContext.configuration(), adGroupIdsToModerate, StatusBsSynced.NO);
        setStatusBlGeneratedForPerformanceAdGroups(dslContext, adGroupIdsToModerate);
    }

    /**
     * Возвращает {@link Map} с соответсвием ID РМП группы и ID домена publisher'а, если такой есть у группы.
     */
    public Map<Long, Long> getPublisherDomainIds(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_MOBILE_CONTENT.PID, MOBILE_CONTENT.PUBLISHER_DOMAIN_ID)
                .from(ADGROUPS_MOBILE_CONTENT)
                .join(MOBILE_CONTENT)
                .on(MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .where(ADGROUPS_MOBILE_CONTENT.PID.in(adGroupIds))
                .and(MOBILE_CONTENT.PUBLISHER_DOMAIN_ID.isNotNull())
                .fetchMap(ADGROUPS_MOBILE_CONTENT.PID, MOBILE_CONTENT.PUBLISHER_DOMAIN_ID);
    }

    /**
     * Возвращает {@link Map} с соответсвием ID динамической группы и mainDomain, если такой есть у группы.
     */
    public Map<Long, String> getDynamicMainDomains(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }

        return ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_DYNAMIC.PID, DOMAINS.DOMAIN)
                .from(ADGROUPS_DYNAMIC)
                .join(DOMAINS)
                .on(DOMAINS.DOMAIN_ID.eq(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID))
                .where(ADGROUPS_DYNAMIC.PID.in(adGroupIds))
                .fetchMap(ADGROUPS_DYNAMIC.PID, DOMAINS.DOMAIN);
    }

    /**
     * Возвращает {@link Map} с соответсвием ID РМП группы и storeContentHref, если такой есть у группы.
     */
    public Map<Long, String> getMobileStoreContentHrefs(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }

        return ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_MOBILE_CONTENT.PID, ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF)
                .from(ADGROUPS_MOBILE_CONTENT)
                .where(ADGROUPS_MOBILE_CONTENT.PID.in(adGroupIds))
                .fetchMap(ADGROUPS_MOBILE_CONTENT.PID, ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF);
    }

    /**
     * @return {@link Map} с соответсвием ID РМП группы и идентификатора приложения в app store / google play.
     */
    public Map<Long, String> getMobileContentAppIds(int shard, Collection<Long> adGroupIds) {
        return StreamEx.of(ppcDslContextProvider.ppc(shard)
                        .select(ADGROUPS_MOBILE_CONTENT.PID, MOBILE_CONTENT.OS_TYPE,
                                MOBILE_CONTENT.BUNDLE_ID, MOBILE_CONTENT.STORE_CONTENT_ID)
                        .from(ADGROUPS_MOBILE_CONTENT)
                        .join(MOBILE_CONTENT)
                        .on(MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID))
                        .where(ADGROUPS_MOBILE_CONTENT.PID.in(adGroupIds))
                        .fetch())
                .mapToEntry(r -> r.get(ADGROUPS_MOBILE_CONTENT.PID),
                        r -> {  // protected/Direct/Model/MobileContent.pm :: get_store_app_id
                            MobileContentOsType osType = r.get(MOBILE_CONTENT.OS_TYPE);
                            String bundleId = r.get(MOBILE_CONTENT.BUNDLE_ID);
                            String storeContentId = r.get(MOBILE_CONTENT.STORE_CONTENT_ID);
                            if (osType == MobileContentOsType.Android || bundleId == null) {
                                return storeContentId;
                            } else {
                                return bundleId;
                            }
                        })
                .nonNullValues()
                .toMap();
    }

    /**
     * Обновить statusBsSynced у переданных adgroup, с учётом statusModerate
     *
     * @param conf       конфиг
     * @param adgroupIds коллекция групп, в которых нужно обновить статус синхронизации
     * @param status     новый статус
     * @param exceptNew  не обновлять ли statusBsSynced у новых
     * @return - количество изменённых строк
     */
    private int updateStatusBsSynced(Configuration conf, Collection<Long> adgroupIds, StatusBsSynced status,
                                     boolean exceptNew) {
        if (adgroupIds.isEmpty()) {
            return 0;
        }
        UpdateConditionStep<PhrasesRecord> step = DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.valueOf(status.toDbFormat()))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adgroupIds));
        if (exceptNew) {
            step = step.and(PHRASES.STATUS_MODERATE.ne(PhrasesStatusmoderate.New));
        }
        return step.execute();
    }

    /**
     * Обновить statusShowsForecast в группах по указанным id
     */
    public int updateStatusShowsForecast(int shard, Collection<Long> adGroupIds, StatusShowsForecast status) {
        if (adGroupIds.isEmpty()) {
            return 0;
        }
        return ppcDslContextProvider.ppc(shard)
                .update(PHRASES)
                .set(PHRASES.STATUS_SHOWS_FORECAST, StatusShowsForecast.toSource(status))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .execute();
    }

    /**
     * Обновить statusShowsForecast в группах по указанным id
     */
    public void updateStatusShowsForecastExceptDrafts(Configuration conf, Collection<Long> adGroupIds,
                                                      StatusShowsForecast status) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_SHOWS_FORECAST, StatusShowsForecast.toSource(status))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.PID.in(adGroupIds))
                .and(PHRASES.STATUS_MODERATE.ne(StatusModerate.toSource(StatusModerate.NEW)))
                .execute();
    }

    /**
     * Обновить statusShowsForecast в группах с указанным campaignId из списка campaignIds
     *
     * @param campaignIds коллекция id кампаний, по которым нужно обновить статус в группах
     * @param status      новый статус
     * @param shard       шард
     * @return количество измененных строк
     */
    public int updateStatusShowsForecastByCampaignIds(int shard, Collection<Long> campaignIds,
                                                      StatusShowsForecast status) {
        return updateStatusShowsForecastByCampaignIds(ppcDslContextProvider.ppc(shard).configuration(), campaignIds,
                status);
    }

    /**
     * Обновить statusShowsForecast в группах с указанным campaignId из списка campaignIds
     *
     * @param campaignIds коллекция id кампаний, по которым нужно обновить статус в группах
     * @param status      новый статус
     * @param conf        конфиг
     * @return количество измененных строк
     */
    public int updateStatusShowsForecastByCampaignIds(Configuration conf, Collection<Long> campaignIds,
                                                      StatusShowsForecast status) {
        if (campaignIds.isEmpty()) {
            return 0;
        }
        return DSL.using(conf)
                .update(PHRASES)
                .set(PHRASES.STATUS_SHOWS_FORECAST, StatusShowsForecast.toSource(status))
                .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE)
                .where(PHRASES.CID.in(campaignIds))
                .execute();
    }

    /**
     * Выставляет у групп с указанными Id флаг {@code hasPhraseIdHrefs} в {@code true}
     *
     * @param shard      шард
     * @param adGroupIds Id групп, которые нужно обновить
     */
    public void setHasPhraseIdHrefs(int shard, Collection<Long> adGroupIds) {
        setHasPhraseIdHrefs(ppcDslContextProvider.ppc(shard), adGroupIds, true);
    }

    public void setHasPhraseIdHrefs(DSLContext dslContext, Collection<Long> adGroupIds) {
        setHasPhraseIdHrefs(dslContext, adGroupIds, true);
    }

    /**
     * Выставляет у групп с указанными Id флаг {@code hasPhraseIdHrefs} в значение, указанное в параметре {@code flag}
     *
     * @param dslContext DSL-контекст
     * @param adGroupIds Id групп, которые нужно обновить
     * @param flag       значение флага hasPhraseIdHrefs
     */
    void setHasPhraseIdHrefs(DSLContext dslContext, Collection<Long> adGroupIds, Boolean flag) {
        if (adGroupIds.isEmpty()) {
            return;
        }

        InsertValuesStep2<GroupParamsRecord, Long, Long> insert = dslContext
                .insertInto(GROUP_PARAMS, GROUP_PARAMS.PID, GROUP_PARAMS.HAS_PHRASEID_HREF);
        adGroupIds.forEach(id -> insert.values(id, RepositoryUtils.booleanToLong(flag)));
        insert.onDuplicateKeyUpdate()
                .set(GROUP_PARAMS.HAS_PHRASEID_HREF, MySQLDSL.values(GROUP_PARAMS.HAS_PHRASEID_HREF))
                .execute();
    }

    /**
     * Для каждой группы генерирует id (pid) на основе id клиента (ClientID),
     * Выставляет группе сгенерированный id группы для последующей записи в базу.
     *
     * @param adGroups список групп для последующего сохранения в базе
     */
    private void generateAdGroupIds(ClientId clientId, List<AdGroup> adGroups) {
        Iterator<Long> adGroupIds = shardHelper.generateAdGroupIds(clientId.asLong(), adGroups.size()).iterator();
        adGroups.forEach(adGroup -> adGroup.setId(adGroupIds.next()));
    }

    /**
     * Добавляет метки для групп
     */
    private void addTags(Configuration config, List<AdGroup> adGroups) {
        Map<Long, List<Long>> adGroupTagIds = adGroups.stream()
                .filter(ag -> ag.getTags() != null)
                .collect(toMap(AdGroup::getId, AdGroup::getTags));
        if (!adGroupTagIds.isEmpty()) {
            adGroupTagsRepository
                    .addAdGroupTags(config, adGroupTagIds);
        }
    }

    /**
     * Добавляет привязки библиотечных наборов минус фраз
     */
    private void addLibraryMinusKeywords(Configuration config, ClientId clientId, List<AdGroup> adGroups) {
        Map<Long, List<Long>> adGroupMinusKeywordsPackIds = StreamEx.of(adGroups)
                .filter(ag -> ag.getLibraryMinusKeywordsIds() != null)
                .toMap(AdGroup::getId, AdGroup::getLibraryMinusKeywordsIds);
        Set<Long> allMinusKeywordsPackIds = StreamEx.of(adGroupMinusKeywordsPackIds.values())
                .flatMap(Collection::stream)
                .toSet();
        if (!allMinusKeywordsPackIds.isEmpty()) {
            minusKeywordsPackRepository.tryLockMinusKeywordsPacks(config, clientId, allMinusKeywordsPackIds);
            minusKeywordsPackRepository.addAdGroupToPackLinks(config, adGroupMinusKeywordsPackIds);
        }
    }

    /**
     * Добавляет привязки геосегмента
     */
    private void addHyperGeoAdditionalTargetings(Configuration config, ClientId clientId, List<AdGroup> adGroups) {
        List<AdGroupAdditionalTargeting> additionalTargetings = StreamEx.of(adGroups)
                .filter(adGroup -> Objects.nonNull(adGroup.getHyperGeoId()))
                .map(adGroupWithHyperGeo -> (AdGroupAdditionalTargeting)
                        new AuditoriumGeoSegmentsAdGroupAdditionalTargeting()
                                .withAdGroupId(adGroupWithHyperGeo.getId())
                                .withJoinType(AdGroupAdditionalTargetingJoinType.ANY)
                                .withTargetingMode(AdGroupAdditionalTargetingMode.TARGETING)
                                .withValue(listToSet(adGroupWithHyperGeo.getHyperGeoSegmentIds(), identity())))
                .toList();

        adGroupAdditionalTargetingRepository.add(config, clientId, additionalTargetings);
    }

    private void addHyperGeos(Configuration config, List<AdGroup> adGroups) {
        Map<Long, Long> hyperGeoIdByAdGroupId = StreamEx.of(adGroups)
                .filter(adGroup -> Objects.nonNull(adGroup.getHyperGeoId()))
                .toMap(AdGroup::getId, AdGroup::getHyperGeoId);

        hyperGeoRepository.linkHyperGeosToAdGroups(config, hyperGeoIdByAdGroupId);
    }

    private void addContentCategoriesAdditionalTargetings(
            Configuration config, ClientId clientId, List<AdGroup> adGroups) {

        var additionalTargetings = StreamEx.of(adGroups)
                .filter(adGroup -> Objects.nonNull(adGroup.getContentCategoriesRetargetingConditionRules()))
                .map(adGroup -> mapList(
                        adGroup.getContentCategoriesRetargetingConditionRules(),
                        category -> ContentCategoriesTargetingConverter.toTargeting(category).withAdGroupId(adGroup.getId())))
                .flatMap(List::stream)
                .map(targeting -> (AdGroupAdditionalTargeting) targeting)
                .collect(toList());

        adGroupAdditionalTargetingRepository.add(config, clientId, additionalTargetings);
    }

    /**
     * Добавляет группы в таблицы в базе
     */
    private void addAdGroupsToDatabaseTables(Configuration config, ClientId clientId, List<AdGroup> adGroups) {
        adGroupTypeSupportDispatcher.addAdGroupsToDatabaseTables(config.dsl(), clientId, adGroups);
    }

    /**
     * Удаление групп объявлений по Id.
     *
     * @param shard      шард для запроса
     * @param clientId   id клиента
     * @param adGroupIds список id групп объявлений (pid)
     * @return список удаленных id групп объявлений
     */
    public List<Long> delete(int shard, ClientId clientId, long operatorUid, List<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyList();
        }

        Map<Long, Long> campaignIdsByAdGroupIds = getCampaignIdsByAdGroupIds(shard, clientId, adGroupIds);
        Set<BidModifierType> types = EnumSet.allOf(BidModifierType.class);
        Set<BidModifierLevel> levels = Collections.singleton(BidModifierLevel.ADGROUP);
        List<BidModifier> bidModifiers =
                bidModifierRepository.getByAdGroupIds(shard, campaignIdsByAdGroupIds, types, levels);

        ppcDslContextProvider.ppc(shard).transaction(config -> {
            config.dsl().deleteFrom(PHRASES)
                    .where(PHRASES.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_MOBILE_CONTENT)
                    .where(ADGROUPS_MOBILE_CONTENT.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_DYNAMIC)
                    .where(ADGROUPS_DYNAMIC.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_CPM_BANNER)
                    .where(ADGROUPS_CPM_BANNER.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_CPM_VIDEO)
                    .where(ADGROUPS_CPM_VIDEO.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_PERFORMANCE)
                    .where(ADGROUPS_PERFORMANCE.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_CONTENT_PROMOTION)
                    .where(ADGROUPS_CONTENT_PROMOTION.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUP_PAGE_TARGETS)
                    .where(ADGROUP_PAGE_TARGETS.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUP_PRIORITY)
                    .where(ADGROUP_PRIORITY.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUP_PROJECT_PARAMS)
                    .where(ADGROUP_PROJECT_PARAMS.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUP_PROMOACTIONS)
                    .where(ADGROUP_PROMOACTIONS.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_INTERNAL)
                    .where(ADGROUPS_INTERNAL.PID.in(adGroupIds))
                    .execute();
            config.dsl().deleteFrom(ADGROUPS_TEXT)
                    .where(ADGROUPS_TEXT.PID.in(adGroupIds))
                    .execute();
            DSLContext txContext = config.dsl();

            // Удаляем корректировки
            bidModifierRepository.deleteAdjustments(txContext, clientId, operatorUid, bidModifiers);

            hyperGeoRepository.unlinkHyperGeosFromAdGroups(config, adGroupIds);
            usersSegmentRepository.deleteSegmentsWithoutAdGroup(txContext, adGroupIds);
            adGroupAdditionalTargetingRepository.deleteByAdGroupIds(txContext, adGroupIds);
        });
        // в коде ваншота DeleteAdGroups есть перечисление списка таблиц с группами
        // возможно при добавлении новых — стоит добавить и там.

        ppcDslContextProvider.ppc(shard).deleteFrom(GROUP_PARAMS)
                .where(GROUP_PARAMS.PID.in(adGroupIds))
                .execute();

        adGroupTagsRepository.deleteAllAdGroupTags(shard, adGroupIds);
        adGroupBsTagsRepository.deleteAllAdGroupBsTags(shard, adGroupIds);
        minusKeywordsPackRepository.deleteAdGroupToPackLinks(shard, adGroupIds);
        return adGroupIds;
    }

    /**
     * Обновить группы объявлений после удаления ключевых фраз, связанных с ними
     *
     * @param conf       JOOQ конфиг
     * @param adGroupIds Список идентификаторов групп
     */
    public void updateAfterKeywordsDeleted(Configuration conf, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        updateStatusBsSyncedWithLastChange(conf, adGroupIds, StatusBsSynced.NO);
    }

    // аналогично методу ниже, только принимает шард
    public void updateModerationStatusesAfterConditionsAreGone(int shard, Collection<Long> adGroupIds) {
        updateModerationStatusesAfterConditionsAreGone(ppcDslContextProvider.ppc(shard).configuration(), adGroupIds);
    }

    /**
     * Обновить статусы модерации групп объявлений после удаления/выключения условий показа, связанных с ними.
     * Меняет их на Ready/Rejected, если они не черновики и если они без этого не готовы к отправке в БК.
     * Предполагается для работы с группами, потерявшими все условия показа.
     *
     * @param conf       JOOQ конфиг
     * @param adGroupIds Коллекция идентификаторов групп, которым нужно сбросить статусы модерации.
     */
    public void updateModerationStatusesAfterConditionsAreGone(Configuration conf, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }

        conf.dsl()
                .update(PHRASES)
                .set(PHRASES.LAST_CHANGE, LocalDateTime.now())
                .set(PHRASES.STATUS_MODERATE, PhrasesStatusmoderate.Ready)
                .set(PHRASES.STATUS_POST_MODERATE, PhrasesStatuspostmoderate.Rejected)
                .where(PHRASES.PID.in(adGroupIds)
                        .and(PHRASES.STATUS_MODERATE.notEqual(PhrasesStatusmoderate.New)))
                .and(PHRASES.STATUS_POST_MODERATE
                        .notIn(PhrasesStatuspostmoderate.Rejected, PhrasesStatuspostmoderate.Yes))
                .execute();
    }

    /**
     * @param shard      шард
     * @param adGroupIds коллекция id групп
     * @return мапа (id группы -> criterion_type)
     */
    public Map<Long, CriterionType> getCriterionTypeByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        return ppcDslContextProvider.ppc(shard)
                .select(asList(ADGROUPS_CPM_BANNER.PID, ADGROUPS_CPM_BANNER.CRITERION_TYPE))
                .from(ADGROUPS_CPM_BANNER)
                .where(ADGROUPS_CPM_BANNER.PID.in(adGroupIds))
                .fetchMap(r -> r.getValue(ADGROUPS_CPM_BANNER.PID),
                        r -> CriterionType.fromSource(r.getValue(ADGROUPS_CPM_BANNER.CRITERION_TYPE)));
    }

    /**
     * Возвращает map id группы -> тип группы
     *
     * @param shard      шард
     * @param adGroupIds идентификаторы групп
     * @return map id группы -> тип группы
     */
    public Map<Long, AdGroupType> getAdGroupTypesByIds(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID, PHRASES.ADGROUP_TYPE)
                .from(PHRASES)
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(r -> r.getValue(PHRASES.PID), r -> AdGroupType.fromSource(r.getValue(PHRASES.ADGROUP_TYPE)));
    }

    /**
     * Возвращает map id группы -> тип группы
     *
     * @param shard      шард
     * @param adGroupIds идентификаторы групп
     * @return map id группы -> тип группы
     */
    public Map<Long, ContentPromotionAdgroupType> getContentPromotionAdGroupTypesByIds(int shard,
                                                                                       Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_CONTENT_PROMOTION.PID, ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE)
                .from(ADGROUPS_CONTENT_PROMOTION)
                .where(ADGROUPS_CONTENT_PROMOTION.PID.in(adGroupIds))
                .fetchMap(r -> r.getValue(ADGROUPS_CONTENT_PROMOTION.PID), r -> ContentPromotionAdgroupType.fromSource(
                        r.getValue(ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE)));
    }

    /**
     * Возвращает map (id кампании -> тип группы)
     *
     * @param shard       шард
     * @param campaignIds id кампаний
     * @return map (id кампании -> тип группы)
     */
    public Map<Long, ContentPromotionAdgroupType> getContentPromotionAdGroupTypeByCampaignId(int shard,
                                                                                             Collection<Long> campaignIds) {
        Map<Long, Set<ContentPromotionAdgroupType>> contentPromotionAdgroupTypesByCampaignId =
                StreamEx.of(ppcDslContextProvider.ppc(shard)
                                .select(PHRASES.CID, ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE)
                                .from(PHRASES)
                                .join(ADGROUPS_CONTENT_PROMOTION).on(PHRASES.PID.eq(ADGROUPS_CONTENT_PROMOTION.PID))
                                .where(PHRASES.CID.in(campaignIds))
                                .fetchStream())
                        .mapToEntry(r -> r.getValue(PHRASES.CID), r -> ContentPromotionAdgroupType.fromSource(
                                r.getValue(ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE)))
                        .grouping(toSet());

        return EntryStream.of(contentPromotionAdgroupTypesByCampaignId)
                .peekKeyValue((campaignId, contentPromotionAdgroupTypes) -> {
                    if (contentPromotionAdgroupTypes.size() > 1) {
                        logger.error("Multiple content promotion types found for campaign {}", campaignId);
                    }
                })
                .mapValues(contentPromotionAdgroupTypes -> contentPromotionAdgroupTypes.iterator().next())
                .toMap();
    }

    /**
     * Возвращает map id группы -> тип группы
     *
     * @param shard      шард
     * @param clientId   Id клиента
     * @param adGroupIds идентификаторы групп
     * @return map id группы -> тип группы
     */
    public Map<Long, ContentPromotionAdgroupType> getContentPromotionAdGroupTypesByIds(int shard,
                                                                                       ClientId clientId,
                                                                                       Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_CONTENT_PROMOTION.PID, ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE)
                .from(ADGROUPS_CONTENT_PROMOTION)
                .join(PHRASES).on(PHRASES.PID.eq(ADGROUPS_CONTENT_PROMOTION.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(ADGROUPS_CONTENT_PROMOTION.PID.in(adGroupIds)
                        .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetchMap(r -> r.getValue(ADGROUPS_CONTENT_PROMOTION.PID), r -> ContentPromotionAdgroupType.fromSource(
                        r.getValue(ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE)));
    }

    /**
     * Возвращает map id кампании -> список простых групп
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     * @return map id кампании -> список простых групп
     */
    public Map<Long, List<AdGroupSimple>> getAdGroupSimpleByCampaignsIds(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, List<AdGroup>> groupsByCampaignId = ppcDslContextProvider.ppc(shard)
                .select(SIMPLE_ADGROUP_FIELDS_TO_READ)
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .fetchGroups(PHRASES.CID, record -> ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(new UntypedAdGroup(),
                        record));
        return EntryStream.of(groupsByCampaignId)
                .mapValues(adGroups ->
                        mapList(adGroups, AdGroupSimple.class::cast))
                .toMap();
    }

    public Map<Long, GeoproductAvailability> getGeoproductAvailabilityByCampaignId(int shard,
                                                                                   Collection<Long> campaignIds) {
        Map<Long, Boolean> anyGeoproductAdGroupByCampaignId = existAnyGeoproductAdGroupByCampaignId(shard, campaignIds);
        return listToMap(campaignIds,
                identity(),
                cid -> {
                    Boolean anyGeoProduct = anyGeoproductAdGroupByCampaignId.get(cid);
                    return GeoproductAvailability.convert(anyGeoProduct);
                });
    }

    public Map<Long, List<Long>> getAdGroupIdsByFeedId(int shard, Collection<Long> feedIds) {
        return getAdGroupIdsByFeedId(shard, feedIds, EnumSet.of(AdGroupType.PERFORMANCE, AdGroupType.DYNAMIC));
    }

    public Map<Long, List<Long>> getAdGroupIdsByFeedId(int shard, Collection<Long> feedIds,
                                                       EnumSet<AdGroupType> adGroupTypes) {
        Field<Long> feedIdField = DSL.field("feed_id", Long.class);
        Field<Long> pidField = DSL.field("pid", Long.class);
        var selectPerfAdGroupIdsByFeedId = ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_PERFORMANCE.FEED_ID.as(feedIdField),
                        ADGROUPS_PERFORMANCE.PID.as(pidField))
                .from(ADGROUPS_PERFORMANCE)
                .where(adGroupTypes.contains(AdGroupType.PERFORMANCE) ? ADGROUPS_PERFORMANCE.FEED_ID.in(feedIds) :
                        DSL.falseCondition());

        var selectDynamicAdGroupIdsByFeedId = ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_DYNAMIC.FEED_ID.as(feedIdField),
                        ADGROUPS_DYNAMIC.PID.as(pidField))
                .from(ADGROUPS_DYNAMIC)
                .where(adGroupTypes.contains(AdGroupType.DYNAMIC) ? ADGROUPS_DYNAMIC.FEED_ID.in(feedIds) :
                        DSL.falseCondition());

        return selectPerfAdGroupIdsByFeedId
                .union(selectDynamicAdGroupIdsByFeedId)
                .fetchGroups(feedIdField, pidField);
    }

    /**
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     * @return отображение: идентификатор кампании -> {@code true}, если для этой кампании есть хотя бы одна группа
     * с типом {@link AdGroupType#CPM_GEOPRODUCT} или {@link AdGroupType#CPM_GEO_PIN}
     */
    private Map<Long, Boolean> existAnyGeoproductAdGroupByCampaignId(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        Field<BigDecimal> geoproductCount =
                sum(DSL.iif(PHRASES.ADGROUP_TYPE.in(PhrasesAdgroupType.cpm_geoproduct,
                        PhrasesAdgroupType.cpm_geo_pin), 1, 0));
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.CID, geoproductCount)
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .groupBy(PHRASES.CID)
                .fetchMap(r -> r.get(PHRASES.CID), r -> r.get(geoproductCount).compareTo(BigDecimal.ZERO) > 0);
    }

    public Map<Long, Long> getAdGroupsPriority(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        return getAdGroupsPriority(ppcDslContextProvider.ppc(shard), adGroupIds);
    }

    public Map<Long, Long> getAdGroupsPriority(DSLContext dslContext, Collection<Long> adGroupIds) {
        return dslContext
                .select(ADGROUP_PRIORITY.PID, ADGROUP_PRIORITY.PRIORITY)
                .from(ADGROUP_PRIORITY)
                .where(ADGROUP_PRIORITY.PID.in(adGroupIds))
                .fetchMap(ADGROUP_PRIORITY.PID, ADGROUP_PRIORITY.PRIORITY);
    }

    public List<Long> getCpmBannersAdGroupsWithoutCriterionType(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID)
                .from(PHRASES)
                .leftJoin(ADGROUPS_CPM_BANNER).on(PHRASES.PID.eq(ADGROUPS_CPM_BANNER.PID))
                .where(PHRASES.PID.in(adGroupIds).and(PHRASES.ADGROUP_TYPE.eq(PhrasesAdgroupType.cpm_banner)
                        .and(ADGROUPS_CPM_BANNER.CRITERION_TYPE.isNull())))
                .fetch(PHRASES.PID);
    }

    /**
     * Получить дефолтные группы прайсовых кампаний.
     * Если по какой-то причине у кампании несколько дефолтных групп, то возвращаем первую попавшуюся.
     * Если дефолтной группы нет - в мапе не будет соответствующей записи
     *
     * @param shard       - шард
     * @param campaignIds - идентификаторы кампаний
     * @return мапа campaignId -> defaultAdGroupId
     */
    public Map<Long, Long> getDefaultPriceSalesAdGroupIdByCampaignId(int shard, Collection<Long> campaignIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.CID, max(PHRASES.PID).as(PHRASES.PID))
                .from(PHRASES)
                .join(ADGROUP_PRIORITY).on(ADGROUP_PRIORITY.PID.eq(PHRASES.PID))
                .where(PHRASES.CID.in(campaignIds))
                .and(ADGROUP_PRIORITY.PRIORITY.eq(AdGroupCpmPriceUtils.PRIORITY_DEFAULT))
                .groupBy(PHRASES.CID)
                .fetchMap(PHRASES.CID, PHRASES.PID);
    }
}
