package ru.yandex.direct.core.entity.adgroup.repository.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertOnDuplicateSetStep;
import org.jooq.InsertValuesStep3;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupBsTags;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupBsTagsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings.targetTagsFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_BS_TAGS;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Repository
@ParametersAreNonnullByDefault
public class AdGroupBsTagsRepository {

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    private static final JooqWriter ADGROUP_BSTAGS_WRITER =
            JooqWriterBuilder.<AdGroupBsTags>builder()
                    .writeField(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON,
                            fromProperty(AdGroupBsTags.PAGE_GROUP_TAGS)
                                    .by(tagsList -> tagsList == null ? null : JsonUtils.toJson(tagsList)))
                    .writeField(ADGROUP_BS_TAGS.TARGET_TAGS_JSON,
                            fromProperty(AdGroupBsTags.TARGET_TAGS)
                                    .by(tagsList -> tagsList == null ? null : JsonUtils.toJson(tagsList)))
                    .build();

    @Autowired
    public AdGroupBsTagsRepository(DslContextProvider dslContextProvider,
                                   ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    /**
     * Проставление дефолтных тегов для отправки в БК (таблица ppc.adgroup_bs_tags) для всех групп переданного списка
     * Не перезаписывает уже приписанные группам теги
     *
     * @param clientId    идентификатор клиента
     * @param adGroupIds  список идентификаторов групп, которым надо проставить теги
     * @param defaultTags дефолтные теги
     */
    public void addDefaultTagsForAdGroupList(ClientId clientId,
                                             Collection<Long> adGroupIds,
                                             AdGroupBsTags defaultTags) {
        Map<Long, AdGroupBsTags> bsTagsByAdGroupIds = adGroupIds.stream()
                .collect(Collectors.toMap(identity(), t -> defaultTags));
        addAdGroupPageTags(clientId, bsTagsByAdGroupIds);
    }

    /**
     * Добавляет теги на группы для отправки в БК (таблица ppc.adgroup_bs_tags). Имеющиеся не обновляет
     *
     * @param bsTagsByAdGroupIds маппинг id группы на списки id добавляемых тегов
     */
    public void addAdGroupPageTags(ClientId clientId, Map<Long, AdGroupBsTags> bsTagsByAdGroupIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        addAdGroupPageTagsInternal(shard, bsTagsByAdGroupIds, false);
    }

    /**
     * Добавляет теги на группы для отправки в БК (таблица ppc.adgroup_bs_tags). Имеющиеся не обновляет
     *
     * @param bsTagsByAdGroupIds маппинг id группы на списки id добавляемых тегов
     */
    public void addAdGroupPageTags(int shard, Map<Long, AdGroupBsTags> bsTagsByAdGroupIds) {
        addAdGroupPageTagsInternal(shard, bsTagsByAdGroupIds, false);
    }

    /**
     * Добавляет теги на группы для отправки в БК (таблица ppc.adgroup_bs_tags)
     * Перезатирает имеющиеся новыми значениями
     *
     * @param bsTagsByAdGroupIds маппинг id группы на списки id добавляемых тегов
     */
    public void addAdGroupPageTagsForce(int shard, Map<Long, AdGroupBsTags> bsTagsByAdGroupIds) {
        addAdGroupPageTagsInternal(shard, bsTagsByAdGroupIds, true);
    }

    /**
     * Добавляет теги на группы для отправки в БК (таблица ppc.adgroup_bs_tags)
     *
     * @param shard              номер шарда
     * @param bsTagsByAdGroupIds маппинг id группы на списки id добавляемых тегов
     * @param isForce            перезаписывать ли имеющиеся записи
     */
    private void addAdGroupPageTagsInternal(int shard, Map<Long, AdGroupBsTags> bsTagsByAdGroupIds, boolean isForce) {
        InsertHelper<AdgroupBsTagsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), ADGROUP_BS_TAGS);
        bsTagsByAdGroupIds.forEach((adGroupId, tags) -> {
            insertHelper.add(ADGROUP_BSTAGS_WRITER, tags)
                    .set(ADGROUP_BS_TAGS.PID, adGroupId)
                    .newRecord();
        });
        if (isForce) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON, MySQLDSL.values(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON))
                    .set(ADGROUP_BS_TAGS.TARGET_TAGS_JSON, MySQLDSL.values(ADGROUP_BS_TAGS.TARGET_TAGS_JSON));
        } else {
            insertHelper.onDuplicateKeyIgnore();
        }
        insertHelper.executeIfRecordsAdded();
    }

    /**
     * Отвязывает теги для отправки в бк
     *
     * @param adGroupIds id групп, от которых надо отвзяать все метки
     */
    public void deleteAllAdGroupBsTags(int shard, List<Long> adGroupIds) {
        deleteAllAdGroupBsTags(dslContextProvider.ppc(shard), adGroupIds);
    }

    public void deleteAllAdGroupBsTags(DSLContext dslContext, List<Long> adGroupIds) {
        dslContext
                .deleteFrom(ADGROUP_BS_TAGS)
                .where(ADGROUP_BS_TAGS.PID.in(adGroupIds))
                .execute();
    }

    public void removeTagsByAdgroupIds(int shard, @Nullable String pageGroupTag, @Nullable String targetTag,
                                       Collection<Long> adgroupIds) {

        if (adgroupIds.isEmpty()) {
            return;
        }

        if (pageGroupTag == null && targetTag == null) {
            return; // throw?
        } else if (pageGroupTag != null) {
            executeRemovingUpdate(shard, ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON, adgroupIds, pageGroupTag);
        } else {
            executeRemovingUpdate(shard, ADGROUP_BS_TAGS.TARGET_TAGS_JSON, adgroupIds, targetTag);
        }

        dslContextProvider.ppc(shard)
                .deleteFrom(ADGROUP_BS_TAGS)
                .where(jsonLength(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON).eq(0L),
                        jsonLength(ADGROUP_BS_TAGS.TARGET_TAGS_JSON).eq(0L),
                        ADGROUP_BS_TAGS.PID.in(adgroupIds)
                )
                .execute();

    }

    private void executeRemovingUpdate(int shard, TableField<AdgroupBsTagsRecord, String> field,
                                       Collection<Long> adgroupIds, String value) {
        dslContextProvider.ppc(shard)
                .update(ADGROUP_BS_TAGS)
                .set(field, jsonRemove(field, value))
                .where(jsonContains(field, value).eq(1), ADGROUP_BS_TAGS.PID.in(adgroupIds))
                .execute();
    }

    public Map<Long, List<String>> getTargetTagsByAdgroupIds(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return Collections.emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(ADGROUP_BS_TAGS.PID, ADGROUP_BS_TAGS.TARGET_TAGS_JSON)
                .from(ADGROUP_BS_TAGS)
                .where(ADGROUP_BS_TAGS.PID.in(adGroupIds))
                .fetchMap(ADGROUP_BS_TAGS.PID, r -> targetTagsFromDb(r.get(ADGROUP_BS_TAGS.TARGET_TAGS_JSON)));
    }

    /**
     * Проставляет теги на группы для отправки в БК (таблица ppc.adgroup_bs_tags)
     * Перезатирает имеющиеся новыми значениями
     */
    public void setRawTagsByAdgroupIds(int shard, Set<String> pageGroupTags, Set<String> targetTags,
                                       Collection<Long> adgroupIds) {
        if (adgroupIds.isEmpty()) {
            return;
        }

        InsertValuesStep3<AdgroupBsTagsRecord, Long, String, String> insertStep = prepareInsertStep(shard,
                pageGroupTags, targetTags, adgroupIds);
        insertStep.onDuplicateKeyUpdate()
                .set(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON, MySQLDSL.values(ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON))
                .set(ADGROUP_BS_TAGS.TARGET_TAGS_JSON, MySQLDSL.values(ADGROUP_BS_TAGS.TARGET_TAGS_JSON));
        insertStep.execute();
    }

    public void setRawTagsByAdgroupIds(int shard, @Nullable String pageGroupTag, @Nullable String targetTag,
                                       Collection<Long> adgroupIds,
                                       boolean update) {

        if (adgroupIds.isEmpty()) {
            return;
        }

        InsertValuesStep3<AdgroupBsTagsRecord, Long, String, String> insertStep = prepareInsertStep(shard,
                pageGroupTag == null ? Set.of() : Set.of(pageGroupTag),
                targetTag == null ? Set.of() : Set.of(targetTag), adgroupIds);

        InsertOnDuplicateSetStep<AdgroupBsTagsRecord> onDuplicateSetMoreStep;
        onDuplicateSetMoreStep = insertStep.onDuplicateKeyUpdate();

        onDuplicateSetMoreStep = prepareOnDuplicateUpdate(onDuplicateSetMoreStep,
                ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON, pageGroupTag, update);

        prepareOnDuplicateUpdate(onDuplicateSetMoreStep, ADGROUP_BS_TAGS.TARGET_TAGS_JSON, targetTag, update);

        insertStep.execute();
    }

    private InsertOnDuplicateSetStep<AdgroupBsTagsRecord> prepareOnDuplicateUpdate(InsertOnDuplicateSetStep<AdgroupBsTagsRecord> onDuplicateSetMoreStep,
                                                                                   TableField<AdgroupBsTagsRecord,
                                                                                           String> field,
                                                                                   @Nullable String candidate,
                                                                                   boolean update) {

        if (update && candidate != null) {
            onDuplicateSetMoreStep = onDuplicateSetMoreStep
                    .set(field,
                            DSL.iif(
                                    DSL.ifnull(jsonContains(field, candidate),0).eq(0),
                                    jsonAppend(field, candidate),
                                    field
                            )
                    );
        } else if (candidate != null) {
            onDuplicateSetMoreStep = onDuplicateSetMoreStep
                    .set(field, MySQLDSL.values(field));
        } else {
            onDuplicateSetMoreStep = onDuplicateSetMoreStep
                    .set(field, field);
        }

        return onDuplicateSetMoreStep;
    }

    private InsertValuesStep3<AdgroupBsTagsRecord, Long, String, String> prepareInsertStep(int shard,
                                                                                           Set<String> pageGroupTags,
                                                                                           Set<String> targetTags,
                                                                                           Collection<Long> adgroupIds) {
        InsertValuesStep3<AdgroupBsTagsRecord, Long, String, String> insertStep = dslContextProvider.ppc(shard)
                .insertInto(ADGROUP_BS_TAGS)
                .columns(ADGROUP_BS_TAGS.PID, ADGROUP_BS_TAGS.PAGE_GROUP_TAGS_JSON, ADGROUP_BS_TAGS.TARGET_TAGS_JSON);

        String pageTagsEncoded = JsonUtils.toJson(pageGroupTags);
        String targetTagsEncoded = JsonUtils.toJson(targetTags);

        for (var pid : adgroupIds) {
            insertStep = insertStep.values(pid, pageTagsEncoded, targetTagsEncoded);
        }

        return insertStep;
    }

    private Field<Long> jsonLength(Field<?> field) {
        return DSL.field("json_length({0})", Long.class, field);
    }

    private Field<String> jsonRemove(Field<?> field, String value) {
        return DSL.field("json_remove({0}, json_unquote(json_search({0}, 'one', {1})))", String.class, field,
                DSL.inline(value));
    }

    private Field<String> jsonAppend(Field<?> field, String value) {
        return DSL.field("json_array_append(ifnull({0},'[]'), '$', {1})", String.class, field, DSL.inline(value));
    }

    private Field<Integer> jsonContains(Field<?> field, String candidate) {
        return DSL.field("json_contains(ifnull({0},'[]'), json_quote({1}))", Integer.class, field, DSL.inline(candidate));
    }

}
