package ru.yandex.direct.core.entity.adgroup.repository.internal;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.InsertValuesStep2;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.tables.records.TagGroupRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.TAG_GROUP;

@Repository
public class AdGroupTagsRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public AdGroupTagsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public Map<Long, List<Long>> getAdGroupsTags(int shard, Collection<Long> adGroupIds) {
        return getAdGroupsTags(dslContextProvider.ppc(shard).configuration(), adGroupIds);
    }

    /**
     * Возвращает все привязанные теги для групп
     *
     * @param adGroupIds id групп, для которых надо получить id привязанных к ним тегов
     * @return маппинг id группы на список id тегов
     */
    public Map<Long, List<Long>> getAdGroupsTags(Configuration config, Collection<Long> adGroupIds) {
        return config.dsl()
                .select(TAG_GROUP.PID, TAG_GROUP.TAG_ID)
                .from(TAG_GROUP)
                .where(TAG_GROUP.PID.in(adGroupIds))
                .fetchGroups(TAG_GROUP.PID, r -> r.get(TAG_GROUP.TAG_ID));
    }

    /**
     * Добавляет метки к группе (таблица ppc.tag_group)
     *
     * @param adGroupTagIds маппинг id группы на список id добавляемых меток
     */
    public void addAdGroupTags(Configuration config, Map<Long, ? extends Collection<Long>> adGroupTagIds) {
        InsertValuesStep2<TagGroupRecord, Long, Long> insertStep = config.dsl()
                .insertInto(TAG_GROUP)
                .columns(TAG_GROUP.PID, TAG_GROUP.TAG_ID);
        adGroupTagIds.forEach((adGroupId, tagIds) ->
                tagIds.forEach(id -> insertStep.values(adGroupId, id)));
        insertStep.execute();
    }

    /**
     * Отвязывает заданные метки от группы
     *
     * @param adGroupTagIds маппинг id группы на список id отвязываемых меток
     */
    public void deleteFromAdGroup(Configuration config, Map<Long, ? extends Collection<Long>> adGroupTagIds) {
        Condition deleteCondition = DSL.falseCondition();
        for (Map.Entry<Long, ? extends Collection<Long>> adGroupTags : adGroupTagIds.entrySet()) {
            deleteCondition = deleteCondition
                    .or(TAG_GROUP.PID.eq(adGroupTags.getKey()).and(TAG_GROUP.TAG_ID.in(adGroupTags.getValue())));
        }

        config.dsl()
                .deleteFrom(TAG_GROUP)
                .where(deleteCondition)
                .execute();
    }

    /**
     * Отвязывает все метки от заданных групп
     *
     * @param adGroupIds id групп, от которых надо отвзяать все метки
     */
    public void deleteAllAdGroupTags(int shard, List<Long> adGroupIds) {
        dslContextProvider.ppc(shard)
                .deleteFrom(TAG_GROUP)
                .where(TAG_GROUP.PID.in(adGroupIds))
                .execute();
    }
}
