package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;

/**
 * Интерфейс классов, которые реализуют зависящую от типа логику работы с группами
 * объявлений в репозиториях: сохранение данных группы в базу, извлечение данных
 * группы из базы, превращение данных из базы в объект AdGroup нужного подкласса
 * с заполненными нужными полями.
 * <p>
 * Один класс отвечает за один тип группы.
 */
@ParametersAreNonnullByDefault
interface AdGroupTypeSupport<T extends AdGroup> {
    /**
     * За какой тип группы отвечает этот класс
     */
    AdGroupType adGroupType();

    Class<T> getAdGroupClass();

    /**
     * Записать данные из групп объявлений в таблицы в БД, специфичные для этого типа.
     * Вызывающий код должен гарантировать, что в списке групп только группы типа,
     * за который отвечает этот объект.
     * <p>
     * Реализации обязаны учесть и данные, специфичные для этого типа, и данные,
     * общие для всех типов. Для общих для всех типов данных можно воспользоваться
     * {@link ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common#addAdGroupsToCommonTables}
     */
    void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId, List<T> adGroups);

    /**
     * Выполнить обновление таблиц базы данных с учетом типа для заданного списка типов объявлений
     */
    default void updateAdGroups(Collection<AppliedChanges<T>> adGroups, ClientId clientId, DSLContext dslContext) {
    }

    /**
     * Превратить запись из результатов SELECT-запроса в базу в объект модели.
     * Вызывающий код гарантирует, что в записи есть все нужные данные из нескольких таблиц.
     * <p>
     * Реализации обязаны учесть и данные, специфичные для этого типа, и данные,
     * общие для всех типов. Для общих для всех типов данных можно воспользоваться
     * {@link ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common#ADGROUP_MAPPER_FOR_COMMON_FIELDS}
     */
    T constructInstanceFromDb(Record record);
}
