package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableSet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;

/**
 * Компонент, который разруливает правильное сохранение и извлечение из базы
 * групп объявлений разных типов. Разные AdGroupTypeSupport в большинстве случаев
 * должны быть скрыты за ним, а он разрулит, какой тип каким образом
 * сохранять/извлекать.
 */
@ParametersAreNonnullByDefault
@Component
public class AdGroupTypeSupportDispatcher {

    private final Map<Class<? extends AdGroup>, AdGroupTypeSupport<AdGroup>> typeSupportMap;

    // NB: здесь могут быть значения null, если по одному лишь PhrasesAdgroupType определить обработчик нельзя
    private final Map<PhrasesAdgroupType, AdGroupTypeSupport<AdGroup>> typeSupportMapByType;

    private final Set<PhrasesAdgroupType> allSupportedDbTypes;

    private final AdGroupTypeSupport<AdGroup> dynamicTextAdGroupSupport;
    private final AdGroupTypeSupport<AdGroup> dynamicFeedAdGroupSupport;

    @Autowired
    public AdGroupTypeSupportDispatcher(
            TextAdGroupSupport textAdGroupSupport,
            DynamicTextAdGroupSupport dynamicTextAdGroupSupport,
            DynamicFeedAdGroupSupport dynamicFeedAdGroupSupport,
            MobileContentAdGroupSupport mobileContentAdGroupSupport,
            PerformanceAdGroupSupport performanceAdGroupSupport,
            McBannerAdGroupSupport mcbannerAdGroupSupport,
            CpmBannerAdGroupSupport cpmBannerAdGroupSupport,
            CpmGeoproductAdGroupSupport cpmGeoproductAdGroupSupport,
            CpmGeoPinAdGroupSupport cpmGeoPinAdGroupSupport,
            CpmAudioAdGroupSupport cpmAudioAdGroupSupport,
            CpmVideoAdGroupSupport cpmVideoAdGroupSupport,
            CpmOutdoorAdGroupSupport cpmOutdoorAdGroupSupport,
            CpmIndoorAdGroupSupport cpmIndoorAdGroupSupport,
            CpmYndxFrontpageAdGroupSupport cpmYndxFrontpageAdGroupSupport,
            ContentPromotionVideoAdGroupSupport contentPromotionVideoAdGroupSupport,
            ContentPromotionAdGroupSupport contentPromotionAdGroupSupport,
            InternalAdGroupSupport internalAdGroupSupport) {
        this.dynamicTextAdGroupSupport = upCast(dynamicTextAdGroupSupport);
        this.dynamicFeedAdGroupSupport = upCast(dynamicFeedAdGroupSupport);

        List<AdGroupTypeSupport<AdGroup>> typeSupports = asList(
                upCast(textAdGroupSupport),
                upCast(dynamicTextAdGroupSupport),
                upCast(dynamicFeedAdGroupSupport),
                upCast(mobileContentAdGroupSupport),
                upCast(performanceAdGroupSupport),
                upCast(mcbannerAdGroupSupport),
                upCast(cpmBannerAdGroupSupport),
                upCast(cpmGeoproductAdGroupSupport),
                upCast(cpmGeoPinAdGroupSupport),
                upCast(cpmAudioAdGroupSupport),
                upCast(cpmVideoAdGroupSupport),
                upCast(cpmOutdoorAdGroupSupport),
                upCast(cpmIndoorAdGroupSupport),
                upCast(cpmYndxFrontpageAdGroupSupport),
                upCast(contentPromotionVideoAdGroupSupport),
                upCast(contentPromotionAdGroupSupport),
                upCast(internalAdGroupSupport));

        this.typeSupportMap = typeSupports.stream().collect(toMap(AdGroupTypeSupport::getAdGroupClass, identity()));
        this.allSupportedDbTypes = unmodifiableSet(typeSupports.stream()
                .map(type -> AdGroupType.toSource(type.adGroupType()))
                .collect(toSet()));

        this.typeSupportMapByType = typeSupports.stream()
                .collect(toMap(type -> AdGroupType.toSource(type.adGroupType()), identity(),
                        // если два класса поддерживают один и тот же adGroupType, разруливать их только по нему
                        // нельзя, надо дописать логику в getTypeSupportForDbRecord или выполение споткнётся
                        // об этот null
                        (u, v) -> null,
                        () -> new EnumMap<>(PhrasesAdgroupType.class)));
    }

    // В typeSupport обработчики типов хранятся типизированные как от AdGroup, хотя
    // на самом деле они типизированы своими типами. Downcast AdGroup -> TypedAdGroup
    // интерпретатор делает за нас.
    @SuppressWarnings("unchecked")
    private <T extends AdGroup> AdGroupTypeSupport<AdGroup> upCast(AdGroupTypeSupport<T> typeSupport) {
        return (AdGroupTypeSupport<AdGroup>) typeSupport;
    }

    /**
     * Создать объект модели из извлечённой записи в базе. Правильно извлечь все поля непросто,
     * код, который это делает, можно посмотреть в {@link AdGroupRepository#getAdGroups(int, Collection)}
     *
     * @param record запись в базе
     * @return объект модели; поскольку возвращаемый класс абстрактный, на самом деле возвращается объект подкласса
     */
    public AdGroup constructInstanceFromDb(Record record) {
        return getTypeSupportForDbRecord(record).constructInstanceFromDb(record);
    }

    private AdGroupTypeSupport<AdGroup> getTypeSupportForDbRecord(Record record) {
        PhrasesAdgroupType dbType = record.get(PHRASES.ADGROUP_TYPE);

        if (dbType == PhrasesAdgroupType.dynamic) {
            Long mainDomainId = record.get(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID);
            Long feedId = record.get(ADGROUPS_DYNAMIC.FEED_ID);

            if (mainDomainId != null && mainDomainId != 0) {
                return dynamicTextAdGroupSupport;
            } else if (feedId != null && feedId != 0) {
                return dynamicFeedAdGroupSupport;
            } else {
                throw new IllegalStateException("invalid database record for dynamic adgroup: " + record.intoMap());
            }
        }

        return checkNotNull(typeSupportMapByType.get(dbType));
    }

    /**
     * Какие типы (с точки зрения хранения в базе) поддерживаются всеми TypeSupport-ами
     *
     * @return коллекция PhrasesAdgroupType, по которой можно фильтровать выборку из базы
     */
    public Collection<PhrasesAdgroupType> allSupportedDbTypes() {
        return allSupportedDbTypes;
    }

    /**
     * Записать объекты моделей в базу новыми записями
     *
     * @param dslContext база, куда писать
     * @param clientId   какой клиент этим всем владеет
     * @param adGroups   данные, которые записывать
     */
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId, List<AdGroup> adGroups) {
        adGroups.stream()
                .collect(groupingBy(AdGroup::getClass, toList()))
                .forEach((clazz, list) -> typeSupportMap.get(clazz)
                        .addAdGroupsToDatabaseTables(dslContext, clientId, list));
    }

    /**
     * Обновить группы объявлений с учетом типа
     */
    public void updateAdGroups(DSLContext dslContext, ClientId clientId, Collection<AppliedChanges<AdGroup>> adGroups) {
        adGroups.stream()
                .collect(groupingBy(changes -> changes.getModel().getClass(), toList()))
                .forEach((clazz, list) -> typeSupportMap.get(clazz)
                        .updateAdGroups(list, clientId, dslContext));
    }
}
