package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsContentPromotion;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CONTENT_PROMOTION;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Реализация логики работы с группой типа CONTENT_PROMOTION в репозиториях:
 * сохранение группы в базу, чтение группы из базы
 */
@Component
@ParametersAreNonnullByDefault
public class ContentPromotionAdGroupSupport implements AdGroupTypeSupport<ContentPromotionAdGroup> {

    public static final JooqMapperWithSupplier<ContentPromotionAdGroup> MAPPER_FOR_ADGROUPS_CONTENT_PROMOTION =
            JooqMapperWithSupplierBuilder.builder(ContentPromotionAdGroup::new)
                    .map(property(ContentPromotionAdGroup.ID, ADGROUPS_CONTENT_PROMOTION.PID))
                    .map(convertibleProperty(ContentPromotionAdGroup.CONTENT_PROMOTION_TYPE,
                            ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE,
                            ContentPromotionAdgroupType::fromSource, ContentPromotionAdgroupType::toSource))
                    .build();

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.CONTENT_PROMOTION;
    }

    @Override
    public Class<ContentPromotionAdGroup> getAdGroupClass() {
        return ContentPromotionAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId,
                                            List<ContentPromotionAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addToAdgroupContentPromotion(dslContext, adGroups);
    }

    private void addToAdgroupContentPromotion(DSLContext dslContext, List<ContentPromotionAdGroup> adGroups) {
        adGroups.forEach(g -> checkArgument(g.getContentPromotionType() != null,
                "Can not guess adgroup_content_type for content_promotion adgroup"));
        new InsertHelper<>(dslContext, ADGROUPS_CONTENT_PROMOTION)
                .addAll(MAPPER_FOR_ADGROUPS_CONTENT_PROMOTION, adGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public ContentPromotionAdGroup constructInstanceFromDb(Record record) {
        ContentPromotionAdGroup adGroup = new ContentPromotionAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        MAPPER_FOR_ADGROUPS_CONTENT_PROMOTION.fromDb(record, adGroup);

        checkArgument(adGroup.getContentPromotionType() != null,
                "Can not guess adgroup_content_type for content_promotion adgroup: " +
                        record.get(AdgroupsContentPromotion.ADGROUPS_CONTENT_PROMOTION.PID));
        return adGroup;
    }
}
