package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jooqmapper.OldJooqMapper;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.userssegments.service.UsersSegmentService;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsCpmBannerCriterionType;
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsCpmBanner;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.adgroup.AdGroupWithUsersSegmentsHelper.adGroupsWithUsersSegmentsToMap;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CPM_BANNER;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
public class CpmBannerAdGroupSupport extends CpmPriceSalesAdGroupSupport implements AdGroupTypeSupport<CpmBannerAdGroup> {

    public static final OldJooqMapper<CpmBannerAdGroup> ADGROUP_MAPPER_FOR_CPM_BANNER_FIELDS =
            new OldJooqMapperBuilder<CpmBannerAdGroup>()
                    .map(field(ADGROUPS_CPM_BANNER.PID, CpmBannerAdGroup.ID))
                    .map(convertibleField(ADGROUPS_CPM_BANNER.HAS_IN_BANNER_CREATIVES, CpmBannerAdGroup.FOR_IN_BANNERS)
                            .convertFromDbBy(e -> nvl(e, 0L) > 0)
                            .convertToDbBy(e -> (e == null) ? 0L : (e ? 1L : 0L))
                            .withDatabaseDefault()
                    )
                    .map(convertibleField(ADGROUPS_CPM_BANNER.CRITERION_TYPE,
                            CpmBannerAdGroup.CRITERION_TYPE)
                            .convertToDbBy(CriterionType::toSource)
                            .convertFromDbBy(CriterionType::fromSource)
                            .withDatabaseDefault())
                    .buildWithoutModelSupplier();

    private final UsersSegmentService usersSegmentService;

    @Autowired
    public CpmBannerAdGroupSupport(UsersSegmentService usersSegmentService) {
        this.usersSegmentService = usersSegmentService;
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.CPM_BANNER;
    }

    @Override
    public Class<CpmBannerAdGroup> getAdGroupClass() {
        return CpmBannerAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId, List<CpmBannerAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addToAdGroupsCpmBanner(dslContext, adGroups);
        usersSegmentService.addSegments(dslContext, adGroupsWithUsersSegmentsToMap(adGroups));
        addPriorities(dslContext, adGroups);
    }

    private void addToAdGroupsCpmBanner(DSLContext dslContext, List<CpmBannerAdGroup> cpmBannerAdGroups) {
        cpmBannerAdGroups.forEach(g -> checkArgument(g.getCriterionType() != null));
        new InsertHelper<>(dslContext, ADGROUPS_CPM_BANNER)
                .addAll(ADGROUP_MAPPER_FOR_CPM_BANNER_FIELDS, cpmBannerAdGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public CpmBannerAdGroup constructInstanceFromDb(Record record) {
        AdgroupsCpmBannerCriterionType criterionType = record.get(AdgroupsCpmBanner.ADGROUPS_CPM_BANNER.CRITERION_TYPE);

        checkArgument(criterionType != null, "Can not guess sub type for cpm_banner adgroup: " + record);

        CpmBannerAdGroup adGroup = new CpmBannerAdGroup();
        ADGROUP_MAPPER_FOR_PRICE_SALES_ADGROUP_FIELDS.fromDb(record, adGroup);
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        ADGROUP_MAPPER_FOR_CPM_BANNER_FIELDS.fromDb(adGroup, record);
        return adGroup;
    }
}
