package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jooqmapper.OldJooqMapper;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmIndoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.userssegments.service.UsersSegmentService;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupPageTargetsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Arrays.asList;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.adgroup.AdGroupWithUsersSegmentsHelper.adGroupsWithUsersSegmentsToMap;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PAGE_TARGETS;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Component
@ParametersAreNonnullByDefault
public class CpmIndoorAdGroupSupport implements AdGroupTypeSupport<CpmIndoorAdGroup> {

    public static final OldJooqMapper<CpmIndoorAdGroup> ADGROUP_MAPPER_FOR_CPM_INDOOR_FIELDS =
            new OldJooqMapperBuilder<CpmIndoorAdGroup>()
                    .map(field(ADGROUP_PAGE_TARGETS.PID, CpmIndoorAdGroup.ID)
                            .disableReadingFromDb())
                    .map(convertibleField(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS,
                            CpmIndoorAdGroup.PAGE_BLOCKS)
                            .convertToDbBy(CpmIndoorAdGroupSupport::pageBlocksToDb)
                            .convertFromDbBy(CpmIndoorAdGroupSupport::pageBlocksFromDb))
                    .buildWithoutModelSupplier();

    private final UsersSegmentService usersSegmentService;

    @Autowired
    public CpmIndoorAdGroupSupport(UsersSegmentService usersSegmentService) {
        this.usersSegmentService = usersSegmentService;
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.CPM_INDOOR;
    }

    @Override
    public Class<CpmIndoorAdGroup> getAdGroupClass() {
        return CpmIndoorAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId,
                                            List<CpmIndoorAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addAdGroupsToAdGroupPageTargets(dslContext, adGroups);
        usersSegmentService.addSegments(dslContext, adGroupsWithUsersSegmentsToMap(adGroups));
    }

    private void addAdGroupsToAdGroupPageTargets(DSLContext dslContext, List<CpmIndoorAdGroup> adGroups) {
        new InsertHelper<>(dslContext, ADGROUP_PAGE_TARGETS)
                .addAll(ADGROUP_MAPPER_FOR_CPM_INDOOR_FIELDS, adGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public CpmIndoorAdGroup constructInstanceFromDb(Record record) {
        CpmIndoorAdGroup adGroup = new CpmIndoorAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        ADGROUP_MAPPER_FOR_CPM_INDOOR_FIELDS.fromDb(adGroup, record);
        return adGroup;
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<CpmIndoorAdGroup>> adGroups, ClientId clientId,
                               DSLContext dslContext) {
        JooqUpdateBuilder<AdgroupPageTargetsRecord, CpmIndoorAdGroup> updateBuilder =
                new JooqUpdateBuilder<>(ADGROUP_PAGE_TARGETS.PID, adGroups);

        updateBuilder
                .processProperty(CpmIndoorAdGroup.PAGE_BLOCKS, ADGROUP_PAGE_TARGETS.PAGE_BLOCKS,
                        CpmIndoorAdGroupSupport::pageBlocksToDb);

        dslContext.update(ADGROUP_PAGE_TARGETS)
                .set(updateBuilder.getValues())
                .where(ADGROUP_PAGE_TARGETS.PID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    private static String pageBlocksToDb(@Nullable List<PageBlock> pageBlocks) {
        return RepositoryUtils.nullSafeWrapper(JsonUtils::toJson).apply(pageBlocks);
    }

    private static List<PageBlock> pageBlocksFromDb(@Nullable String jsonPageBlocks) {
        return jsonPageBlocks == null ? null : asList(fromJson(jsonPageBlocks, PageBlock[].class));
    }
}
