package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jooqmapper.OldJooqMapper;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmOutdoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.banner.container.OutdoorModerateBannerPagesUpdateParams;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.service.OutdoorModerateBannerPagesUpdater;
import ru.yandex.direct.core.entity.userssegments.service.UsersSegmentService;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupPageTargetsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.adgroup.AdGroupWithUsersSegmentsHelper.adGroupsWithUsersSegmentsToMap;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PAGE_TARGETS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Component
@ParametersAreNonnullByDefault
public class CpmOutdoorAdGroupSupport implements AdGroupTypeSupport<CpmOutdoorAdGroup> {
    public static final OldJooqMapper<CpmOutdoorAdGroup> ADGROUP_MAPPER_FOR_CPM_OUTDOOR_FIELDS =
            new OldJooqMapperBuilder<CpmOutdoorAdGroup>()
                    .map(field(ADGROUP_PAGE_TARGETS.PID, CpmOutdoorAdGroup.ID)
                            .disableReadingFromDb())
                    .map(convertibleField(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS,
                            CpmOutdoorAdGroup.PAGE_BLOCKS)
                            .convertToDbBy(CpmOutdoorAdGroupSupport::pageBlocksToDb)
                            .convertFromDbBy(CpmOutdoorAdGroupSupport::pageBlocksFromDb))
                    .buildWithoutModelSupplier();

    private final OutdoorModerateBannerPagesUpdater outdoorModerateBannerPagesUpdater;

    private final BannerModerationRepository bannerModerationRepository;

    private final UsersSegmentService usersSegmentService;

    @Autowired
    public CpmOutdoorAdGroupSupport(
            OutdoorModerateBannerPagesUpdater outdoorModerateBannerPagesUpdater,
            BannerModerationRepository bannerModerationRepository,
            UsersSegmentService usersSegmentService) {
        this.outdoorModerateBannerPagesUpdater = outdoorModerateBannerPagesUpdater;
        this.bannerModerationRepository = bannerModerationRepository;
        this.usersSegmentService = usersSegmentService;
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.CPM_OUTDOOR;
    }

    @Override
    public Class<CpmOutdoorAdGroup> getAdGroupClass() {
        return CpmOutdoorAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId,
                                            List<CpmOutdoorAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addAdGroupsToAdGroupPageTargets(dslContext, adGroups);
        usersSegmentService.addSegments(dslContext, adGroupsWithUsersSegmentsToMap(adGroups));
    }

    private void addAdGroupsToAdGroupPageTargets(DSLContext dslContext, List<CpmOutdoorAdGroup> adGroups) {
        new InsertHelper<>(dslContext, ADGROUP_PAGE_TARGETS)
                .addAll(ADGROUP_MAPPER_FOR_CPM_OUTDOOR_FIELDS, adGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public CpmOutdoorAdGroup constructInstanceFromDb(Record record) {
        CpmOutdoorAdGroup adGroup = new CpmOutdoorAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        ADGROUP_MAPPER_FOR_CPM_OUTDOOR_FIELDS.fromDb(adGroup, record);
        return adGroup;
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<CpmOutdoorAdGroup>> adGroups, ClientId clientId,
                               DSLContext dslContext) {
        JooqUpdateBuilder<AdgroupPageTargetsRecord, CpmOutdoorAdGroup> updateBuilder =
                new JooqUpdateBuilder<>(ADGROUP_PAGE_TARGETS.PID, adGroups);

        updateBuilder
                .processProperty(CpmOutdoorAdGroup.PAGE_BLOCKS, ADGROUP_PAGE_TARGETS.PAGE_BLOCKS,
                        CpmOutdoorAdGroupSupport::pageBlocksToDb);

        dslContext.update(ADGROUP_PAGE_TARGETS)
                .set(updateBuilder.getValues())
                .where(ADGROUP_PAGE_TARGETS.PID.in(updateBuilder.getChangedIds()))
                .execute();

        updateModerateBannerPages(adGroups, dslContext);
    }

    /**
     * Обновление данных в moderate_banner_pages.
     * При изменении page'ей в группе необходимо обновить данные moderate_banner_pages
     * (внешняя модерация баннеров по каждому пейджу).
     * <p>
     * Кейс по шагам:
     * 1. Внутренняя модерация одобрила баннер
     * 2. Баннер отправили во внешнюю модерацию по каждому пейджу из группы
     * 3. Пользователь обновил таргетинг по пейджам
     * 4. Потому как баннер уже одобрен внутренней модерацией, то его можно сразу отправить во внешнюю модерацию
     * 5. Добавляем запись в moderate_banner_pages со статусом READY (флаг отправки во внешнюю модерацию)
     * <p>
     * Обновление данных в moderate_banner_pages:
     * <p>
     * 1. Ищем одобренные внутренней модерацией баннеры в каждой группе (banners.statusModerate == YES),
     * все дальнейшие шаги исполняются только для этих баннеров
     * 2. Обновляемые записи в moderate_banner_pages и mod_reasons.
     * 3. Сбрасываем statusBsSynced у затронутых баннеров.
     */
    void updateModerateBannerPages(Collection<AppliedChanges<CpmOutdoorAdGroup>> cpmOutdoorAdGroups,
                                   DSLContext dslContext) {
        Map<Long, AppliedChanges<CpmOutdoorAdGroup>> adGroups = StreamEx.of(cpmOutdoorAdGroups)
                .filter(adGroup -> adGroup.changed(CpmOutdoorAdGroup.PAGE_BLOCKS))
                .toMap(adGroup -> adGroup.getModel().getId(), identity());

        Map<Long, Long> bannerIds = selectApprovedBannersForUpdate(dslContext, adGroups.keySet());
        Map<Long, List<Long>> bannersMinusGeo =
                bannerModerationRepository.getCurrentBannersMinusGeo(dslContext, bannerIds.keySet());
        Map<Long, Long> bannersVersion = bannerModerationRepository.getBannerModerateVersions(dslContext, bannerIds.keySet());

        Map<Long, OutdoorModerateBannerPagesUpdateParams> bannerIdToUpdateParams = EntryStream.of(bannerIds)
                .mapToValue((bannerId, adGroupId) -> {
                    AppliedChanges<CpmOutdoorAdGroup> adGroup = adGroups.get(adGroupId);
                    List<PageBlock> adGroupPageBlocks = nvl(adGroup.getModel().getPageBlocks(), emptyList());
                    List<Long> adGroupGeo = nvl(adGroup.getModel().getGeo(), emptyList());
                    List<Long> bannerMinusGeo = nvl(bannersMinusGeo.get(bannerId), emptyList());
                    Long bannerVersion = bannersVersion.get(bannerId);

                    return OutdoorModerateBannerPagesUpdateParams.builder()
                            .withAdGroupPageBlocks(adGroupPageBlocks)
                            .withAdGroupGeo(adGroupGeo)
                            .withBannerVersion(bannerVersion)
                            .withBannerMinusGeo(bannerMinusGeo)
                            .build();
                })
                .toMap();

        outdoorModerateBannerPagesUpdater.updateModerateBannerPages(bannerIdToUpdateParams, dslContext);
    }

    /**
     * Достаем из бд одобренные внутренней модерацией баннеры в каждой группе.
     * Метод берет лок на BANNERS
     *
     * @param dslContext - контекст транзакции
     * @param adGroupIds - id групп для которых надо получить баннеры
     * @return - мапа bannerId -> adGroupId
     */
    private Map<Long, Long> selectApprovedBannersForUpdate(DSLContext dslContext, Collection<Long> adGroupIds) {
        return dslContext
                .select(BANNERS.BID, BANNERS.PID)
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds)
                        .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes)))
                .forUpdate()
                .fetchMap(BANNERS.BID, BANNERS.PID);
    }

    public static String pageBlocksToDb(@Nullable List<PageBlock> pageBlocks) {
        return RepositoryUtils.nullSafeWrapper(JsonUtils::toJson).apply(pageBlocks);
    }

    public static List<PageBlock> pageBlocksFromDb(@Nullable String jsonPageBlocks) {
        return jsonPageBlocks == null ? null : asList(fromJson(jsonPageBlocks, PageBlock[].class));
    }
}
