package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.userssegments.service.UsersSegmentService;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsCpmBannerCriterionType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.adgroup.AdGroupWithUsersSegmentsHelper.adGroupsWithUsersSegmentsToMap;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CPM_BANNER;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CPM_VIDEO;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
public class CpmVideoAdGroupSupport extends CpmPriceSalesAdGroupSupport implements AdGroupTypeSupport<CpmVideoAdGroup> {

    public static final JooqMapperWithSupplier<CpmVideoAdGroup> MAPPER_FOR_ADGROUPS_CPM_BANNER =
            JooqMapperWithSupplierBuilder.builder(CpmVideoAdGroup::new)
                    .writeField(ADGROUPS_CPM_BANNER.PID, fromProperty(CpmVideoAdGroup.ID))
                    .map(convertibleProperty(
                            CpmVideoAdGroup.CRITERION_TYPE,
                            ADGROUPS_CPM_BANNER.CRITERION_TYPE,
                            dbValue -> nvl(CriterionType.fromSource(dbValue), CriterionType.USER_PROFILE),
                            javaValue -> nvl(CriterionType.toSource(javaValue), AdgroupsCpmBannerCriterionType.user_profile)
                    ))
                    .build();

    public static final JooqMapperWithSupplier<CpmVideoAdGroup> MAPPER_FOR_ADGROUPS_CPM_VIDEO =
            JooqMapperWithSupplierBuilder.builder(CpmVideoAdGroup::new)
                    .writeField(ADGROUPS_CPM_VIDEO.PID, fromProperty(CpmVideoAdGroup.ID))
                    .map(convertibleProperty(
                            CpmVideoAdGroup.IS_NON_SKIPPABLE,
                            ADGROUPS_CPM_VIDEO.IS_NON_SKIPPABLE,
                            aLong -> Boolean.TRUE.equals(RepositoryUtils.booleanFromLong(aLong)),
                            CpmVideoAdGroupSupport::convertBooleanToLong
                    ))
                    .build();

    private final UsersSegmentService usersSegmentService;

    @Autowired
    public CpmVideoAdGroupSupport(UsersSegmentService usersSegmentService) {
        this.usersSegmentService = usersSegmentService;
    }

    private static Long convertBooleanToLong(Boolean aBoolean) {
        return RepositoryUtils.booleanToLong(Boolean.TRUE.equals(aBoolean));
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.CPM_VIDEO;
    }

    @Override
    public Class<CpmVideoAdGroup> getAdGroupClass() {
        return CpmVideoAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId, List<CpmVideoAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addToAdGroupsCpmBanner(dslContext, adGroups);
        addToAdGroupsCpmVideo(dslContext, adGroups);
        usersSegmentService.addSegments(dslContext, adGroupsWithUsersSegmentsToMap(adGroups));
        addPriorities(dslContext, adGroups);
    }

    private void addToAdGroupsCpmBanner(DSLContext dslContext, List<CpmVideoAdGroup> cpmVideoAdGroups) {
        new InsertHelper<>(dslContext, ADGROUPS_CPM_BANNER)
                .addAll(MAPPER_FOR_ADGROUPS_CPM_BANNER, cpmVideoAdGroups)
                .executeIfRecordsAdded();
    }

    private void addToAdGroupsCpmVideo(DSLContext dslContext, List<CpmVideoAdGroup> cpmVideoAdGroups) {
        new InsertHelper<>(dslContext, ADGROUPS_CPM_VIDEO)
                .addAll(MAPPER_FOR_ADGROUPS_CPM_VIDEO, cpmVideoAdGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public CpmVideoAdGroup constructInstanceFromDb(Record record) {
        CpmVideoAdGroup adGroup = new CpmVideoAdGroup();
        ADGROUP_MAPPER_FOR_PRICE_SALES_ADGROUP_FIELDS.fromDb(record, adGroup);
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        MAPPER_FOR_ADGROUPS_CPM_BANNER.fromDb(record, adGroup);
        MAPPER_FOR_ADGROUPS_CPM_VIDEO.fromDb(record, adGroup);
        return adGroup;
    }
}
