package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jooqmapper.OldJooqMapper;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchMapping;
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsDynamic;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsDynamicRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;

@Component
@ParametersAreNonnullByDefault
public class DynamicFeedAdGroupSupport implements AdGroupTypeSupport<DynamicFeedAdGroup> {

    public static final OldJooqMapper<DynamicFeedAdGroup> ADGROUP_MAPPER_FOR_DYNAMIC_FEED_FIELDS =
            new OldJooqMapperBuilder<DynamicFeedAdGroup>()
                    .map(field(ADGROUPS_DYNAMIC.PID, DynamicFeedAdGroup.ID))
                    .map(convertibleField(ADGROUPS_DYNAMIC.RELEVANCE_MATCH_CATEGORIES,
                            DynamicFeedAdGroup.RELEVANCE_MATCH_CATEGORIES)
                            .convertToDbBy(RelevanceMatchMapping::relevanceMatchCategoriesToDbFormat)
                            .convertFromDbBy(RelevanceMatchMapping::relevanceMatchCategoriesFromDbFormat)
                            .withDatabaseDefault())
                    .map(field(ADGROUPS_DYNAMIC.FEED_ID, DynamicFeedAdGroup.FEED_ID))
                    .map(convertibleField(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED,
                            DynamicFeedAdGroup.STATUS_B_L_GENERATED)
                            .convertToDbBy(Common::statusBLGeneratedToDbAdgroupsDynamic)
                            .convertFromDbBy(Common::statusBLGeneratedFromDbAdgroupsDynamic)
                            .withDatabaseDefault())
                    .map(field(ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_NAME, DynamicFeedAdGroup.FIELD_TO_USE_AS_NAME))
                    .map(field(ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_BODY, DynamicFeedAdGroup.FIELD_TO_USE_AS_BODY))
                    .buildWithoutModelSupplier();

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.DYNAMIC;
    }

    @Override
    public Class<DynamicFeedAdGroup> getAdGroupClass() {
        return DynamicFeedAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId,
                                            List<DynamicFeedAdGroup> adGroups) {
        List<DynamicFeedAdGroup> dynamicFeedAdGroups = StreamEx.of(adGroups)
                .select(DynamicFeedAdGroup.class)
                .toList();

        addAdGroupsToCommonTables(dslContext, adGroups);
        addToAdGroupsDynamicFeed(dslContext, dynamicFeedAdGroups);
    }

    private void addToAdGroupsDynamicFeed(DSLContext dslContext, List<DynamicFeedAdGroup> dynamicFeedAdGroup) {
        dynamicFeedAdGroup.forEach(g -> checkArgument(g.getFeedId() != null));
        new InsertHelper<>(dslContext, ADGROUPS_DYNAMIC)
                .addAll(ADGROUP_MAPPER_FOR_DYNAMIC_FEED_FIELDS, dynamicFeedAdGroup)
                .executeIfRecordsAdded();
    }

    @Override
    public DynamicFeedAdGroup constructInstanceFromDb(Record record) {
        Long mainDomainId = record.get(AdgroupsDynamic.ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID);
        Long feedId = record.get(AdgroupsDynamic.ADGROUPS_DYNAMIC.FEED_ID);

        checkArgument(mainDomainId == null && feedId != null || mainDomainId != null && feedId == null,
                "Can not guess sub type for dynamic adgroup: " + record);

        DynamicFeedAdGroup result = new DynamicFeedAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(result, record);
        ADGROUP_MAPPER_FOR_DYNAMIC_FEED_FIELDS.fromDb(result, record);

        return result;
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<DynamicFeedAdGroup>> adGroups, ClientId clientId,
                               DSLContext dslContext) {
        updateTextAdGroups(adGroups, dslContext);
    }

    private void updateTextAdGroups(Collection<AppliedChanges<DynamicFeedAdGroup>> groups, DSLContext dslContext) {
        JooqUpdateBuilder<AdgroupsDynamicRecord, DynamicFeedAdGroup> updateBuilder =
                new JooqUpdateBuilder<>(ADGROUPS_DYNAMIC.PID, groups);

        updateBuilder.processProperty(
                DynamicFeedAdGroup.RELEVANCE_MATCH_CATEGORIES,
                ADGROUPS_DYNAMIC.RELEVANCE_MATCH_CATEGORIES,
                RelevanceMatchMapping::relevanceMatchCategoriesToDbFormat);

        updateBuilder.processProperty(DynamicFeedAdGroup.FEED_ID, ADGROUPS_DYNAMIC.FEED_ID);
        updateBuilder.processProperty(
                DynamicFeedAdGroup.STATUS_B_L_GENERATED,
                ADGROUPS_DYNAMIC.STATUS_BL_GENERATED,
                Common::statusBLGeneratedToDbAdgroupsDynamic);
        updateBuilder
                .processProperty(DynamicFeedAdGroup.FIELD_TO_USE_AS_NAME, ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_NAME);
        updateBuilder
                .processProperty(DynamicFeedAdGroup.FIELD_TO_USE_AS_BODY, ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_BODY);

        dslContext.update(ADGROUPS_DYNAMIC)
                .set(updateBuilder.getValues())
                .where(ADGROUPS_DYNAMIC.PID.in(updateBuilder.getChangedIds()))
                .execute();
    }
}
