package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jooqmapper.OldJooqMapper;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchMapping;
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsDynamic;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsDynamicRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.DOMAINS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class DynamicTextAdGroupSupport implements AdGroupTypeSupport<DynamicTextAdGroup> {
    public static final OldJooqMapper<DynamicTextAdGroup> ADGROUP_MAPPER_FOR_DYNAMIC_DOMAIN_FIELDS =
            new OldJooqMapperBuilder<DynamicTextAdGroup>()
                    .map(field(ADGROUPS_DYNAMIC.PID, DynamicTextAdGroup.ID))
                    .map(convertibleField(ADGROUPS_DYNAMIC.RELEVANCE_MATCH_CATEGORIES,
                            DynamicTextAdGroup.RELEVANCE_MATCH_CATEGORIES)
                            .convertToDbBy(RelevanceMatchMapping::relevanceMatchCategoriesToDbFormat)
                            .convertFromDbBy(RelevanceMatchMapping::relevanceMatchCategoriesFromDbFormat)
                            .withDatabaseDefault())
                    .map(field(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID, DynamicTextAdGroup.MAIN_DOMAIN_ID))
                    .map(convertibleField(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED,
                            DynamicTextAdGroup.STATUS_B_L_GENERATED)
                            .convertToDbBy(Common::statusBLGeneratedToDbAdgroupsDynamic)
                            .convertFromDbBy(Common::statusBLGeneratedFromDbAdgroupsDynamic)
                            .withDatabaseDefault())
                    .map(field(ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_NAME, DynamicTextAdGroup.FIELD_TO_USE_AS_NAME))
                    .map(field(ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_BODY, DynamicTextAdGroup.FIELD_TO_USE_AS_BODY))
                    .map(field(DOMAINS.DOMAIN, DynamicTextAdGroup.DOMAIN_URL))
                    .buildWithoutModelSupplier();

    private final DomainService domainService;

    @Autowired
    public DynamicTextAdGroupSupport(DomainService domainService) {
        this.domainService = domainService;
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.DYNAMIC;
    }

    @Override
    public Class<DynamicTextAdGroup> getAdGroupClass() {
        return DynamicTextAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId,
                                            List<DynamicTextAdGroup> adGroups) {
        List<DynamicTextAdGroup> dynamicTextAdGroups = StreamEx.of(adGroups)
                .select(DynamicTextAdGroup.class)
                .toList();

        addAdGroupsToCommonTables(dslContext, adGroups);
        addToDomainsAndFillMainDomainId(dslContext, dynamicTextAdGroups);
        addToAdGroupsDynamicDomain(dslContext, dynamicTextAdGroups);
    }

    private void addToDomainsAndFillMainDomainId(DSLContext dslContext,
                                                 List<DynamicTextAdGroup> typedDynamicTextAdGroups) {
        List<String> domainUrls = mapList(typedDynamicTextAdGroups, DynamicTextAdGroup::getDomainUrl);
        List<Long> domainIds = domainService.getOrCreate(dslContext, domainUrls);
        EntryStream.zip(typedDynamicTextAdGroups, domainIds)
                .forKeyValue(DynamicTextAdGroup::setMainDomainId);
    }

    private void addToAdGroupsDynamicDomain(DSLContext dslContext, List<DynamicTextAdGroup> dynamicTextAdGroups) {
        dynamicTextAdGroups.forEach(g -> checkArgument(g.getMainDomainId() != null));
        new InsertHelper<>(dslContext, ADGROUPS_DYNAMIC)
                .addAll(ADGROUP_MAPPER_FOR_DYNAMIC_DOMAIN_FIELDS, dynamicTextAdGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public DynamicTextAdGroup constructInstanceFromDb(Record record) {
        Long mainDomainId = record.get(AdgroupsDynamic.ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID);
        Long feedId = record.get(AdgroupsDynamic.ADGROUPS_DYNAMIC.FEED_ID);

        checkArgument(mainDomainId == null && feedId != null || mainDomainId != null && feedId == null,
                "Can not guess sub type for dynamic adgroup: " + record);

        DynamicTextAdGroup result = new DynamicTextAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(result, record);
        ADGROUP_MAPPER_FOR_DYNAMIC_DOMAIN_FIELDS.fromDb(result, record);

        return result;
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<DynamicTextAdGroup>> adGroups, ClientId clientId,
                               DSLContext dslContext) {
        updateTextAdGroups(adGroups, dslContext);
    }

    private void updateTextAdGroups(
            Collection<AppliedChanges<DynamicTextAdGroup>> dynamicTextAdGroups, DSLContext dslContext) {
        JooqUpdateBuilder<AdgroupsDynamicRecord, DynamicTextAdGroup> updateBuilder =
                new JooqUpdateBuilder<>(ADGROUPS_DYNAMIC.PID, dynamicTextAdGroups);

        updateBuilder.processProperty(
                DynamicTextAdGroup.RELEVANCE_MATCH_CATEGORIES,
                ADGROUPS_DYNAMIC.RELEVANCE_MATCH_CATEGORIES,
                RelevanceMatchMapping::relevanceMatchCategoriesToDbFormat);

        updateBuilder.processProperty(DynamicTextAdGroup.MAIN_DOMAIN_ID, ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID);
        updateBuilder.processProperty(
                DynamicTextAdGroup.STATUS_B_L_GENERATED,
                ADGROUPS_DYNAMIC.STATUS_BL_GENERATED,
                Common::statusBLGeneratedToDbAdgroupsDynamic);
        updateBuilder
                .processProperty(DynamicTextAdGroup.FIELD_TO_USE_AS_NAME, ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_NAME);
        updateBuilder
                .processProperty(DynamicTextAdGroup.FIELD_TO_USE_AS_BODY, ADGROUPS_DYNAMIC.FIELD_TO_USE_AS_BODY);

        dslContext.update(ADGROUPS_DYNAMIC)
                .set(updateBuilder.getValues())
                .where(ADGROUPS_DYNAMIC.PID.in(updateBuilder.getChangedIds()))
                .execute();
    }
}
