package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsInternalRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_INTERNAL;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
public class InternalAdGroupSupport implements AdGroupTypeSupport<InternalAdGroup> {

    public static final Long DEFAULT_LEVEL = 1L;

    public static final JooqMapper<InternalAdGroup> ADGROUP_MAPPER_FOR_INTERNAL_FIELDS =
            JooqMapperBuilder.<InternalAdGroup>builder()
                    .map(property(InternalAdGroup.ID, ADGROUPS_INTERNAL.PID))
                    .map(convertibleProperty(InternalAdGroup.LEVEL, ADGROUPS_INTERNAL.LEVEL,
                            InternalAdGroupSupport::nullToDefault, InternalAdGroupSupport::nullToDefault))
                    .map(convertibleProperty(InternalAdGroup.RF, ADGROUPS_INTERNAL.RF,
                            InternalAdGroupSupport::zeroLongToNullOrInteger,
                            InternalAdGroupSupport::nullToZeroLong))
                    .map(convertibleProperty(InternalAdGroup.RF_RESET, ADGROUPS_INTERNAL.RF_RESET,
                            InternalAdGroupSupport::zeroLongToNullOrInteger,
                            InternalAdGroupSupport::nullToZeroLong))
                    .map(property(InternalAdGroup.START_TIME, ADGROUPS_INTERNAL.START_TIME))
                    .map(property(InternalAdGroup.FINISH_TIME, ADGROUPS_INTERNAL.FINISH_TIME))
                    .map(integerProperty(InternalAdGroup.MAX_CLICKS_COUNT, ADGROUPS_INTERNAL.MAX_CLICKS_COUNT))
                    .map(integerProperty(InternalAdGroup.MAX_CLICKS_PERIOD, ADGROUPS_INTERNAL.MAX_CLICKS_PERIOD))
                    .map(integerProperty(InternalAdGroup.MAX_STOPS_COUNT, ADGROUPS_INTERNAL.MAX_STOPS_COUNT))
                    .map(integerProperty(InternalAdGroup.MAX_STOPS_PERIOD, ADGROUPS_INTERNAL.MAX_STOPS_PERIOD))
                    .build();

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.INTERNAL;
    }

    @Override
    public Class<InternalAdGroup> getAdGroupClass() {
        return InternalAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId, List<InternalAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addAdGroupsInternal(dslContext, adGroups);
    }

    private void addAdGroupsInternal(DSLContext dslContext, List<InternalAdGroup> adGroups) {
        new InsertHelper<>(dslContext, ADGROUPS_INTERNAL)
                .addAll(ADGROUP_MAPPER_FOR_INTERNAL_FIELDS, adGroups)
                .executeIfRecordsAdded();
    }

    @Override
    public InternalAdGroup constructInstanceFromDb(Record record) {
        InternalAdGroup adGroup = new InternalAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        ADGROUP_MAPPER_FOR_INTERNAL_FIELDS.fromDb(record, adGroup);
        return adGroup;
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<InternalAdGroup>> adGroups, ClientId clientId,
                               DSLContext dslContext) {
        updateInternalAdGroups(adGroups, dslContext);
    }

    private void updateInternalAdGroups(Collection<AppliedChanges<InternalAdGroup>> internalAdGroups,
                                        DSLContext dslContext) {
        JooqUpdateBuilder<AdgroupsInternalRecord, InternalAdGroup> updateBuilder =
                new JooqUpdateBuilder<>(ADGROUPS_INTERNAL.PID, internalAdGroups);

        updateBuilder.processProperty(InternalAdGroup.LEVEL, ADGROUPS_INTERNAL.LEVEL,
                InternalAdGroupSupport::nullToDefault);
        updateBuilder.processProperty(InternalAdGroup.RF, ADGROUPS_INTERNAL.RF, InternalAdGroupSupport::nullToZeroLong);
        updateBuilder.processProperty(InternalAdGroup.RF_RESET, ADGROUPS_INTERNAL.RF_RESET,
                InternalAdGroupSupport::nullToZeroLong);
        updateBuilder.processProperty(InternalAdGroup.START_TIME, ADGROUPS_INTERNAL.START_TIME);
        updateBuilder.processProperty(InternalAdGroup.FINISH_TIME, ADGROUPS_INTERNAL.FINISH_TIME);
        updateBuilder.processProperty(InternalAdGroup.MAX_CLICKS_COUNT, ADGROUPS_INTERNAL.MAX_CLICKS_COUNT,
                RepositoryUtils::intToLong);
        updateBuilder.processProperty(InternalAdGroup.MAX_CLICKS_PERIOD, ADGROUPS_INTERNAL.MAX_CLICKS_PERIOD,
                RepositoryUtils::intToLong);
        updateBuilder.processProperty(InternalAdGroup.MAX_STOPS_COUNT, ADGROUPS_INTERNAL.MAX_STOPS_COUNT,
                RepositoryUtils::intToLong);
        updateBuilder.processProperty(InternalAdGroup.MAX_STOPS_PERIOD, ADGROUPS_INTERNAL.MAX_STOPS_PERIOD,
                RepositoryUtils::intToLong);

        dslContext.update(ADGROUPS_INTERNAL)
                .set(updateBuilder.getValues())
                .where(ADGROUPS_INTERNAL.PID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    private static Long nullToZeroLong(@Nullable Integer num) {
        return num == null ? 0 : RepositoryUtils.intToLong(num);
    }

    @Nullable
    private static Integer zeroLongToNullOrInteger(@Nonnull Long num) {
        return num == 0L ? null : RepositoryUtils.intFromLong(num);
    }

    private static Long nullToDefault(@Nullable Long val) {
        return nvl(val, DEFAULT_LEVEL);
    }

}
