package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jooqmapper.OldJooqMapper;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAutomaticCreationRequestWithSuppliedMobileContent;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppAutomaticCreationService;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsMobileContentRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository.MAPPER_FOR_MOBILE_CONTENT_FIELDS;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsMobileContent.ADGROUPS_MOBILE_CONTENT;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class MobileContentAdGroupSupport implements AdGroupTypeSupport<MobileContentAdGroup> {
    private static final Function<Set<MobileContentAdGroupDeviceTypeTargeting>, String>
            DEVICE_TYPE_TARGETING_CONVERTER = targeting -> targeting.stream()
            .map(MobileContentAdGroupDeviceTypeTargeting::getTypedValue)
            .collect(joining(","));
    private static final Function<Set<MobileContentAdGroupNetworkTargeting>, String> NETWORK_TARGETING_CONVERTER =
            targeting -> targeting.stream()
                    .map(MobileContentAdGroupNetworkTargeting::getTypedValue)
                    .collect(joining(","));
    public static final OldJooqMapper<MobileContentAdGroup> ADGROUP_MAPPER_FOR_MOBILE_CONTENT_FIELDS =
            new OldJooqMapperBuilder<>(MobileContentAdGroup::new)
                    .map(field(ADGROUPS_MOBILE_CONTENT.PID, AdGroup.ID))
                    .map(field(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID, MobileContentAdGroup.MOBILE_CONTENT_ID))
                    .map(field(ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF, MobileContentAdGroup.STORE_URL))
                    .map(field(ADGROUPS_MOBILE_CONTENT.MIN_OS_VERSION,
                            MobileContentAdGroup.MINIMAL_OPERATING_SYSTEM_VERSION))
                    .map(convertibleField(ADGROUPS_MOBILE_CONTENT.DEVICE_TYPE_TARGETING,
                            MobileContentAdGroup.DEVICE_TYPE_TARGETING)
                            .convertToDbBy(DEVICE_TYPE_TARGETING_CONVERTER)
                            .convertFromDbBy(targeting -> Arrays.stream(targeting.split(","))
                                    .map(MobileContentAdGroupDeviceTypeTargeting::fromTypedValue)
                                    .collect(toSet()))
                            .withDatabaseDefault())
                    .map(convertibleField(ADGROUPS_MOBILE_CONTENT.NETWORK_TARGETING,
                            MobileContentAdGroup.NETWORK_TARGETING)
                            .convertToDbBy(NETWORK_TARGETING_CONVERTER)
                            .convertFromDbBy(targeting -> Arrays.stream(targeting.split(","))
                                    .map(MobileContentAdGroupNetworkTargeting::fromTypedValue)
                                    .collect(toSet()))
                            .withDatabaseDefault())
                    .build();

    private final MobileContentRepository mobileContentRepository;
    private final CampaignRepository campaignRepository;
    private final MobileAppAutomaticCreationService mobileAppAutomaticCreationService;

    public MobileContentAdGroupSupport(MobileContentRepository mobileContentRepository,
                                       CampaignRepository campaignRepository,
                                       MobileAppAutomaticCreationService mobileAppAutomaticCreationService) {
        this.mobileContentRepository = mobileContentRepository;
        this.campaignRepository = campaignRepository;
        this.mobileAppAutomaticCreationService = mobileAppAutomaticCreationService;
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.MOBILE_CONTENT;
    }

    @Override
    public Class<MobileContentAdGroup> getAdGroupClass() {
        return MobileContentAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId,
                                            List<MobileContentAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addToMobileContent(dslContext, clientId, adGroups);

        ensureMobileAppCreatedAndAttachedToCampaign(dslContext, clientId, adGroups);

        InsertHelper<AdgroupsMobileContentRecord> insertHelper =
                new InsertHelper<>(dslContext, ADGROUPS_MOBILE_CONTENT);

        insertHelper.addAll(ADGROUP_MAPPER_FOR_MOBILE_CONTENT_FIELDS, adGroups);
        insertHelper.executeIfRecordsAdded();
    }

    private void ensureMobileAppCreatedAndAttachedToCampaign(DSLContext dslContext, ClientId clientId,
                                                             List<MobileContentAdGroup> adGroups) {
        Set<Long> campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        Set<Long> campaignsWithMobileApps =
                campaignRepository.getCampaignMobileAppIds(dslContext, campaignIds).keySet();
        Set<Long> campaignsWithoutMobileApp = Sets.difference(campaignIds, campaignsWithMobileApps);

        Map<Long, MobileContentAdGroup> campaignToSomeAdGroup = StreamEx.of(adGroups)
                .mapToEntry(AdGroup::getCampaignId, Function.identity())
                .filterKeys(campaignsWithoutMobileApp::contains)
                .toMap((a, b) -> a);

        Map<Long, MobileContent> mobileContentById =
                mobileContentRepository.getMobileContent(dslContext, listToSet(campaignToSomeAdGroup.values(),
                        MobileContentAdGroup::getMobileContentId))
                        .stream()
                        .collect(toMap(MobileContent::getId, Function.identity()));
        Map<Long, MobileAppAutomaticCreationRequestWithSuppliedMobileContent> campaignToReq =
                EntryStream.of(campaignToSomeAdGroup)
                        .mapValues(g -> {
                            Long mobileContentId = g.getMobileContentId();
                            checkState(mobileContentById.containsKey(mobileContentId));
                            MobileContent mobileContent = mobileContentById.get(mobileContentId);
                            return new MobileAppAutomaticCreationRequestWithSuppliedMobileContent(mobileContent,
                                    g.getStoreUrl(), g.getMinimalOperatingSystemVersion(), null, null);
                        })
                        .toMap();

        Map<MobileAppAutomaticCreationRequestWithSuppliedMobileContent, Long> results =
                mobileAppAutomaticCreationService.getOrCreateMobileAppsFromMobileContentModels(
                        clientId, campaignToReq.values());

        Map<Long, Long> campaignToMobileApp = EntryStream.of(campaignToReq)
                .mapValues(results::get)
                .toMap();
        campaignRepository.setMobileAppIds(dslContext, campaignToMobileApp);
    }

    @SuppressWarnings("Duplicates")
    private void addToMobileContent(DSLContext dslContext, ClientId clientId,
                                    List<MobileContentAdGroup> mobileAdGroups) {
        // урлы в storeUrl должны быть предварительно провалидированы и тут мы ожидаем, что все они парсятся
        List<MobileAppStoreUrl> parsedUrls = StreamEx.of(mobileAdGroups)
                .map(MobileContentAdGroup::getStoreUrl)
                .map(MobileAppStoreUrlParser::parseStrict)
                .toList();

        List<Long> mobileContentIds = mobileContentRepository.getOrCreate(dslContext, clientId, parsedUrls);
        EntryStream.zip(mobileAdGroups, mobileContentIds)
                .forKeyValue(MobileContentAdGroup::setMobileContentId);
    }

    @Override
    public MobileContentAdGroup constructInstanceFromDb(Record record) {
        MobileContentAdGroup adGroup = new MobileContentAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        ADGROUP_MAPPER_FOR_MOBILE_CONTENT_FIELDS.fromDb(adGroup, record);

        MobileContent mobileContent = MAPPER_FOR_MOBILE_CONTENT_FIELDS.fromDb(record);
        adGroup.setMobileContent(mobileContent);

        return adGroup;
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<MobileContentAdGroup>> adGroups,
                               ClientId clientId, DSLContext dslContext) {
        JooqUpdateBuilder<AdgroupsMobileContentRecord, MobileContentAdGroup> updateBuilder =
                new JooqUpdateBuilder<>(ADGROUPS_MOBILE_CONTENT.PID, adGroups);

        updateBuilder.processProperty(
                MobileContentAdGroup.DEVICE_TYPE_TARGETING,
                ADGROUPS_MOBILE_CONTENT.DEVICE_TYPE_TARGETING,
                DEVICE_TYPE_TARGETING_CONVERTER);
        updateBuilder.processProperty(
                MobileContentAdGroup.NETWORK_TARGETING,
                ADGROUPS_MOBILE_CONTENT.NETWORK_TARGETING,
                NETWORK_TARGETING_CONVERTER);
        updateBuilder.processProperty(
                MobileContentAdGroup.MINIMAL_OPERATING_SYSTEM_VERSION,
                ADGROUPS_MOBILE_CONTENT.MIN_OS_VERSION);

        dslContext.update(ADGROUPS_MOBILE_CONTENT)
                .set(updateBuilder.getValues())
                .where(ADGROUPS_MOBILE_CONTENT.PID.in(updateBuilder.getChangedIds()))
                .execute();
    }
}
