package ru.yandex.direct.core.entity.adgroup.repository.typesupport;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.feedfilter.converter.FeedFilterConverters;
import ru.yandex.direct.core.entity.feedfilter.service.FeedFilterConverter;
import ru.yandex.direct.core.entity.userssegments.service.UsersSegmentService;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsTextStatusblgenerated;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsTextRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.adgroup.AdGroupWithUsersSegmentsHelper.adGroupsWithUsersSegmentsToMap;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.ADGROUP_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.Common.addAdGroupsToCommonTables;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_TEXT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class TextAdGroupSupport implements AdGroupTypeSupport<TextAdGroup> {

    private static final FeedFilterConverter FEED_FILTER_CONVERTER = FeedFilterConverters.DB_CONVERTER;

    public static final JooqMapperWithSupplier<TextAdGroup> MAPPER_FOR_TEXT_ADGROUPS_WITH_FEED =
            JooqMapperWithSupplierBuilder.builder(TextAdGroup::new)
                    .writeField(ADGROUPS_TEXT.PID, fromProperty(TextAdGroup.ID))
                    .map(property(TextAdGroup.FIELD_TO_USE_AS_NAME, ADGROUPS_TEXT.FIELD_TO_USE_AS_NAME))
                    .map(property(TextAdGroup.FIELD_TO_USE_AS_BODY, ADGROUPS_TEXT.FIELD_TO_USE_AS_BODY))
                    .map(convertibleProperty(TextAdGroup.STATUS_B_L_GENERATED, ADGROUPS_TEXT.STATUS_BL_GENERATED,
                            Common::statusBLGeneratedFromDbAdgroupsText,
                            javaValue -> ifNotNullOrDefault(javaValue,
                                    Common::statusBLGeneratedToDbAdgroupsText, AdgroupsTextStatusblgenerated.No)))
                    .map(property(TextAdGroup.FEED_ID, ADGROUPS_TEXT.FEED_ID))
                    .map(convertibleProperty(TextAdGroup.FEED_FILTER, ADGROUPS_TEXT.FILTER_DATA,
                            dbValue -> ifNotNull(dbValue, FEED_FILTER_CONVERTER::deserializeFeedFilter),
                            javaValue -> ifNotNull(javaValue, FEED_FILTER_CONVERTER::serializeFeedFilter)))
                    .build();

    private final UsersSegmentService usersSegmentService;

    public TextAdGroupSupport(UsersSegmentService usersSegmentService) {
        this.usersSegmentService = usersSegmentService;
    }

    @Override
    public AdGroupType adGroupType() {
        return AdGroupType.BASE;
    }

    @Override
    public Class<TextAdGroup> getAdGroupClass() {
        return TextAdGroup.class;
    }

    @Override
    public void addAdGroupsToDatabaseTables(DSLContext dslContext, ClientId clientId, List<TextAdGroup> adGroups) {
        addAdGroupsToCommonTables(dslContext, adGroups);
        addToAdGroupsText(dslContext, adGroups);
        usersSegmentService.addSegments(dslContext, adGroupsWithUsersSegmentsToMap(adGroups));
    }

    private void addToAdGroupsText(DSLContext dslContext, List<TextAdGroup> adGroups) {
        var insertHelper = new InsertHelper<>(dslContext, ADGROUPS_TEXT);
        var groupsToAdd = filterList(adGroups, TextAdGroupSupport::needsAdGroupsTextRow);
        insertHelper.addAll(MAPPER_FOR_TEXT_ADGROUPS_WITH_FEED, groupsToAdd);

        insertHelper.executeIfRecordsAdded();
    }

    @Override
    public void updateAdGroups(Collection<AppliedChanges<TextAdGroup>> adGroups, ClientId clientId,
                               DSLContext dslContext) {
        updateAdGroupsText(adGroups, dslContext);
        updateUsersSegments(adGroups, dslContext);
    }

    private void updateAdGroupsText(Collection<AppliedChanges<TextAdGroup>> adGroups, DSLContext dslContext) {
        var toUpdate = new ArrayList<AppliedChanges<TextAdGroup>>();
        var toAdd = new ArrayList<TextAdGroup>();
        var idsToDelete = new ArrayList<Long>();

        // Определяем, что делать с записью в adgroups_text:
        for (var ac : adGroups) {
            var hadRowBefore = hadAdGroupsTextRow(ac);
            var needsRowAfter = needsAdGroupsTextRow(ac.getModel());
            if (hadRowBefore) {
                if (needsRowAfter) {
                    toUpdate.add(ac);
                } else {
                    idsToDelete.add(ac.getModel().getId());
                }
            } else if (needsRowAfter) {
                toAdd.add(ac.getModel());
            }
        }

        if (!toAdd.isEmpty()) {
            addToAdGroupsText(dslContext, toAdd);
        }

        if (!toUpdate.isEmpty()) {
            JooqUpdateBuilder<AdgroupsTextRecord, TextAdGroup> updateBuilder =
                    new JooqUpdateBuilder<>(ADGROUPS_TEXT.PID, toUpdate);

            updateBuilder.processProperty(TextAdGroup.FIELD_TO_USE_AS_NAME, ADGROUPS_TEXT.FIELD_TO_USE_AS_NAME);
            updateBuilder.processProperty(TextAdGroup.FIELD_TO_USE_AS_BODY, ADGROUPS_TEXT.FIELD_TO_USE_AS_BODY);
            updateBuilder.processProperty(TextAdGroup.STATUS_B_L_GENERATED, ADGROUPS_TEXT.STATUS_BL_GENERATED,
                    javaValue -> ifNotNullOrDefault(javaValue,
                            Common::statusBLGeneratedToDbAdgroupsText, AdgroupsTextStatusblgenerated.No));
            updateBuilder.processProperty(TextAdGroup.FEED_ID, ADGROUPS_TEXT.FEED_ID);
            updateBuilder.processProperty(TextAdGroup.FEED_FILTER, ADGROUPS_TEXT.FILTER_DATA,
                    javaValue -> ifNotNull(javaValue, FEED_FILTER_CONVERTER::serializeFeedFilter));

            dslContext.update(ADGROUPS_TEXT)
                    .set(updateBuilder.getValues())
                    .where(ADGROUPS_TEXT.PID.in(updateBuilder.getChangedIds()))
                    .execute();
        }

        if (!idsToDelete.isEmpty()) {
            dslContext.deleteFrom(ADGROUPS_TEXT)
                    .where(ADGROUPS_TEXT.PID.in(idsToDelete))
                    .execute();
        }
    }

    private void updateUsersSegments(Collection<AppliedChanges<TextAdGroup>> adGroups, DSLContext dslContext) {
        List<TextAdGroup> adGroupModels = mapList(adGroups, AppliedChanges::getModel);
        if (!adGroupModels.isEmpty()) {
            Long campaignId = adGroupModels.get(0).getCampaignId();
            usersSegmentService.updateAdGroupsSegments(dslContext, campaignId,
                    adGroupsWithUsersSegmentsToMap(adGroupModels));
        }
    }

    private static boolean needsAdGroupsTextRow(TextAdGroup adGroup) {
        return adGroup.getFeedId() != null;
    }

    private static boolean hadAdGroupsTextRow(AppliedChanges<TextAdGroup> appliedChanges) {
        return appliedChanges.getOldValue(TextAdGroup.FEED_ID) != null;
    }

    @Override
    public TextAdGroup constructInstanceFromDb(Record record) {
        TextAdGroup adGroup = new TextAdGroup();
        ADGROUP_MAPPER_FOR_COMMON_FIELDS.fromDb(adGroup, record);
        MAPPER_FOR_TEXT_ADGROUPS_WITH_FEED.fromDb(record, adGroup);
        return adGroup;
    }
}
