package ru.yandex.direct.core.entity.adgroup.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupBsTags;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.repository.internal.AdGroupBsTagsRepository;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.rbac.RbacService;

@Service
@ParametersAreNonnullByDefault
public class AdGroupBsTagsService {
    private final AdGroupRepository adGroupRepository;
    private final AdGroupBsTagsRepository adGroupBsTagsRepository;
    private final ShardHelper shardHelper;
    private final RbacService rbacService;

    @Autowired
    public AdGroupBsTagsService(
            AdGroupRepository adGroupRepository,
            AdGroupBsTagsRepository adGroupBsTagsRepository,
            ShardHelper shardHelper,
            RbacService rbacService) {
        this.adGroupRepository = adGroupRepository;
        this.adGroupBsTagsRepository = adGroupBsTagsRepository;
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
    }

    /**
     * Проставляем теги всем группам переданных кампаний
     * Если у группы уже есть теги, перезаписываем
     *
     * @param operatorUid       - uid оператора
     * @param tagsByCampaignIds - мапа идентификаторов кампаний в теги
     */
    public void setAdGroupBsTagsByCampaignIds(long operatorUid, Map<Long, AdGroupBsTags> tagsByCampaignIds) {
        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, tagsByCampaignIds.keySet());
        shardHelper.groupByShard(writableCampaignIds, ShardKey.CID).forEach(
                (shard, cids) -> {
                    Map<Long, List<Long>> adGroupIdsByCids = adGroupRepository.getAdGroupIdsByCampaignIds(shard, cids);
                    Map<Long, AdGroupBsTags> tagsByAdGroupIds = EntryStream.of(adGroupIdsByCids)
                            .flatMapValues(StreamEx::of)
                            .invert()
                            .mapValues(tagsByCampaignIds::get)
                            .toMap();
                    adGroupBsTagsRepository.addAdGroupPageTagsForce(shard, tagsByAdGroupIds);
                    adGroupRepository.updateStatusBsSynced(shard, tagsByAdGroupIds.keySet(), StatusBsSynced.NO);
                }
        );
    }

    public void setRawTagsByAdgroupIds(int shard, @Nullable String pageGroupTag, @Nullable String targetTag,
                                       Collection<Long> adgroupIds,
                                       boolean update) {
        adGroupBsTagsRepository.setRawTagsByAdgroupIds(shard, pageGroupTag, targetTag, adgroupIds, update);
        adGroupRepository.updateStatusBsSynced(shard, adgroupIds, StatusBsSynced.NO);
    }

    @QueryWithForbiddenShardMapping("pid")
    public void setRawTagsByAdgroupIds(Set<String> pageGroupTags, Set<String> targetTags,
                                       Collection<Long> adGroupIds) {
        shardHelper.groupByShard(adGroupIds, ShardKey.PID)
                .chunkedBy(1000)
                .forEach((shard, pids) -> {
                    adGroupBsTagsRepository.setRawTagsByAdgroupIds(shard, pageGroupTags, targetTags, pids);
                    adGroupRepository.updateStatusBsSynced(shard, pids, StatusBsSynced.NO);
                });
    }

    @QueryWithForbiddenShardMapping("pid")
    public Map<Long, List<String>> getTargetTagsByAdgroupIds(Collection<Long> adGroupIds) {
        Map<Long, List<String>> tagsByAdGroupId = new HashMap<>();
        shardHelper.groupByShard(adGroupIds, ShardKey.PID)
                .chunkedBy(1000)
                .forEach((shard, pids) ->
                        tagsByAdGroupId.putAll(adGroupBsTagsRepository.getTargetTagsByAdgroupIds(shard, pids))
                );
        return tagsByAdGroupId;
    }

    public void removeRawTagsByAdgroupIds(int shard, @Nullable String pageGroupTag, @Nullable String targetTag,
                                          Collection<Long> adgroupIds) {
        adGroupBsTagsRepository.removeTagsByAdgroupIds(shard, pageGroupTag, targetTag, adgroupIds);
        adGroupRepository.updateStatusBsSynced(shard, adgroupIds, StatusBsSynced.NO);
    }

}
