package ru.yandex.direct.core.entity.adgroup.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupNewMinusKeywords;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupUpdateOperationParams;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithTypeAndGeo;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.adgroup.model.ProductRestrictionKey;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupValidationService;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.core.entity.product.model.ProductRestriction;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUtils;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerAdGroupIdFilter;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
@ParametersAreNonnullByDefault
public class AdGroupService implements EntityService<AdGroup, Long> {
    private static final Logger logger = LoggerFactory.getLogger(AdGroupService.class);

    private final CampaignRepository campaignRepository;
    private final DomainRepository domainRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final BannerModerationRepository bannerModerationRepository;
    private final ShardHelper shardHelper;
    private final ShardSupport shardSupport;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final UpdateAdGroupMinusKeywordsOperationFactory appendMinusKeywordsOperationFactory;
    private final AdGroupsAddOperationFactory addOperationFactory;
    private final AdGroupsUpdateOperationFactory updateOperationFactory;
    private final AdGroupValidationService adGroupValidationService;
    private final KeywordRepository keywordRepository;
    private final ModerationService moderationService;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final ProductService productService;
    private final GeoTreeFactory geoTreeFactory;

    @Autowired
    public AdGroupService(
            CampaignRepository campaignRepository,
            DomainRepository domainRepository,
            AdGroupRepository adGroupRepository,
            BannerTypedRepository bannerTypedRepository,
            BannerRelationsRepository bannerRelationsRepository,
            BannerModerationRepository bannerModerationRepository,
            ShardHelper shardHelper,
            ShardSupport shardSupport,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            UpdateAdGroupMinusKeywordsOperationFactory appendMinusKeywordsOperationFactory,
            AdGroupsAddOperationFactory addOperationFactory,
            AdGroupsUpdateOperationFactory updateOperationFactory,
            AdGroupValidationService adGroupValidationService,
            KeywordRepository keywordRepository,
            ModerationService moderationService,
            ClientGeoService clientGeoService,
            ClientService clientService, ProductService productService,
            GeoTreeFactory geoTreeFactory) {
        this.campaignRepository = campaignRepository;
        this.domainRepository = domainRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bannerModerationRepository = bannerModerationRepository;
        this.shardHelper = shardHelper;
        this.shardSupport = shardSupport;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.appendMinusKeywordsOperationFactory = appendMinusKeywordsOperationFactory;
        this.addOperationFactory = addOperationFactory;
        this.updateOperationFactory = updateOperationFactory;
        this.adGroupValidationService = adGroupValidationService;
        this.keywordRepository = keywordRepository;
        this.moderationService = moderationService;
        this.clientGeoService = clientGeoService;
        this.clientService = clientService;
        this.productService = productService;
        this.geoTreeFactory = geoTreeFactory;
    }

    // конструктор для тестов
    AdGroupService(CampaignRepository campaignRepository,
                   ShardHelper shardHelper,
                   CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        //noinspection ConstantConditions
        this(campaignRepository,
                null,
                null,
                null,
                null,
                null,
                shardHelper,
                null,
                campaignSubObjectAccessCheckerFactory,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                null
        );
    }

    @Override
    public List<AdGroup> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return getAdGroups(clientId, ids);
    }

    /**
     * @return набор {@link AdGroup} по заданным {@code ids}
     */
    public List<AdGroup> getAdGroups(ClientId clientId, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyList();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, ids);

        // fix BS_RARELY_LOADED value for:
        //      - dynamic or mobile_content adgroups
        //      - adgroups in archived campaigns

        // аналог в perl-е в Direct::AdGroups2::get_by:
        // "IF(`campaigns`.`archived` = 'Yes' OR `phrases`.`adgroup_type` IN ('dynamic', 'performance'), 0,
        // `phrases`.`is_bs_rarely_loaded`) AS `is_bs_rarely_loaded`")

        Set<Long> campaignIds = adGroups.stream().map(AdGroup::getCampaignId).collect(toSet());
        Set<Long> archivedCampaignIds = campaignRepository.getArchivedCampaigns(shard, campaignIds);
        adGroups.forEach(group -> {
            AdGroupType type = group.getType();
            if (archivedCampaignIds.contains(group.getCampaignId())
                    || type == AdGroupType.DYNAMIC
                    || type == AdGroupType.PERFORMANCE) {
                group.setBsRarelyLoaded(false);
            }
        });

        return adGroups;
    }

    /**
     * Чтение всех существующих групп (без ограничения на поддержку разных типов групп)
     *
     * @return мапа {@link AdGroupSimple} по идентификаторам групп.
     */
    public Map<Long, AdGroupSimple> getSimpleAdGroups(ClientId clientId, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyMap();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return adGroupRepository.getAdGroupSimple(shard, clientId, ids);
    }

    /**
     * Чтение групп по номерам кампаний
     *
     * @return мапа {@link AdGroupSimple} по идентификаторам кампаний.
     */
    public Map<Long, List<AdGroupSimple>> getSimpleAdGroupsByCampaignIds(ClientId clientId,
                                                                         Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, campaignIds);
    }

    public Map<Long, GeoproductAvailability> getGeoproductAvailabilityByCampaignId(int shard,
                                                                                   Collection<Long> campaignIds) {
        return adGroupRepository.getGeoproductAvailabilityByCampaignId(shard, campaignIds);
    }

    public Map<Long, GeoproductAvailability> getGeoproductAvailabilityByCampaignId(ClientId clientId,
                                                                                   Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getGeoproductAvailabilityByCampaignId(shard, campaignIds);
    }

    /**
     * Чтение всех существующих групп с типами (без ограничения на поддержку разных типов групп)
     *
     * @return мапа {@link AdGroupWithType} по идентификаторам групп.
     */
    public Map<Long, AdGroupWithType> getAdGroupsWithType(ClientId clientId, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyMap();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return adGroupRepository.getAdGroupsWithType(shard, clientId, ids);
    }

    public Map<Long, AdGroupWithTypeAndGeo> getAdGroupsWithTypeAndGeo(ClientId clientId, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyMap();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return adGroupRepository.getAdGroupsWithTypeAndGeo(shard, clientId, ids);
    }

    /**
     * Возвращает тип группы по ее идентификатору (без ограничения на поддержку разных типов групп).
     *
     * @return мапа {@link AdGroupType} по идентификаторам групп.
     */
    public Map<Long, AdGroupType> getAdGroupTypes(ClientId clientId, Collection<Long> ids) {
        return EntryStream.of(getAdGroupsWithType(clientId, ids))
                .mapValues(AdGroupWithType::getType)
                .toMap();
    }

    /**
     * Заполняет баннера для каждой смарт-группы из переданного списка
     */
    public void enrichPerformanceAdGroups(ClientId clientId, List<AdGroup> adGroups) {
        Set<Long> performanceAdGroupIds = StreamEx.of(adGroups)
                .filterBy(AdGroup::getType, AdGroupType.PERFORMANCE)
                .map(AdGroup::getId)
                .toSet();
        if (!performanceAdGroupIds.isEmpty()) {
            List<Banner> banners = bannerTypedRepository.getSafely(shardHelper.getShardByClientIdStrictly(clientId),
                    bannerAdGroupIdFilter(performanceAdGroupIds), List.of(PerformanceBannerMain.class));
            Map<Long, List<BannerWithSystemFields>> bannersByAdGroupId = StreamEx.of(banners)
                    .select(BannerWithSystemFields.class)
                    .groupingBy(BannerWithSystemFields::getAdGroupId);
            adGroups.forEach(adGroup -> {
                if (performanceAdGroupIds.contains(adGroup.getId())) {
                    adGroup.setBanners(bannersByAdGroupId.getOrDefault(adGroup.getId(), emptyList()));
                }
            });
        }
    }

    /**
     * Вычисляет эффективные и отключенные гео и заполняет их для каждой группы из переданного списка
     *
     * @param shard    id шарда
     * @param adGroups список групп
     */
    public void enrichAdGroupsWithEffectiveAndRestrictedGeo(int shard, List<AdGroup> adGroups) {
        if (adGroups.isEmpty()) {
            return;
        }

        GeoTree geoTree = geoTreeFactory.getApiGeoTree();

        List<Long> adGroupIds = adGroups.stream().map(AdGroup::getId).collect(toList());

        Map<Long, List<Long>> bannersMinusGeoByAdGroupIds =
                bannerModerationRepository.getBannersMinusGeoByAdGroupIds(shard, adGroupIds);

        for (AdGroup adGroup : adGroups) {
            if (bannersMinusGeoByAdGroupIds.containsKey(adGroup.getId())) {
                List<Long> minusRegions = bannersMinusGeoByAdGroupIds.get(adGroup.getId());

                List<Long> effectiveGeo = geoTree.excludeRegions(adGroup.getGeo(), minusRegions);
                adGroup.setEffectiveGeo(effectiveGeo);

                List<Long> restrictedGeo = geoTree.getDiffScaledToCountryLevel(adGroup.getGeo(), effectiveGeo);
                adGroup.setRestrictedGeo(restrictedGeo);
            }
        }
    }

    /**
     * Вычисляет эффективные и отключенные гео и заполняет их для каждой группы из переданного списка
     * позорная копипаста, можно переделать на дженерики, но тогда у моделей группы должен быть общий интерфейс с Id
     * и geo
     *
     * @param shard    id шарда
     * @param adGroups список групп
     */
    public void enrichAdGroupsWithEffectiveAndRestrictedGeoForStatuses(int shard,
                                                                       List<AggregatedStatusAdGroup> adGroups) {
        if (adGroups.isEmpty()) {
            return;
        }

        GeoTree geoTree = geoTreeFactory.getApiGeoTree();

        List<Long> adGroupIds = adGroups.stream().map(AggregatedStatusAdGroup::getId).collect(toList());

        Map<Long, List<Long>> bannersMinusGeoByAdGroupIds =
                bannerModerationRepository.getBannersMinusGeoByAdGroupIds(shard, adGroupIds);

        for (AggregatedStatusAdGroup adGroup : adGroups) {
            if (bannersMinusGeoByAdGroupIds.containsKey(adGroup.getId())) {
                List<Long> minusRegions = bannersMinusGeoByAdGroupIds.get(adGroup.getId());

                List<Long> effectiveGeo = geoTree.excludeRegions(adGroup.getGeo(), minusRegions);
                adGroup.setEffectiveGeo(effectiveGeo);

                List<Long> restrictedGeo = geoTree.getDiffScaledToCountryLevel(adGroup.getGeo(), effectiveGeo);
                adGroup.setRestrictedGeo(restrictedGeo);
            }
        }
    }

    /**
     * Отфильтровывает id тех групп и кампаний, на которые не могут быть прочитаны оператором
     *
     * @param operatorUid       uid оператора
     * @param clientId          id клиента
     * @param selectionCriteria условия отбора групп
     *                          <p>
     *                          NB - метод модифицирует свой параметр selectionCriteria!
     */
    void filterSelectionCriteriaAdGroupAndCampaignIds(long operatorUid, ClientId clientId,
                                                      AdGroupsSelectionCriteria selectionCriteria) {
        if (!selectionCriteria.getAdGroupIds().isEmpty()) {
            Set<Long> adGroupIds = selectionCriteria.getAdGroupIds();
            CampaignSubObjectAccessValidator checker = campaignSubObjectAccessCheckerFactory
                    .newAdGroupChecker(operatorUid, clientId, adGroupIds)
                    .createValidator(CampaignAccessType.READ);
            selectionCriteria.setAdGroupIds(adGroupIds
                    .stream()
                    .map(checker)
                    .filter(vr -> !vr.hasAnyErrors())
                    .map(ValidationResult::getValue)
                    .collect(toSet()));
        }

        if (!selectionCriteria.getCampaignIds().isEmpty()) {
            Set<Long> campaignIds = selectionCriteria.getCampaignIds();
            CampaignSubObjectAccessValidator checker = campaignSubObjectAccessCheckerFactory
                    .newCampaignChecker(operatorUid, clientId, campaignIds)
                    .createValidator(CampaignAccessType.READ);
            selectionCriteria.setCampaignIds(campaignIds
                    .stream()
                    .map(checker)
                    .filter(vr -> !vr.hasAnyErrors())
                    .map(ValidationResult::getValue)
                    .collect(toSet()));
        }
    }

    /**
     * Возвращает {@link List}, где элементами являются найденные группы
     *
     * @param operatorUid                           uid оператора
     * @param clientId                              id клиента
     * @param selectionCriteria                     условия отбора групп
     * @param limitOffset                           ограничения на полученный список
     * @param shouldEnrichEffectiveAndRestrictedGeo вычислять эффективные и отключенные гео
     * @return список найденных групп
     */
    public List<AdGroup> getAdGroupsBySelectionCriteria(long operatorUid, ClientId clientId,
                                                        AdGroupsSelectionCriteria selectionCriteria,
                                                        LimitOffset limitOffset,
                                                        boolean shouldEnrichEffectiveAndRestrictedGeo) {
        filterSelectionCriteriaAdGroupAndCampaignIds(operatorUid, clientId, selectionCriteria);

        if (selectionCriteria.getAdGroupIds().isEmpty() && selectionCriteria.getCampaignIds().isEmpty()) {
            return emptyList();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<Long> adGroupIds =
                adGroupRepository.getAdGroupIdsBySelectionCriteria(shard, selectionCriteria, limitOffset);

        List<AdGroup> adGroups = this.getAdGroups(clientId, adGroupIds);

        if (shouldEnrichEffectiveAndRestrictedGeo) {
            enrichAdGroupsWithEffectiveAndRestrictedGeo(shard, adGroups);
        }

        return adGroups;
    }

    @QueryWithForbiddenShardMapping("pid")
    public List<AdGroup> getAdGroupsBySelectionCriteria(AdGroupsSelectionCriteria selectionCriteria,
                                                        LimitOffset limitOffset,
                                                        boolean shouldEnrichEffectiveAndRestrictedGeo) {

        AdGroupsSelectionCriteria selectionCriteriaCopy = selectionCriteria.copy(selectionCriteria);
        if (!CollectionUtils.isEmpty(selectionCriteriaCopy.getCampaignIds())) {
            return shardHelper.groupByShard(selectionCriteria.getCampaignIds(), ShardKey.CID).stream()
                    .mapKeyValue((shard, campaignIds) -> getAdGroupsBySelectionCriteria(shard, selectionCriteriaCopy
                                    .withCampaignIds(new HashSet<>(campaignIds)),
                            limitOffset,
                            shouldEnrichEffectiveAndRestrictedGeo))
                    .toFlatList(identity());
        } else {
            return shardHelper.groupByShard(selectionCriteria.getAdGroupIds(), ShardKey.PID).stream()
                    .mapKeyValue((shard, adGroupIds) -> getAdGroupsBySelectionCriteria(shard, selectionCriteriaCopy
                                    .withAdGroupIds(new HashSet<>(adGroupIds)),
                            limitOffset,
                            shouldEnrichEffectiveAndRestrictedGeo))
                    .toFlatList(identity());
        }
    }

    public List<AdGroup> getAdGroupsBySelectionCriteria(int shard, AdGroupsSelectionCriteria selectionCriteria,
                                                        LimitOffset limitOffset,
                                                        boolean shouldEnrichEffectiveAndRestrictedGeo) {
        final List<Long> ids =
                adGroupRepository.getAdGroupIdsBySelectionCriteria(shard, selectionCriteria, limitOffset);
        final List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, ids);
        if (shouldEnrichEffectiveAndRestrictedGeo) {
            enrichAdGroupsWithEffectiveAndRestrictedGeo(shard, adGroups);
        }
        return adGroups;
    }

    /**
     * Для тех РМП-групп с ID {@code adGroupIds}, для которых задан {@code publisher_domain_id},
     * возвращает соответствующие {@link Domain}.
     * Если группа не является РМП, или у неё не задан {@code publisher_domain_id}, её ID будет отсутствовать
     * в результирующей {@link Map}'е.
     */
    public Map<Long, Domain> getMobileContentPublisherDomains(ClientId clientId, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, Long> publisherDomainIdsByAdGroupIds = adGroupRepository.getPublisherDomainIds(shard, adGroupIds);

        Set<Long> allPublisherDomainIds = StreamEx.ofValues(publisherDomainIdsByAdGroupIds).toSet();
        List<Domain> domains = domainRepository.getDomainsByIdsFromDict(allPublisherDomainIds);
        Map<Long, Domain> domainsByIds = listToMap(domains, Domain::getId);

        List<Long> missedDomains = StreamEx.of(allPublisherDomainIds).remove(domainsByIds::containsKey).toList();
        if (!missedDomains.isEmpty()) {
            logger.error("Can't find domain by ids {} in PPCDICT", missedDomains);
        }

        return EntryStream.of(publisherDomainIdsByAdGroupIds)
                .mapValues(domainsByIds::get)
                .toMap();
    }

    /**
     * Получить идентификаторы приложений из соответствующих магазинов приложений.
     * Они нужны, например, для таргетинга по установленным приложениям, поэтому используются
     * при отправке запросов в Торги.
     *
     * @param clientId   идентификатор клиента
     * @param adGroupIds идентификаторы групп объявлений
     * @return {@link Map} с соответсвием ID РМП группы и идентификатора приложения в app store / google play.
     */
    public Map<Long, String> getMobileContentAppIds(ClientId clientId, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return adGroupRepository.getMobileContentAppIds(shard, adGroupIds);
    }

    public AdGroupsUpdateOperation createPartialUpdateOperation(
            List<ModelChanges<AdGroup>> modelChangesList,
            AdGroupUpdateOperationParams operationParams,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            GeoTree geoTree,
            Long operatorUid,
            ClientId clientId) {
        return createUpdateOperation(
                modelChangesList, operationParams,
                geoTree, minusPhraseValidationMode,
                operatorUid,
                clientId, Applicability.PARTIAL);
    }

    public MassResult<Long> updateAdGroupsPartialWithFullValidation(
            List<ModelChanges<AdGroup>> modelChangesList,
            GeoTree geoTree,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            Long operatorUid,
            ClientId clientId) {
        Objects.requireNonNull(modelChangesList, "modelChangesList");
        Objects.requireNonNull(geoTree, "geoTree");
        Objects.requireNonNull(minusPhraseValidationMode, "minusPhraseValidationMode");
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");

        AdGroupUpdateOperationParams operationParams = AdGroupUpdateOperationParams.builder()
                .withModerationMode(ModerationMode.DEFAULT)
                .withValidateInterconnections(true)
                .build();

        return createUpdateOperation(
                modelChangesList, operationParams,
                geoTree, minusPhraseValidationMode,
                operatorUid, clientId,
                Applicability.PARTIAL)
                .prepareAndApply();
    }

    public UpdateAdGroupMinusKeywordsOperation createAppendMinusKeywordsOperation(
            Applicability applicability,
            List<AdGroupNewMinusKeywords> adGroupNewMinusKeywordsList,
            GeoTree geoTree,
            long operatorUid,
            ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return appendMinusKeywordsOperationFactory.newInstance(
                applicability,
                adGroupNewMinusKeywordsList,
                geoTree,
                UpdateMinusKeywordsMode.ADD,
                operatorUid,
                clientId,
                shard);
    }

    public UpdateAdGroupMinusKeywordsOperation createRemoveMinusKeywordsOperation(
            Applicability applicability,
            List<AdGroupNewMinusKeywords> adGroupNewMinusKeywordsList,
            GeoTree geoTree,
            long operatorUid,
            ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return appendMinusKeywordsOperationFactory.newInstance(
                applicability,
                adGroupNewMinusKeywordsList,
                geoTree,
                UpdateMinusKeywordsMode.REMOVE,
                operatorUid,
                clientId,
                shard);
    }

    public UpdateAdGroupMinusKeywordsOperation createReplaceMinusKeywordsOperation(
            Applicability applicability,
            List<AdGroupNewMinusKeywords> adGroupNewMinusKeywordsList,
            GeoTree geoTree,
            long operatorUid,
            ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return appendMinusKeywordsOperationFactory.newInstance(
                applicability,
                adGroupNewMinusKeywordsList,
                geoTree,
                UpdateMinusKeywordsMode.REPLACE,
                operatorUid,
                clientId,
                shard);
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid, List<AdGroup> entities,
                                Applicability applicability) {

        if (applicability == Applicability.FULL) {
            throw new UnsupportedOperationException("New adGroup service does not support full applicability.");
        }

        GeoTree clientTranslocalGeoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        AdGroupsAddOperation addOperation = createAddOperation(
                entities, clientTranslocalGeoTree, ONE_ERROR_PER_TYPE, operatorUid, clientId,
                Applicability.PARTIAL, true, true);
        return addOperation.prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(
            CopyOperationContainer copyContainer, List<AdGroup> entities, Applicability applicability) {
        if (applicability == Applicability.FULL) {
            throw new UnsupportedOperationException("New adGroup service does not support full applicability.");
        }

        GeoTree clientTranslocalGeoTree = clientGeoService.getClientTranslocalGeoTree(copyContainer.getClientIdTo());
        AdGroupsAddOperation addOperation = createAddOperation(
                entities, clientTranslocalGeoTree,
                ONE_ERROR_PER_TYPE,
                copyContainer.getOperatorUid(),
                copyContainer.getClientIdTo(),
                applicability, true, false); // Запрещаем делать refineGeo, так как для
                                                                // копирования гео надо оставить как есть
        return addOperation.prepareAndApply();
    }

    /**
     * Добавление групп. Добавил только те группы, которые прошли этап валидации
     * сейчас недопустимо null в minusWords. Нужно явно выставлять emptyList
     *
     * @param adGroups                  - список групп
     * @param minusPhraseValidationMode - режим валидации минус-фраз
     * @param operatorUid               - Инициатор операции
     * @param clientId                  - ид клиента, которому будут добавлены группы
     */
    public MassResult<Long> addAdGroupsPartial(
            List<AdGroup> adGroups,
            GeoTree geoTree,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            Long operatorUid,
            ClientId clientId) {
        AdGroupsAddOperation addOperation = createAddOperation(
                adGroups, geoTree, minusPhraseValidationMode, operatorUid, clientId,
                Applicability.PARTIAL, true, true);
        return addOperation.prepareAndApply();
    }

    public AdGroupsAddOperation createAddOperation(
            List<AdGroup> adGroups,
            GeoTree geoTree,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            long operatorUid,
            ClientId clientId, Applicability applicability, boolean saveDraft, boolean refineGeo) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return addOperationFactory.newInstance(
                applicability,
                adGroups,
                geoTree,
                minusPhraseValidationMode,
                operatorUid,
                clientId,
                shard,
                saveDraft,
                refineGeo);
    }

    public AdGroupsUpdateOperation createUpdateOperation(
            List<ModelChanges<AdGroup>> modelChangesList,
            AdGroupUpdateOperationParams operationParams,
            GeoTree geoTree,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            Long operatorUid,
            ClientId clientId,
            Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return updateOperationFactory.newInstance(
                applicability,
                modelChangesList,
                operationParams,
                geoTree,
                minusPhraseValidationMode,
                operatorUid,
                clientId,
                shard);
    }


    /**
     * Массовое удаление групп объявлений для клиента.
     * Считается успешным, если удалена хотя бы одна группа.
     *
     * @param clientId   id клиента
     * @param adGroupIds удаляемые id сайтлинк сетов
     * @return результат удаления сайтлинк сетов
     */
    public MassResult<Long> deleteAdGroups(long operatorUid, ClientId clientId, List<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        ValidationResult<List<Long>, Defect> vr =
                adGroupValidationService.validateDelete(shard, operatorUid, clientId, adGroupIds);
        List<Long> validAdGroupIds = getValidItems(vr);
        if (vr.hasErrors() || validAdGroupIds.isEmpty()) {
            return MassResult.brokenMassAction(adGroupIds, vr);
        }

        Map<Long, Long> adGroupIdToCampaignId = adGroupRepository.getCampaignIdsByAdGroupIds(shard, validAdGroupIds);
        ListMultimap<Long, Long> campaignIdToAdGroupId = Multimaps.index(validAdGroupIds, adGroupIdToCampaignId::get);

        moderationService.deleteAdGroups(shard, clientId, campaignIdToAdGroupId);
        adGroupRepository.delete(shard, clientId, operatorUid, validAdGroupIds);
        shardSupport.deleteValues(ShardKey.PID, validAdGroupIds);

        return MassResult.successfulMassAction(adGroupIds, vr);
    }

    public Map<Long, List<Long>> getAdGroupIdsByCampaignIds(Set<Long> campaignIds) {
        Map<Long, List<Long>> collectResult = new HashMap<>();
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedByDefault()
                .forEach((shard, shardedCids) ->
                        collectResult.putAll(adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds)));
        return collectResult;
    }

    /**
     * Возвращает Map'у adGroupId -> DbStrategy (стратегия на кампанию)
     *
     * @param clientId   идентификатор клиента
     * @param adGroupIds идентификаторы групп объявлений
     * @return {@link Map} с соответствием ID группы и стратегии кампании
     */
    public Map<Long, DbStrategy> getStrategiesByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, AdGroupSimple> adGroupSimple = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);
        Map<Long, Long> adGroupToCampaignId = EntryStream.of(adGroupSimple)
                .mapValues(AdGroupSimple::getCampaignId)
                .toMap();

        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, adGroupToCampaignId.values());
        Map<Long, DbStrategy> campaignIdToStrategy = listToMap(campaigns, Campaign::getId, Campaign::getStrategy);
        return EntryStream.of(adGroupToCampaignId)
                .mapValues(campaignIdToStrategy::get)
                .toMap();
    }

    /**
     * Возвращает Map'у adGroupId -> дефолтная цена
     *
     * @param clientId       идентификатор клиента
     * @param adGroupIds     идентификаторы групп объявлений
     * @param clientCurrency валюта клиента
     * @return {@link Map} с соответствием ID группы и дефолтной цены
     */
    public Map<Long, BigDecimal> getRelevanceMatchDefaultPricesByAdGroupIds(ClientId clientId,
                                                                            Set<Long> adGroupIds,
                                                                            Currency clientCurrency) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, List<Keyword>> keywordsByAdGroupIds =
                keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds);
        return StreamEx.of(adGroupIds)
                .mapToEntry(adGroupId -> keywordsByAdGroupIds.getOrDefault(adGroupId, emptyList()))
                .mapValues(kwList -> kwList.stream().map(Keyword::getPrice).filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .mapValues(prices -> RelevanceMatchUtils.calculatePrice(prices, clientCurrency))
                .toMap();
    }

    /**
     * Возвращает Map'у bannerId -> тип группы
     *
     * @param clientId  идентификатор клиента
     * @param bannerIds идентификаторы баннеров
     * @return {@link Map} с соответствием ID баннера и типа группы
     */
    public Map<Long, AdGroupType> getAdGroupTypesByBannerIds(ClientId clientId, Collection<Long> bannerIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return EntryStream.of(bannerRelationsRepository.getAdGroupTypesByBannerIds(shard, bannerIds))
                .mapValues(AdGroupType::fromSource)
                .toMap();
    }

    public MassResult<Long> addAdGroupsPartial(List<AdGroup> adGroups,
                                               MinusPhraseValidator.ValidationMode validationMode, Long uid,
                                               ClientId clientId) {
        GeoTree clientTranslocalGeoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        return addAdGroupsPartial(adGroups, clientTranslocalGeoTree, validationMode, uid, clientId);
    }

    public Map<Long, List<Long>> getDefaultGeoByCampaignId(ClientId clientId,
                                                           Collection<Long> campaignIds) {
        return getDefaultGeoByCampaignId(clientId, campaignIds, true);
    }

    public Map<Long, List<Long>> getDefaultGeoByCampaignId(ClientId clientId,
                                                           Collection<Long> campaignIds,
                                                           boolean fallbackOnNonAdGroupGeo) {
        Long clientRegion = clientService.getCountryRegionIdByClientIdStrict(clientId);
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        List<Long> clientGeo = List.of(clientRegion);
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, CampaignType> campaignTypeById = campaignRepository.getCampaignsTypeMap(shard, campaignIds);
        Map<Long, List<AdGroupSimple>> adGroupsSimpleByCampaignId =
                adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, campaignIds);

        Map<Long, Set<Integer>> adGroupsUniformGeoByCampaignIds = EntryStream.of(adGroupsSimpleByCampaignId)
                .mapValues(this::getAdGroupsUniformGeoIfApplicable)
                .toMap();
        Map<Long, Set<Integer>> geoByCampaignIds = campaignRepository.getGeoByCampaignIds(shard, campaignIds);

        HashMap<Long, List<Long>> geoByCampaignId = new HashMap<>(campaignIds.size());
        for (Long campaignId : campaignIds) {
            Set<Integer> geoFromAdGroups = adGroupsUniformGeoByCampaignIds.get(campaignId);
            Set<Integer> geoFromCampaign = geoByCampaignIds.get(campaignId);
            CampaignType campaignType = campaignTypeById.get(campaignId);
            if (isNotEmpty(geoFromAdGroups)) {
                geoByCampaignId.put(campaignId, clientGeoService.convertForWeb(mapList(geoFromAdGroups,
                        Integer::longValue), geoTree));
            } else if (shouldFallbackOnNonAdGroupGeo(fallbackOnNonAdGroupGeo, campaignType)) {
                if (isNotEmpty(geoFromCampaign)) {
                    geoByCampaignId.put(campaignId, clientGeoService.convertForWeb(mapList(geoFromCampaign,
                            Integer::longValue), geoTree));
                } else {
                    geoByCampaignId.put(campaignId, clientGeo);
                }
            } else {
                geoByCampaignId.put(campaignId, emptyList());
            }
        }
        return geoByCampaignId;
    }

    private boolean shouldFallbackOnNonAdGroupGeo(boolean fallbackOnNonAdGroupGeo, CampaignType campaignType) {
        boolean nonTextCampaign = campaignType != CampaignType.TEXT;
        return fallbackOnNonAdGroupGeo || nonTextCampaign;
    }

    @Nullable
    @QueryWithForbiddenShardMapping("Используется только в тестах")
    public AdGroup getAdGroup(Long adGroupId) {
        int shard = shardHelper.getShardByGroupId(adGroupId);
        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, Collections.singletonList(adGroupId));
        return adGroups.isEmpty() ? null : adGroups.get(0);
    }

    public Map<Long, Long> getCampaignIdsByAdgroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);
    }

    /**
     * Получение мапы pid->product_restriction_id
     */
    public Map<Long, Long> getProductRestrictionIdsByAdgroupIds(int shard, Collection<AdGroup> adGroups) {
        Map<Long, Long> productIdsByCid = campaignRepository.getProductIds(shard, mapList(adGroups,
                AdGroupWithType::getCampaignId));

        //Мапа ProductId+AdgroupType+CriterionType -> ProductRestriction
        //Задвоение ключей игнорируем, чтобы не падать при ошибочном повторном добавлении записи в product_restrictions
        Map<ProductRestrictionKey, ProductRestriction> productRestrictionsByKey =
                productService.getProductRestrictions()
                        .values()
                        .stream()
                        .flatMap(Collection::stream)
                        .collect(toMap(ProductService::calculateUniqueProductRestrictionKey, identity(),
                                (a1, a2) -> a1));

        //listToMap не работает из-за возможных null в значениях
        Map<Long, Long> productRestrictionIdsByAdgroupIds = new HashMap<>();
        adGroups.forEach(g -> productRestrictionIdsByAdgroupIds.put(g.getId(),
                ifNotNull(productRestrictionsByKey.get(convertGroupToProductRestrictionKey(productIdsByCid, g)),
                        ProductRestriction::getId)));

        return productRestrictionIdsByAdgroupIds;
    }

    private ProductRestrictionKey convertGroupToProductRestrictionKey(Map<Long, Long> productIdsByCid, AdGroup g) {
        CriterionType criterionType;
        if (g instanceof CpmBannerAdGroup) {
            criterionType = ((CpmBannerAdGroup) g).getCriterionType();
        } else if (g instanceof CpmVideoAdGroup) {
            criterionType = ((CpmVideoAdGroup) g).getCriterionType();
        } else {
            criterionType = null;
        }

        return new ProductRestrictionKey()
                .withProductId(productIdsByCid.get(g.getCampaignId()))
                .withAdGroupType(g.getType())
                .withCriterionType(criterionType);
    }

    private Set<Integer> getAdGroupsUniformGeoIfApplicable(Collection<AdGroupSimple> adGroupsSimple) {
        List<Set<Integer>> distictSetsOfGeo = adGroupsSimple.stream()
                .map(s -> listToSet(s.getGeo(), Long::intValue))
                .distinct()
                .collect(toList());
        if (distictSetsOfGeo.size() == 1) {
            return distictSetsOfGeo.get(0);
        }
        return Collections.emptySet();
    }
}
