package ru.yandex.direct.core.entity.adgroup.service;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettings;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettingsProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProviderFactory;
import ru.yandex.direct.core.entity.adgroup.service.validation.AddAdGroupValidationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.service.AddMinusKeywordsPackSubOperation;
import ru.yandex.direct.core.entity.minuskeywordspack.service.AddMinusKeywordsPackSubOperationFactory;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.OperationsUtils;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.tree.ItemSubOperationExecutor;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsUtils.calcBsTagsForDb;
import static ru.yandex.direct.core.entity.adgroup.service.validation.types.CpmIndoorAdGroupValidation.INDOOR_GEO_DEFAULT;
import static ru.yandex.direct.core.entity.adgroup.service.validation.types.CpmOutdoorAdGroupValidation.OUTDOOR_GEO_DEFAULT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция для добавления групп.
 * <p>
 * Также используется при копировании групп, часть служебных полей может быть задана извне.
 */
public class AdGroupsAddOperation extends AbstractAddOperation<AdGroup, Long> {

    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final HyperGeoRepository hyperGeoRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final AddAdGroupValidationService addAdGroupValidationService;
    private final DslContextProvider dslContextProvider;
    private final ItemSubOperationExecutor<AdGroup, List<String>, AddMinusKeywordsPackSubOperation>
            minusKeywordsPackExecutor;
    private final AdGroupOperationsHelper helper;

    private final IdentityHashMap<AdGroup, AdGroupBsTagsSettings> adGroupsBsTagsSettings;


    private final boolean saveDraft;
    private final boolean refineGeo;

    private final GeoTree defaultGeoTree; //TODO
    private final AdGroupGeoTreeProviderFactory geoTreeProviderFactory;
    private final ClientGeoService clientGeoService;
    private final long operatorUid;
    private final ClientId clientId;
    private final int shard;

    private Runnable additionalTask;
    private Map<Long, CampaignSimple> campaignsMap;
    private Map<Long, HyperGeo> hyperGeoById;
    private Map<Long, CampaignTypeSource> campaignTypeSourceById;
    private AdGroupGeoTreeProvider geoTreeProvider;

    public AdGroupsAddOperation(Applicability applicability, List<AdGroup> models,
                                CampaignRepository campaignRepository,
                                AdGroupRepository adGroupRepository,
                                HyperGeoRepository hyperGeoRepository,
                                CryptaSegmentRepository cryptaSegmentRepository,
                                AddAdGroupValidationService addAdGroupValidationService,
                                AdGroupBsTagsSettingsProvider adGroupBsTagsSettingsProvider,
                                AddMinusKeywordsPackSubOperationFactory addMinusKeywordsPackSubOperationFactory,
                                DslContextProvider dslContextProvider,
                                GeoTree defaultGeoTree,
                                AdGroupGeoTreeProviderFactory geoTreeProviderFactory,
                                ClientGeoService clientGeoService,
                                AdGroupOperationsHelper helper,
                                MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                long operatorUid, ClientId clientId, int shard, boolean saveDraft, boolean refineGeo) {
        super(applicability, models);
        this.refineGeo = refineGeo;
        checkMinusKeywordsIdNotSpecified(models);

        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.hyperGeoRepository = hyperGeoRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.addAdGroupValidationService = addAdGroupValidationService;
        this.dslContextProvider = dslContextProvider;
        this.defaultGeoTree = defaultGeoTree;
        this.geoTreeProviderFactory = geoTreeProviderFactory;
        this.clientGeoService = clientGeoService;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;
        this.saveDraft = saveDraft;
        this.helper = helper;

        this.minusKeywordsPackExecutor = addMinusKeywordsPackSubOperationFactory
                .newExecutor(models, AdGroup.MINUS_KEYWORDS, applicability, minusPhraseValidationMode, clientId, shard);

        this.adGroupsBsTagsSettings = adGroupBsTagsSettingsProvider.getAdGroupBsTagsSettings(models, clientId);
    }

    private void checkMinusKeywordsIdNotSpecified(List<AdGroup> adGroups) {
        boolean noMinusKeywordsId = adGroups.stream()
                .map(AdGroup::getMinusKeywordsId).noneMatch(Objects::nonNull);
        checkArgument(noMinusKeywordsId, "minusKeywordsId must not be specified (it's a system field)");
    }

    @Override
    protected ValidationResult<List<AdGroup>, Defect> preValidate(List<AdGroup> models) {
        return addAdGroupValidationService.preValidate(
                models,
                operatorUid,
                clientId,
                shard);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<AdGroup> modelsPreValidatedStep) {
        setSystemFields(modelsPreValidatedStep.getPreValidModelsMap().values());
        modelsPreValidatedStep.getPreValidModelsMap().values()
                .forEach(adGroup -> {
                    if (adGroup.getType() == AdGroupType.CPM_OUTDOOR) {
                        fillCpmOutdoorUnsupportedProperties(adGroup);
                    }
                    if (adGroup.getType() == AdGroupType.CPM_INDOOR) {
                        fillCpmIndoorUnsupportedProperties(adGroup);
                    }
                });
        initCampaignsMap(modelsPreValidatedStep.getModels());
        initHyperGeoMap(modelsPreValidatedStep.getModels());
        initGeoTreeProvider();
    }

    private void setSystemFields(Collection<AdGroup> adGroups) {
        adGroups.forEach(adGroup -> adGroup
                .withStatusBsSynced(StatusBsSynced.NO)
                .withPriorityId(0L));
    }

    private void initCampaignsMap(Collection<AdGroup> models) {
        Set<Long> campaignIds = listToSet(models, AdGroup::getCampaignId);
        campaignsMap = campaignRepository.getCampaignsSimple(shard, campaignIds);
        campaignTypeSourceById = EntryStream.of(campaignsMap)
                .mapValues(campaign -> new CampaignTypeSource(campaign.getType(), campaign.getSource()))
                .toMap();
    }

    private void initHyperGeoMap(Collection<AdGroup> models) {
        Set<Long> hyperGeoIds = StreamEx.of(models)
                .map(AdGroup::getHyperGeoId)
                .nonNull()
                .toSet();
        hyperGeoById = hyperGeoRepository.getHyperGeoById(shard, clientId, hyperGeoIds);
    }

    private void initGeoTreeProvider() {
        var campaignTypes = EntryStream.of(campaignsMap)
                .mapValues(CampaignSimple::getType)
                .toMap();
        geoTreeProvider = geoTreeProviderFactory.create(defaultGeoTree, campaignTypes);
    }

    /**
     * Заполняет дефолтными значениями поля, которые не поддерживаются outdoor группой.
     */
    private static void fillCpmOutdoorUnsupportedProperties(AdGroup adGroup) {
        adGroup.setGeo(OUTDOOR_GEO_DEFAULT);
    }

    /**
     * Заполняет дефолтными значениями поля, которые не поддерживаются indoor группой.
     */
    private static void fillCpmIndoorUnsupportedProperties(AdGroup adGroup) {
        adGroup.setGeo(INDOOR_GEO_DEFAULT);
    }

    @Override
    protected void validate(ValidationResult<List<AdGroup>, Defect> preValidationResult) {
        new ListValidationBuilder<>(preValidationResult)
                .checkBy(adGroups -> addAdGroupValidationService.validate(adGroups, hyperGeoById,
                        campaignTypeSourceById,
                        adGroupsBsTagsSettings, geoTreeProvider, clientId, operatorUid, shard));
        minusKeywordsPackExecutor.prepare(preValidationResult);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<AdGroup> modelsValidatedStep) {
        Collection<AdGroup> adGroups = modelsValidatedStep.getValidModelsMap().values();
        setStatuses(adGroups, campaignsMap);
        helper.fillHyperGeoSegmentIds(adGroups, hyperGeoById);
        refineGeo(adGroups);
        prepareBsTags(adGroups);
        computeAdditionalTask(campaignsMap);
        prepareCryptaGoals(adGroups);
    }

    /**
     * При создании группы учитываем флаг {@code saveDraft} для выставления статусов.
     * Необходимо для различия состояний создания новой группы и копирования существующей вместе с служебными полями.
     */
    private void setStatuses(Collection<AdGroup> validModels, Map<Long, CampaignSimple> campaignsMap) {
        for (AdGroup model : validModels) {
            model.setBsRarelyLoaded(false);
            model.setStatusAutobudgetShow(true);

            CampaignSimple campaign = campaignsMap.get(model.getCampaignId());

            boolean moderationAllowed = campaign.getStatusModerate() != CampaignStatusModerate.NEW;

            boolean shouldSaveAsDraft = !moderationAllowed || saveDraft;
            if (shouldSaveAsDraft) {
                model.setStatusModerate(StatusModerate.NEW);
            } else {
                model.setStatusModerate(StatusModerate.READY);
            }

            model.setStatusPostModerate(model.getStatusPostModerate() == StatusPostModerate.REJECTED
                    ? StatusPostModerate.REJECTED
                    : StatusPostModerate.NO);

            if (!shouldSaveAsDraft &&
                    model.getType().equals(AdGroupType.PERFORMANCE)) {
                // группу смарт-объявлений не отправляем на модерацию, а сразу устанвливаем флаг промодерированности
                model.setStatusModerate(StatusModerate.YES);
                model.setStatusPostModerate(StatusPostModerate.YES);
            }

            if (model instanceof InternalAdGroup) {
                model.setStatusModerate(StatusModerate.YES);
                model.setStatusPostModerate(StatusPostModerate.YES);
            }
        }
    }

    private void refineGeo(Collection<AdGroup> adGroups) {
        adGroups.stream()
                // для outdoor и indoor групп geo всегда должен быть дефолтным
                .filter(this::needRefineGeo)
                .forEach(adGroup -> {
                    Long hyperGeoId = adGroup.getHyperGeoId();
                    if (hyperGeoId != null) {
                        HyperGeo hyperGeo = hyperGeoById.get(hyperGeoId);
                        List<HyperGeoSegment> hyperGeoSegments = hyperGeo.getHyperGeoSegments();

                        checkState(hyperGeoSegments.size() > 0, "there must be at least one geosegment");
                        HyperGeoSegment hyperGeoSegment = hyperGeoSegments.get(0);

                        // в случае гиперлокального гео на группе, нужно поставить минимальный покрывающий регион
                        // в качестве обычного гео, чтобы БК и модерация не сломались
                        adGroup.setGeo(hyperGeoSegment.getCoveringGeo());
                    } else {
                        var geoTree = geoTreeProvider.getGeoTree(adGroup); // SUSP
                        adGroup.setGeo(clientGeoService.convertForSave(adGroup.getGeo(), geoTree));
                        adGroup.setGeo(geoTree.refineGeoIds(adGroup.getGeo()));
                    }
                });
    }

    private boolean needRefineGeo(AdGroup adGroup) {
        return refineGeo && adGroup.getType() != AdGroupType.CPM_OUTDOOR && adGroup.getType() != AdGroupType.CPM_INDOOR;
    }

    private void prepareBsTags(Collection<AdGroup> adGroups) {
        adGroups.forEach(adGroup -> {
            var adGroupBsTagsSettings = adGroupsBsTagsSettings.get(adGroup);
            adGroup
                    .withPageGroupTags(calcBsTagsForDb(adGroup.getPageGroupTags(),
                            adGroupBsTagsSettings.getRequiredPageGroupTags(),
                            adGroupBsTagsSettings.getDefaultPageGroupTags()))
                    .withTargetTags(calcBsTagsForDb(adGroup.getTargetTags(),
                            adGroupBsTagsSettings.getRequiredTargetTags(),
                            adGroupBsTagsSettings.getDefaultTargetTags()));
        });
    }

    private void computeAdditionalTask(Map<Long, CampaignSimple> campaignsMap) {
        additionalTask = () -> {
            if (!saveDraft) {
                campaignRepository.sendRejectedCampaignsToModerate(shard, campaignsMap.keySet());
            }
        };
    }

    @Override
    protected Map<Integer, Long> execute(Map<Integer, AdGroup> validModelsMapToApply) {
        Map<Integer, Long> adGroupIds = new HashMap<>();

        addMinusKeywordsPacks(validModelsMapToApply);
        dslContextProvider.ppcTransaction(shard, conf -> {
            Map<Integer, Long> addedAdGroupIds = OperationsUtils.applyForMapValues(validModelsMapToApply,
                    validModelsToApply -> adGroupRepository.addAdGroups(conf, clientId, validModelsToApply));

            adGroupIds.putAll(addedAdGroupIds);
        });
        additionalTask.run();
        return adGroupIds;
    }

    /**
     * Для валидных групп с минус фразами добавляем в базу приватные наборы минус фраз и проставляем в группы их id,
     * а поле {@link AdGroup#MINUS_KEYWORDS} очищаем.
     */
    private void addMinusKeywordsPacks(Map<Integer, AdGroup> adGroupsMap) {
        Map<Integer, Integer> adGroupIndexToPackIndex = minusKeywordsPackExecutor.getIndexMap();
        Map<Integer, Integer> adGroupValidIndexesToPackIndexes = EntryStream.of(adGroupIndexToPackIndex)
                .filterKeys(adGroupsMap::containsKey)
                .toMap();
        Set<Integer> packIndexesToApply = new HashSet<>(adGroupValidIndexesToPackIndexes.values());
        minusKeywordsPackExecutor.getSubOperation().setIndexesToApply(packIndexesToApply);
        minusKeywordsPackExecutor.apply();

        List<Long> minusKeywordPackIds = minusKeywordsPackExecutor.getSubOperation().getMinusKeywordPackIds();
        adGroupValidIndexesToPackIndexes.forEach((adGroupIndex, packIndex) -> {
            AdGroup adGroup = adGroupsMap.get(adGroupIndex);
            adGroup.setMinusKeywordsId(minusKeywordPackIds.get(packIndex));
            adGroup.setMinusKeywords(null);
        });
    }

    private void prepareCryptaGoals(Collection<AdGroup> adGroups) {
        var adGroupGoals = flatMap(adGroups,
                adGroup -> flatMap(
                        nvl(adGroup.getContentCategoriesRetargetingConditionRules(), Collections.emptyList()),
                        Rule::getGoals));

        var dbGoals = cryptaSegmentRepository.getByIds(mapList(adGroupGoals, Goal::getId));
        filterList(adGroupGoals, goal -> dbGoals.get(goal.getId()) != null)
                .forEach(goal -> goal.setParentId(
                        goal.getType().equals(GoalType.CONTENT_GENRE) && goal.getId() != Goal.CINEMA_GENRES_GOAL_ID ?
                                Goal.CINEMA_GENRES_GOAL_ID : // у Жанров в пакете сохраняется -42 как категория
                                dbGoals.get(goal.getId()).getParentId())
                );
    }
}
