package ru.yandex.direct.core.entity.adgroup.service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.SearchKeywordResult;
import ru.yandex.direct.advq.exception.AdvqClientException;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupShowsForecast;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.KeywordForecast;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.KeywordShowsForecastService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис работы с прогнозами показов для групп объявлений
 */
@Service
@ParametersAreNonnullByDefault
public class AdGroupsShowsForecastService {
    private static final int KEYWORD_FORECAST_UPDATE_PARTITION_SIZE = 100;
    private static final int KEYWORD_FORECAST_UPDATE_PARTITION_SIZE_FROM_JOB = 1_000;

    private static final Logger logger = LoggerFactory.getLogger(AdGroupsShowsForecastService.class);

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final KeywordRepository keywordRepository;
    private final KeywordShowsForecastService keywordShowsForecastService;

    @Autowired
    public AdGroupsShowsForecastService(
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            KeywordRepository keywordRepository,
            KeywordShowsForecastService keywordShowsForecastService) {
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
        this.keywordRepository = keywordRepository;
        this.keywordShowsForecastService = keywordShowsForecastService;
    }

    /**
     * Обновляет прогнозы показов для <i>всех</i> фраз групп, для которых требуется обновление.
     * Необходимость обновления прогнозов определяется значением {@link AdGroup#STATUS_SHOWS_FORECAST}.
     * Условие необходимости обновления: {@code statusShowsForecast != 'Processed' and statusShowsForecast != 'Archive'}
     *
     * @param advqTimeout если задан, ограничивает время выполнения запроса к ADVQ. Если за этот таймаут данные получить
     *                    не удаётся, прогнозы показоов не обновляются в группе, для которой были ошибки.
     */
    public void updateShowsForecastIfNeeded(ClientId clientId, Collection<Long> adGroupIds,
                                            @Nullable Duration advqTimeout) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);
        // 0. Отобрать те группы, у которых statusShowsForecast != 'Processed' and statusShowsForecast != 'Archive'

        List<AdGroup> adGroupsToUpdate = StreamEx.of(adGroups)
                .remove(adGroup -> adGroup.getStatusShowsForecast() == StatusShowsForecast.PROCESSED)
                .remove(adGroup -> adGroup.getStatusShowsForecast() == StatusShowsForecast.ARCHIVED)
                .toList();
        if (!adGroupsToUpdate.isEmpty()) {
            updateShowsForecast(clientId, adGroupsToUpdate, advqTimeout);
        }
    }

    private void updateShowsForecast(ClientId clientId, List<AdGroup> adGroups, @Nullable Duration advqTimeout) {
        logger.debug("updateShowsForecast({}, {})", clientId, adGroups);
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<Long> adGroupToUpdateIds = mapList(adGroups, AdGroup::getId);

        adGroupRepository.updateStatusShowsForecast(shard, adGroupToUpdateIds, StatusShowsForecast.SENDING);

        Map<Long, List<Keyword>> keywordsByAdGroupIds =
                keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupToUpdateIds);

        IdentityHashMap<Keyword, SearchKeywordResult> phrasesShows;
        try {
            List<Keyword> keywords = StreamEx.ofValues(keywordsByAdGroupIds)
                    .flatMap(Collection::stream)
                    .toList();

            List<Long> campaignIds = mapList(adGroups, AdGroup::getCampaignId);
            Map<Long, Campaign> campaignByIds = campaignRepository.getCampaignsMap(shard, campaignIds);

            phrasesShows = keywordShowsForecastService.getPhrasesShows(adGroups, keywords, advqTimeout, clientId,
                    campaignByIds);
        } catch (AdvqClientException e) {
            // ошибки вызова ADVQ пробрасываются в runtime AdvqClientException-обёртке
            logger.error("Error when requesting ADVQ/search", e);
            phrasesShows = new IdentityHashMap<>();
        }

        // сохраняем полученные прогнозы для фраз
        List<AdGroupShowsForecast> adGroupsForecasts = new ArrayList<>(adGroups.size());
        Set<Long> adGroupsWithErrorsIds = new HashSet<>();
        List<Long> keywordsWithMissedForecast = new ArrayList<>();
        for (AdGroup adGroup : adGroups) {
            Long adGroupId = adGroup.getId();
            List<KeywordForecast> newShowsForecast = new ArrayList<>();
            List<Keyword> currentKeywords = keywordsByAdGroupIds.get(adGroupId);
            for (Keyword keyword : currentKeywords) {
                // Пара проверок, что прогнозы вообще пришли
                if (!phrasesShows.containsKey(keyword)) {
                    keywordsWithMissedForecast.add(keyword.getId());
                    adGroupsWithErrorsIds.add(adGroupId);
                    continue;
                }
                SearchKeywordResult searchKeywordResult = phrasesShows.get(keyword);
                if (searchKeywordResult.hasErrors()) {
                    logger.debug("Errors while getting showsForecast for {}: {}", keyword.getId(),
                            searchKeywordResult.getErrors());
                    keywordsWithMissedForecast.add(keyword.getId());
                    adGroupsWithErrorsIds.add(adGroupId);
                    continue;
                }
                @SuppressWarnings("ConstantConditions") // если ошибок не было getResult() должен присутствовать
                        Long showsForecast = searchKeywordResult.getResult().getTotalCount();

                KeywordForecast keywordForecast = new KeywordForecast()
                        .withId(keyword.getId())
                        .withKeyword(keyword.getPhrase())
                        .withShowsForecast(showsForecast);
                newShowsForecast.add(keywordForecast);
            }

            String geoStr = AdGroupMappings.geoToDb(adGroup.getGeo());
            AdGroupShowsForecast adGroupShowsForecast = new AdGroupShowsForecast()
                    .withId(adGroupId)
                    .withGeo(geoStr)
                    .withForecastDate(LocalDateTime.now())
                    .withKeywordForecasts(newShowsForecast);
            adGroupsForecasts.add(adGroupShowsForecast);
        }
        if (!keywordsWithMissedForecast.isEmpty()) {
            // логируем ID ключевых фраз, по которым не удалось получить прогнозы
            logger.warn("Missed showsForecast for keywords: {}", keywordsWithMissedForecast);
        }

        updateKeywordShowsForecastByPartitions(shard, adGroupsForecasts, KEYWORD_FORECAST_UPDATE_PARTITION_SIZE);

        // Выставляем forecastDate и statusShowsForecast только на группах, все фразы которых были успешно обработаны
        List<AdGroupShowsForecast> adGroupsForecastsWithoutErrors = StreamEx.of(adGroupsForecasts)
                .remove(adGroupShowsForecast -> adGroupsWithErrorsIds.contains(adGroupShowsForecast.getId()))
                .toList();
        // этот вызов помимо forecastDate обновляет statusShowsForecast -> Processed и не трогает LastChange
        adGroupRepository.updateForecastDateForSendingStatus(shard, adGroupsForecastsWithoutErrors);

        // Для групп, по которым не смогли получить данныу обо всех фразах, оставляем статус 'Sending'
    }

    /**
     * Обновляет прогнозы для фраз из переданного набора {@link AdGroupShowsForecast}.
     * Фразы обновляются пачками по {@value #KEYWORD_FORECAST_UPDATE_PARTITION_SIZE} элементов.
     */
    private void updateKeywordShowsForecastByPartitions(int shard, List<AdGroupShowsForecast> adGroupsForecasts,
                                                        int partitionSize) {
        List<KeywordForecast> allKeywordForecasts = StreamEx.of(adGroupsForecasts)
                .flatCollection(AdGroupShowsForecast::getKeywordForecasts)
                .toList();

        // дробим все прогнозы по группам и обновляем
        for (List<KeywordForecast> keywordForecasts : Lists.partition(allKeywordForecasts, partitionSize)) {
            keywordRepository.updateShowsForecast(shard, keywordForecasts);
        }
    }

    /**
     * Обновить прогноз показов для фраз групп из списка в заданном шарде.
     * <p>
     * Пропускает группы, у которых не совпадает значение геотаргетинга,
     * затем обновляет значения прогнозов при совпадении текстов фраз,
     * после чего обновляет дату пересчета прогноза и статус при совпадении геотаргетинга групп.
     * Статус прогноза показов - никак не учитывает!
     * <p>
     * Важно - обновление никак не учитывает расхождения между переданными с прогнозом фразами и фактическими в группе.
     * Из-за этого могут получаться группы в "обработанном" статусе и фразами без прогнозов.
     *
     * @param shard             номер шарда для работа
     * @param adGroupsForecasts список прогнозов показов для групп объявлений
     * @param newForecastDate   новое значение (одинаковое для всех групп) времени расчета прогноза
     */
    public void updateAdGroupsShowsForecast(int shard, List<AdGroupShowsForecast> adGroupsForecasts,
                                            LocalDateTime newForecastDate) {
        Set<Long> adGroupIdsForForecastUpdate =
                adGroupRepository.getAdGroupIdsWithUnchangedGeo(shard, adGroupsForecasts);

        if (adGroupIdsForForecastUpdate.isEmpty()) {
            return;
        }

        List<AdGroupShowsForecast> groupsToUpdate = adGroupsForecasts.stream()
                .filter(a -> adGroupIdsForForecastUpdate.contains(a.getId()))
                .map(a -> a.withForecastDate(newForecastDate))
                .collect(Collectors.toList());

        updateKeywordShowsForecastByPartitions(shard, groupsToUpdate, KEYWORD_FORECAST_UPDATE_PARTITION_SIZE_FROM_JOB);
        adGroupRepository.updateForecastDateForAnyStatus(shard, groupsToUpdate);
    }
}
