package ru.yandex.direct.core.entity.adgroup.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupUpdateOperationParams;
import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettings;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettingsProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProviderFactory;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupUpdateData;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupUpdateServices;
import ru.yandex.direct.core.entity.adgroup.service.validation.UpdateAdGroupValidationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.service.AddMinusKeywordsPackSubOperation;
import ru.yandex.direct.core.entity.minuskeywordspack.service.AddMinusKeywordsPackSubOperationFactory;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.tree.ItemSubOperationExecutor;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsUtils.calcBsTagsForDb;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.restrictedRegions;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

public class AdGroupsUpdateOperation extends SimpleAbstractUpdateOperation<AdGroup, Long> {
    private static final Set<ModelProperty<? super AdGroup, ?>> SENSITIVE_PROPERTIES =
            ImmutableSet.of(
                    AdGroup.STATUS_BS_SYNCED, AdGroup.STATUS_MODERATE, AdGroup.STATUS_POST_MODERATE,
                    AdGroup.STATUS_SHOWS_FORECAST, AdGroup.STATUS_AUTOBUDGET_SHOW);

    private static final Set<ModelProperty<?, ?>> SENSITIVE_PROPERTIES_UNBOUNDED =
            ImmutableSet.copyOf(SENSITIVE_PROPERTIES);

    private final Applicability applicability;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final HyperGeoRepository hyperGeoRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final UpdateAdGroupValidationService updateValidationService;
    private final AdGroupGeoTreeProviderFactory geoTreeProviderFactory;
    private final ClientGeoService clientGeoService;
    private final AdGroupUpdateServices updateServices;
    private final AddMinusKeywordsPackSubOperationFactory addMinusKeywordsPackSubOperationFactory;
    private final AdGroupBsTagsSettingsProvider adGroupBsTagsSettingsProvider;
    private final MinusPhraseValidator.ValidationMode minusPhraseValidationMode;
    private final AdGroupOperationsHelper helper;

    private ItemSubOperationExecutor<AdGroup, List<String>, AddMinusKeywordsPackSubOperation> minusKeywordsPackExecutor;
    private IdentityHashMap<AdGroup, AdGroupBsTagsSettings> adGroupsBsTagsSettings;

    private final AdGroupUpdateOperationParams operationParams;
    private final Long operatorUid;
    private final ClientId clientId;
    private final int shard;
    private final GeoTree defaultGeoTree;

    private AdGroupGeoTreeProvider geoTreeProvider;
    private Map<Long, Long> campaignIdByAdGroupId;
    private Map<Long, HyperGeo> hyperGeoById;
    private Map<Long, CampaignTypeSource> campaignTypeSourceById;
    private Map<Long, CampaignSimple> campaignSimpleByCampaignId;
    private Set<Long> adGroupsWithChangedGeoIds;

    AdGroupsUpdateOperation(
            Applicability applicability,
            List<ModelChanges<AdGroup>> modelChangesList,
            AdGroupUpdateOperationParams operationParams,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            HyperGeoRepository hyperGeoRepository,
            CryptaSegmentRepository cryptaSegmentRepository,
            UpdateAdGroupValidationService updateValidationService,
            GeoTree defaultGeoTree,
            AdGroupGeoTreeProviderFactory geoTreeProviderFactory,
            ClientGeoService clientGeoService,
            AdGroupUpdateServices updateServices,
            AdGroupOperationsHelper helper,
            AdGroupBsTagsSettingsProvider adGroupBsTagsSettingsProvider,
            AddMinusKeywordsPackSubOperationFactory addMinusKeywordsPackSubOperationFactory,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            Long operatorUid, ClientId clientId, int shard) {
        super(applicability, modelChangesList, id -> new UntypedAdGroup().withId(id), SENSITIVE_PROPERTIES_UNBOUNDED);
        checkMinusKeywordsIdNotChanged(modelChangesList);

        this.applicability = applicability;
        this.operationParams = operationParams;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.hyperGeoRepository = hyperGeoRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.updateValidationService = updateValidationService;
        this.defaultGeoTree = defaultGeoTree;
        this.geoTreeProviderFactory = geoTreeProviderFactory;
        this.clientGeoService = clientGeoService;
        this.updateServices = updateServices;
        this.helper = helper;
        this.adGroupBsTagsSettingsProvider = adGroupBsTagsSettingsProvider;
        this.addMinusKeywordsPackSubOperationFactory = addMinusKeywordsPackSubOperationFactory;
        this.minusPhraseValidationMode = minusPhraseValidationMode;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;
    }

    private void checkMinusKeywordsIdNotChanged(List<ModelChanges<AdGroup>> adGroupsChanges) {
        boolean noMinusKeywordsIdChanged = adGroupsChanges.stream()
                .noneMatch(changes -> changes.isPropChanged(AdGroup.MINUS_KEYWORDS_ID));
        checkArgument(noMinusKeywordsIdChanged, "minusKeywordsId must not be changed (it's a system field)");
    }

    @Override
    protected ValidationResult<List<ModelChanges<AdGroup>>, Defect> validateModelChanges(
            List<ModelChanges<AdGroup>> modelChangesList) {
        ValidationResult<List<ModelChanges<AdGroup>>, Defect> accessValidationResult =
                updateValidationService.preValidateAccess(
                        modelChangesList,
                        operatorUid,
                        clientId,
                        shard
                );
        if (accessValidationResult.hasErrors()) {
            return accessValidationResult;
        }

        List<ModelChanges<AdGroup>> validChanges = getValidItems(accessValidationResult);
        initCampaignMaps(validChanges);
        initHyperGeoMap(validChanges);
        initGeoTreeProvider(defaultGeoTree, geoTreeProviderFactory);

        return updateValidationService.preValidate(
                accessValidationResult,
                hyperGeoById,
                campaignTypeSourceById,
                geoTreeProvider,
                clientId,
                shard);
    }

    private void initCampaignMaps(List<ModelChanges<AdGroup>> changes) {
        var adGroupIds = listToSet(changes, ModelChanges::getId);
        campaignIdByAdGroupId = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);
        var campaignIds = campaignIdByAdGroupId.values();
        campaignSimpleByCampaignId = campaignRepository.getCampaignsSimple(shard, campaignIds);
        campaignTypeSourceById = EntryStream.of(campaignSimpleByCampaignId)
                .mapValues(campaign -> new CampaignTypeSource(campaign.getType(), campaign.getSource()))
                .toMap();
    }

    private void initHyperGeoMap(List<ModelChanges<AdGroup>> changes) {
        Set<Long> hyperGeoIds = StreamEx.of(changes)
                .map(mc -> mc.getPropIfChanged(AdGroup.HYPER_GEO_ID))
                .nonNull()
                .toSet();
        hyperGeoById = hyperGeoRepository.getHyperGeoById(shard, clientId, hyperGeoIds);
    }

    private void initGeoTreeProvider(GeoTree defaultGeoTree, AdGroupGeoTreeProviderFactory geoTreeProviderFactory) {
        var campaignTypes = EntryStream.of(campaignSimpleByCampaignId)
                .mapValues(CampaignSimple::getType)
                .toMap();
        geoTreeProvider = geoTreeProviderFactory.create(defaultGeoTree, campaignTypes);
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<AdGroup> modelChangesValidatedStep) {
        prepareGeoForSaving(modelChangesValidatedStep.getValidModelChanges());
        prepareDeletingMinusKeywords(modelChangesValidatedStep.getValidModelChanges());
        prepareCryptaGoals(modelChangesValidatedStep.getValidModelChanges());
    }

    void prepareGeoForSaving(Collection<ModelChanges<AdGroup>> validAdGroups) {
        setAdGroupsWithChangedGeoIds(validAdGroups);
        refineGeo(validAdGroups);
    }

    /**
     * Для удаляемых минус фраз, удаляем и id набора минус фраз
     */
    private void prepareDeletingMinusKeywords(Collection<ModelChanges<AdGroup>> validAdGroups) {
        validAdGroups.forEach(adGroupModelChanges -> {
            if (adGroupModelChanges.isPropChanged(AdGroup.MINUS_KEYWORDS)
                    && CollectionUtils.isEmpty(adGroupModelChanges.getChangedProp(AdGroup.MINUS_KEYWORDS))) {
                adGroupModelChanges.process(null, AdGroup.MINUS_KEYWORDS_ID);
            }
        });
    }

    @Override
    protected ValidationResult<List<ModelChanges<AdGroup>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<AdGroup>>, Defect> preValidateResult,
            Map<Long, AdGroup> models) {
        return updateValidationService.validateAdGroupsType(preValidateResult, models);
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<AdGroup> changesAppliedStep) {
        var validAdGroups = mapList(changesAppliedStep.getAppliedChangesForValidModelChanges(),
                AppliedChanges::getModel);

        updateServices.prefetchAdGroupsBanners(shard, validAdGroups);
        createMinusKeywordsPackSubOperation(changesAppliedStep);
        initAdGroupsBsTagsSettings(validAdGroups);
    }

    private void createMinusKeywordsPackSubOperation(ChangesAppliedStep<AdGroup> changesAppliedStep) {
        minusKeywordsPackExecutor = addMinusKeywordsPackSubOperationFactory.newExecutor(
                mapList(changesAppliedStep.getModelChanges(), ModelChanges::toModel),
                AdGroup.MINUS_KEYWORDS,
                applicability, minusPhraseValidationMode, clientId, shard);
    }

    private void initAdGroupsBsTagsSettings(List<AdGroup> adGroups) {
        adGroupsBsTagsSettings = adGroupBsTagsSettingsProvider.getAdGroupBsTagsSettings(adGroups, clientId);
    }

    @Override
    protected ValidationResult<List<AdGroup>, Defect> validateAppliedChanges(
            ValidationResult<List<AdGroup>, Defect> validationResult,
            Map<Integer, AppliedChanges<AdGroup>> appliedChangesForValidModelChanges) {
        ValidationResult<List<AdGroup>, Defect> adGroupsVr =
                updateValidationService.validate(shard, clientId, operatorUid, validationResult,
                        appliedChangesForValidModelChanges,
                        adGroupsBsTagsSettings, adGroupsWithChangedGeoIds,
                        operationParams.isValidateInterconnections());
        minusKeywordsPackExecutor.prepare(adGroupsVr);
        return adGroupsVr;
    }

    @Override
    protected Collection<AdGroup> getModels(Collection<Long> ids) {
        return adGroupRepository.getAdGroups(shard, ids);
    }

    @Override
    protected void beforeExecution(ExecutionStep<AdGroup> executionStep) {
        appendWarningForAdGroupsWithRestrictedRegions(executionStep);
        addMinusKeywordsPacks(executionStep.getAppliedChangesForExecutionWithIndex());
        prepareBsTags(executionStep);
        helper.setHyperRegionGoalIdsForChanges(executionStep.getAppliedChangesForExecution(), hyperGeoById);
    }

    /**
     * Добавить warning для групп объявлений, в которых новый Geo содержит хотя бы один минус-регион хотя бы одного
     * объявления, входящего в группу. Например, если для группы объявлений задается новый Geo "Москва, Киев" и в данную
     * группу входит объявление с минус-регионом "Россия", то для данной группы нужно добавить warning
     */
    private void appendWarningForAdGroupsWithRestrictedRegions(ExecutionStep<AdGroup> executionStep) {
        List<Long> adGroupIds = executionStep.getAppliedChangesForExecution()
                .stream()
                .map(r -> r.getModel().getId())
                .collect(toList());

        Map<Long, List<Long>> bannersMinusGeoByAdGroupIds = updateServices.getBannersMinusGeoByAdGroupIds(
                shard, adGroupIds);

        ValidationResult<List<AdGroup>, Defect> validationResult = executionStep.getValidationResult();

        EntryStream.of(executionStep.getAppliedChangesForExecutionWithIndex())
                .forKeyValue((idx, changes) -> {
                    AdGroup adGroup = changes.getModel();
                    GeoTree geoTree = geoTreeProvider.getGeoTree(adGroup);

                    List<Long> bannerMinusRegions = bannersMinusGeoByAdGroupIds.getOrDefault(
                            adGroup.getId(), emptyList());
                    if (bannerMinusRegions.isEmpty()) {
                        return;
                    }

                    List<Long> effectiveGeoIds = geoTree.excludeRegions(adGroup.getGeo(), bannerMinusRegions);
                    List<Long> restrictedGeoIds =
                            geoTree.getDiffScaledToCountryLevel(adGroup.getGeo(), effectiveGeoIds);
                    if (!restrictedGeoIds.isEmpty()) {
                        validationResult.getOrCreateSubValidationResult(index(idx), adGroup)
                                .addWarning(restrictedRegions());
                    }
                });
    }

    /**
     * Для валидных групп с минус фразами добавляем в базу приватные наборы минус фраз и проставляем в группы их id,
     * а поле {@link AdGroup#MINUS_KEYWORDS} очищаем.
     */
    private void addMinusKeywordsPacks(Map<Integer, AppliedChanges<AdGroup>> validAppliedChangesWithIndex) {
        Map<Integer, Integer> adGroupIndexToPackIndex = minusKeywordsPackExecutor.getIndexMap();
        Map<Integer, Integer> adGroupValidIndexesToPackIndexes = EntryStream.of(adGroupIndexToPackIndex)
                .filterKeys(validAppliedChangesWithIndex::containsKey)
                .toMap();
        Set<Integer> packIndexesToApply = new HashSet<>(adGroupValidIndexesToPackIndexes.values());
        minusKeywordsPackExecutor.getSubOperation().setIndexesToApply(packIndexesToApply);
        minusKeywordsPackExecutor.apply();

        List<Long> minusKeywordPackIds = minusKeywordsPackExecutor.getSubOperation().getMinusKeywordPackIds();
        adGroupValidIndexesToPackIndexes.forEach((adGroupIndex, packIndex) -> {
            AppliedChanges<AdGroup> adGroupAppliedChanges = validAppliedChangesWithIndex.get(adGroupIndex);
            adGroupAppliedChanges.modify(AdGroup.MINUS_KEYWORDS, null);
            adGroupAppliedChanges.modify(AdGroup.MINUS_KEYWORDS_ID, minusKeywordPackIds.get(packIndex));
        });
    }

    private void prepareBsTags(ExecutionStep<AdGroup> executionStep) {
        executionStep.getAppliedChangesForExecution().stream()
                .filter(ch -> ch.changed(AdGroup.PAGE_GROUP_TAGS) || ch.changed(AdGroup.TARGET_TAGS))
                .map(AppliedChanges::getModel)
                .forEach(adGroup -> {
                    var adGroupBsTagsSettings = adGroupsBsTagsSettings.get(adGroup);
                    adGroup
                            .withPageGroupTags(calcBsTagsForDb(adGroup.getPageGroupTags(),
                                    adGroupBsTagsSettings.getRequiredPageGroupTags(),
                                    adGroupBsTagsSettings.getDefaultPageGroupTags()))
                            .withTargetTags(calcBsTagsForDb(adGroup.getTargetTags(),
                                    adGroupBsTagsSettings.getRequiredTargetTags(),
                                    adGroupBsTagsSettings.getDefaultTargetTags()));
                });
    }

    private void prepareCryptaGoals(Collection<ModelChanges<AdGroup>> validAdGroups) {
        var adGroupGoals = flatMap(validAdGroups,
                adGroup -> flatMap(
                        nvl(adGroup.toModel().getContentCategoriesRetargetingConditionRules(), emptyList()),
                        Rule::getGoals));

        var dbGoals = cryptaSegmentRepository.getByIds(mapList(adGroupGoals, Goal::getId));
        filterList(adGroupGoals, goal -> dbGoals.get(goal.getId()) != null)
                .forEach(goal -> goal.setParentId(
                        goal.getType().equals(GoalType.CONTENT_GENRE) && goal.getId() != Goal.CINEMA_GENRES_GOAL_ID ?
                                Goal.CINEMA_GENRES_GOAL_ID : // у Жанров в пакете сохраняется -42 как категория
                                dbGoals.get(goal.getId()).getParentId())
                );
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<AdGroup>> applicableAppliedChanges) {
        List<AdGroupUpdateData> adGroupsUpdateData = obtainAdGroupsUpdateData(applicableAppliedChanges);
        adGroupsUpdateData.stream()
                .collect(groupingBy(c -> c.getAdGroupChanges().getModel().getType()))
                .forEach((adGroupType, adGroupUpdateData) -> updateServices.getUpdateService(adGroupType)
                        .update(shard, clientId, operationParams.getModerationMode(), adGroupUpdateData));
        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    private List<AdGroupUpdateData> obtainAdGroupsUpdateData(
            Collection<AppliedChanges<AdGroup>> applicableAppliedChanges) {
        return StreamEx.of(applicableAppliedChanges)
                .map(appliedChanges -> {
                    Long campaignId = appliedChanges.getModel().getCampaignId();
                    CampaignSimple campaign = campaignSimpleByCampaignId.get(campaignId);
                    var banners = appliedChanges.getModel().getBanners();
                    return new AdGroupUpdateData(appliedChanges, campaign, banners);
                })
                .toList();
    }

    /**
     * Запомнить идентификаторы групп объявлений с измененным Geo
     */
    private void setAdGroupsWithChangedGeoIds(Collection<ModelChanges<AdGroup>> validModelChanges) {
        adGroupsWithChangedGeoIds = validModelChanges
                .stream()
                .filter(mc -> mc.isPropChanged(AdGroup.GEO) && mc.getChangedProp(AdGroup.GEO) != null)
                .map(ModelChanges::getId)
                .collect(toSet());
    }

    /**
     * Подготовка списков регионов к сохранению в базу. Вызывается после валидации и с помощью refineGeoIds
     * причёсывает AdGroup.GEO во всех валидных ModelChanges.
     */
    private void refineGeo(Collection<ModelChanges<AdGroup>> validModelChanges) {
        validModelChanges.stream()
                .filter(mc -> mc.isPropChanged(AdGroup.GEO) || mc.isPropChanged(AdGroup.HYPER_GEO_ID))
                .forEach(mc -> {
                    var adGroupId = mc.getId();
                    var campaignId = campaignIdByAdGroupId.get(adGroupId);

                    Long hyperGeoId = mc.getPropIfChanged(AdGroup.HYPER_GEO_ID);
                    if (hyperGeoId != null) {
                        HyperGeo hyperGeo = hyperGeoById.get(hyperGeoId);
                        List<HyperGeoSegment> hyperGeoSegments = hyperGeo.getHyperGeoSegments();

                        checkState(hyperGeoSegments.size() > 0, "there must be at least one geosegment");
                        HyperGeoSegment hyperGeoSegment = hyperGeoSegments.get(0);

                        // в случае гиперлокального гео на группе, нужно поставить минимальный покрывающий регион
                        // в качестве обычного гео, чтобы БК и модерация не сломались
                        mc.process(hyperGeoSegment.getCoveringGeo(), AdGroup.GEO);
                    } else {
                        var geoIds = mc.getPropIfChanged(AdGroup.GEO);
                        var geoTree = geoTreeProvider.getGeoTree(campaignId);
                        List<Long> geoIdsForSave = clientGeoService.convertForSave(geoIds, geoTree);
                        geoIdsForSave = geoTree.refineGeoIds(geoIdsForSave);
                        mc.process(geoIdsForSave, AdGroup.GEO);
                    }
                });
    }
}
