package ru.yandex.direct.core.entity.adgroup.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.keyword.processing.KeywordStopwordsFixer;
import ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator;
import ru.yandex.direct.libs.keywordutils.helper.ParseKeywordCache;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;

import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeDuplicatesStr;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeStr;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class MinusKeywordPreparingTool {

    private final KeywordStopwordsFixer keywordStopwordsFixer;
    private final SingleKeywordsCache singleKeywordsCache;
    private final ParseKeywordCache parseKeywordCache;

    @Autowired
    public MinusKeywordPreparingTool(
            KeywordStopwordsFixer keywordStopwordsFixer,
            SingleKeywordsCache singleKeywordsCache,
            ParseKeywordCache parseKeywordCache) {
        this.keywordStopwordsFixer = keywordStopwordsFixer;
        this.singleKeywordsCache = singleKeywordsCache;
        this.parseKeywordCache = parseKeywordCache;
    }

    /**
     * Первый этап подготовки минус-фразы к сохранению.
     * <ul>
     * <li>1. В каждой минус-фразе обрезает пробелы по краям.</li>
     * <li>2. Каждую минус-фразу приводит к нижнему регистру.</li>
     * <li>3. Разделяет составные слова на части.</li>
     * <li>4. Принудительно фиксирует стоп-слова во фразе с помощью "!",
     * см. {@link KeywordStopwordsFixer#fixStopwords(List, WordKind)} </li>
     * </ul>
     *
     * @param minusKeywords список валидных минус-фраз
     * @return список предобработанных минус-фраз
     */
    public List<String> preprocess(List<String> minusKeywords) {
        return keywordStopwordsFixer.fixStopwords(mapList(minusKeywords, String::trim), WordKind.FIXED);
    }

    /**
     * Удаляет дубли и сортирует минус-фразы
     * <ul>
     * <li>1. Удаляет "дубликаты" минус-фраз, см. {@link MinusKeywordsDeduplicator#removeDuplicatesStr}</li>
     * <li>2. Сортирует минус-фразы.</li>
     * </ul>
     *
     * @param preprocessedMinusKeywords список валидных минус-фраз,
     *                                  прошедших предвартельную обработку в {@link #preprocess(List)}
     * @return отсортированный список подготовленных к сохранению минус-фраз без "дубликатов"
     */
    public List<String> removeDuplicatesAndSort(List<String> preprocessedMinusKeywords) {
        List<String> uniqueMinusKeywords = removeDuplicatesStr(singleKeywordsCache, preprocessedMinusKeywords,
                t -> parseKeywordCache.parse(t));
        Collections.sort(uniqueMinusKeywords);
        return uniqueMinusKeywords;
    }

    public List<String> removeMinusKeywords(List<String> existingMinusKeywords, List<String> removeMinusKeywords) {
        return removeStr(singleKeywordsCache, existingMinusKeywords, removeMinusKeywords,
                t -> parseKeywordCache.parse(t));
    }

    public List<String> replaceMinusKeywords(List<String> replaceMinusKeywords) {
        List<String> uniqueMinusKeywords = removeDuplicatesStr(singleKeywordsCache, replaceMinusKeywords,
                t -> parseKeywordCache.parse(t));
        Collections.sort(uniqueMinusKeywords);
        return uniqueMinusKeywords;
    }

    /**
     * Полная подготовка списка минус-фраз к сохранению:
     * <ul>
     * <li>1. В каждой минус-фразе обрезает пробелы по краям.</li>
     * <li>2. Каждую минус-фразу приводит к нижнему регистру.</li>
     * <li>3. Принудительно фиксирует стоп-слова во фразе с помощью "!",
     * см. {@link KeywordStopwordsFixer#fixStopwords(List, WordKind)} </li>
     * <li>4. Удаляет "дубликаты" минус-фраз, см. {@link MinusKeywordsDeduplicator#removeDuplicatesStr}</li>
     * <li>5. Сортирует минус-фразы.</li>
     * </ul>
     *
     * @param minusKeywords список валидных минус-фраз
     * @return готовый к сохранению список минус-фраз
     */
    public List<String> fullPrepareForSaving(List<String> minusKeywords) {
        List<String> preprocessedMinusKeywords = preprocess(minusKeywords);
        return removeDuplicatesAndSort(preprocessedMinusKeywords);
    }

    /**
     * Объединяет приватные и библиотечные минус фразы. Удаляет дубликаты и сортирует.
     * Может испоьзоваться, когда надо объединить все минус фразы и передать стороннему сервису.
     */
    public List<String> mergePrivateAndLibrary(List<String> privateMinusKeywords,
                                               Collection<List<String>> libraryMinusKeywords) {
        List<String> allMinusKeywords = new ArrayList<>(privateMinusKeywords);
        libraryMinusKeywords.forEach(allMinusKeywords::addAll);
        return removeDuplicatesAndSort(allMinusKeywords);
    }
}
