package ru.yandex.direct.core.entity.adgroup.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupNewMinusKeywords;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupUpdateOperationParams;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettingsProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProviderFactory;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupUpdateServices;
import ru.yandex.direct.core.entity.adgroup.service.validation.UpdateAdGroupValidationService;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils;
import ru.yandex.direct.core.entity.keyword.processing.KeywordStopwordsFixer;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.minuskeywordspack.service.AddMinusKeywordsPackSubOperationFactory;
import ru.yandex.direct.core.entity.phrase.NormalizedPhrasePair;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.model.ModelChanges.propertyModifier;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция добавления минус-фраз в группу.
 * <p>
 * На этапе предварительной валидации в ModelChanges хранятся только добавляемые минус-фразы.
 * На этом этапе валидируется каждая минус-фраза по отдельности. Это позволяет отдавать ошибки
 * в отдельных минус-фразах по индексам, и они будут точно соответствовать индексам добавляемых
 * минус-фраз в запросе.
 * <p>
 * Если предварительная валидация прошла успешно (что значит, что в отдельных минус-фразах ошибок нет),
 * тогда в методе {@link #onModelChangesValidated(ModelChangesValidatedStep)} добавляемые минус-фразы
 * проходят предварительную обработку (принудительно фиксируются минус-слова...), затем они суммируются
 * с существующими минус-фразами в группе, после чего полный список очищается от дублей и сортируется.
 * После этого в ModelChanges каждой группы уже хранится полный список минус-фраз, готовый для сохранения.
 * <p>
 * На втором этапе валидации минус-фразы групп проверяются только на суммарную длину.
 */
public class UpdateAdGroupMinusKeywordsOperation extends AdGroupsUpdateOperation {

    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final UpdateAdGroupValidationService updateAdGroupValidationService;
    private final KeywordStopwordsFixer keywordStopwordsFixer;
    private final KeywordNormalizer keywordNormalizer;
    private final UpdateMinusKeywordsMode updateMode;
    private final int shard;

    /**
     * Добавляемые (а не суммарные) минус-фразы
     * в нормальной форме для каждой обновляемой группы объявлений.
     */
    private Map<Long, List<String>> normalMinusKeywords;

    /**
     * Реально добавленное число фраз для каждой обновляемой группы объявлений.
     */
    private Map<Long, Integer> actualAddedMinusKeywordsCount;

    /**
     * Суммарная длина всех минус-фраз после добавления
     * для каждой обновляемой группы объявлений (за вычетом спец-символов).
     */
    private Map<Long, Integer> sumMinusKeywordsLength;

    /**
     * Исходные значения фраз для групп
     */
    private Map<Long, List<NormalizedPhrasePair>> origPhrases;

    /**
     * Отображение 'исходная фраза' => 'нормализованная фраза'
     */
    private Map<String, String> origPhraseToNormPhrase;

    UpdateAdGroupMinusKeywordsOperation(
            Applicability applicability,
            List<AdGroupNewMinusKeywords> adGroupNewMinusKeywordsList,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            HyperGeoRepository hyperGeoRepository,
            CryptaSegmentRepository cryptaSegmentRepository,
            MinusKeywordsPackRepository minusKeywordsPackRepository,
            UpdateAdGroupValidationService updateAdGroupValidationService,
            MinusKeywordPreparingTool minusKeywordPreparingTool,
            KeywordStopwordsFixer keywordStopwordsFixer,
            KeywordNormalizer keywordNormalizer,
            GeoTree defaultGeoTree,
            AdGroupGeoTreeProviderFactory geoTreeProviderFactory,
            ClientGeoService clientGeoService,
            AdGroupUpdateServices adGroupUpdateServices,
            AdGroupBsTagsSettingsProvider adGroupBsTagsSettingsProvider,
            AddMinusKeywordsPackSubOperationFactory addMinusKeywordsPackSubOperationFactory,
            AdGroupOperationsHelper adGroupOperationsHelper,
            UpdateMinusKeywordsMode updateMode,
            Long operatorUid,
            ClientId clientId,
            int shard) {
        super(
                applicability,
                createModelChanges(adGroupNewMinusKeywordsList),
                AdGroupUpdateOperationParams.builder()
                        .withModerationMode(ModerationMode.FORCE_MODERATE)
                        .withValidateInterconnections(true)
                        .build(),
                campaignRepository,
                adGroupRepository,
                hyperGeoRepository,
                cryptaSegmentRepository,
                updateAdGroupValidationService,
                defaultGeoTree,
                geoTreeProviderFactory,
                clientGeoService,
                adGroupUpdateServices,
                adGroupOperationsHelper,
                adGroupBsTagsSettingsProvider,
                addMinusKeywordsPackSubOperationFactory,
                MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD,
                operatorUid,
                clientId,
                shard);
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.updateAdGroupValidationService = updateAdGroupValidationService;
        this.keywordStopwordsFixer = keywordStopwordsFixer;
        this.keywordNormalizer = keywordNormalizer;
        this.updateMode = updateMode;
        this.shard = shard;
    }

    private static List<ModelChanges<AdGroup>> createModelChanges(
            List<AdGroupNewMinusKeywords> adGroupNewMinusKeywordsList) {
        Function<AdGroupNewMinusKeywords, ModelChanges<AdGroup>> toModelChanges =
                mk -> ModelChanges.build(mk.getId(), AdGroup.class, AdGroup.MINUS_KEYWORDS, mk.getMinusKeywords());

        return mapList(adGroupNewMinusKeywordsList, toModelChanges);
    }

    /**
     * Возвращает добавляемые (а не суммарные) минус-фразы
     * в нормальной форме с предварительно зафиксированными
     * стоп-словами для каждой обновляемой группы объявлений.
     * Ключами являются id группы объявлений.
     * <p>
     * Метод актуален после выполнения {@link Operation#prepare()}.
     * Мапа содержит значения только для успешно провалидированных
     * {@link ModelChanges}.
     */
    public Map<Long, List<String>> getNormalMinusKeywords() {
        return normalMinusKeywords;
    }

    /**
     * Для всех обновляемых групп возвращает количество минус-фраз,
     * которое будет реально добавлено в группу, оно равно размеру
     * добавляемых минус-фраз минус количество удаленных дубликатов.
     * Ключами являются id группы объявлений.
     * <p>
     * Метод актуален после выполнения {@link Operation#prepare()},
     * если валидация списка ModelChanges прошла успешно.
     * В противном случае будет сгенерировано исключение {@code IllegalStateException}.
     */
    public Map<Long, Integer> getActualAddedMinusKeywordsCount() {
        checkState(actualAddedMinusKeywordsCount != null,
                "prepare() is not called or validation of AppliedChanges failed");
        return actualAddedMinusKeywordsCount;
    }

    /**
     * Для всех обновляемых групп возвращает длину минус-фраз (без учета спец. символов),
     * которые будут выставлены в группах в базе после обновления (имеющиеся + новые - дубликаты).
     * Ключами являются id группы объявлений.
     * <p>
     * Метод актуален после выполнения {@link Operation#prepare()},
     * если валидация списка AppliedChanges прошла успешно.
     * В противном случае будет сгенерировано исключение {@code IllegalStateException}.
     */
    public Map<Long, Integer> getSumMinusKeywordsLength() {
        checkState(sumMinusKeywordsLength != null,
                "prepare() is not called or validation of AppliedChanges failed");
        return sumMinusKeywordsLength;
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<AdGroup> modelChangesValidatedStep) {
        List<ModelChanges<AdGroup>> validModelChanges =
                getModelChangesWithPreValidMinusKeywords(modelChangesValidatedStep);

        preprocessNewMinusKeywords(validModelChanges);
        rememberNewMinusKeywordsInNormalForm(validModelChanges);
        appendNewMinusKeywordsToExisting(validModelChanges);
        removeDuplicatesAndSortMinusKeywords(validModelChanges);

        prepareGeoForSaving(validModelChanges);
        super.onModelChangesValidated(modelChangesValidatedStep);
    }

    /**
     * Проводит превалидацию минус фраз и возвращает model changes только для валидных.
     * Результаты валидации будут использоваться локально только в этой операции, чтобы
     * {@link #preprocessNewMinusKeywords}
     * выполнялся только на валидных фразах. Итоговый результат валидации минус фраз (который будет виден в итоге
     * операции),
     * будет сделан в {@link ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksAddOperation}
     */
    private List<ModelChanges<AdGroup>> getModelChangesWithPreValidMinusKeywords(
            ModelChangesValidatedStep<AdGroup> modelChangesValidatedStep) {
        List<ModelChanges<AdGroup>> validModelChanges =
                new ArrayList<>(modelChangesValidatedStep.getValidModelChanges());
        ValidationResult<List<ModelChanges<AdGroup>>, Defect> localValidationResult =
                ListValidationBuilder.of(validModelChanges, Defect.class)
                        .checkEachBy(updateAdGroupValidationService.minusKeywordsBeforeNormalizationValidator(
                                MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD))
                        .getResult();

        return ValidationResult.getValidItems(localValidationResult);
    }

    private void preprocessNewMinusKeywords(Collection<ModelChanges<AdGroup>> validModelChanges) {
        Function<ModelChanges<AdGroup>, List<NormalizedPhrasePair>> createNormalizedPhrasePair =
                modelChanges ->
                        FunctionalUtils.mapList(modelChanges.getChangedProp(AdGroup.MINUS_KEYWORDS),
                                phrase -> new NormalizedPhrasePair(phrase, null));
        origPhrases = StreamEx.of(validModelChanges.stream())
                .toMap(ModelChanges::getId, createNormalizedPhrasePair);
        validModelChanges.forEach(propertyModifier(AdGroup.MINUS_KEYWORDS, minusKeywordPreparingTool::preprocess));
    }

    /**
     * Запоминает добавляемые минус-фразы, приведя их к нормальной форме после фиксации стоп-слов.
     * Вызывается в тот момент, когда в ModelChanges находятся еще только добавляемые минус-фразы,
     * т.е. до того, когда будут извлечены имеющиеся минус-фразы и объединены с добавляемыми.
     *
     * @param validModelChanges список валидных {@link ModelChanges}.
     */
    private void rememberNewMinusKeywordsInNormalForm(Collection<ModelChanges<AdGroup>> validModelChanges) {
        Function<ModelChanges<AdGroup>, List<String>> getUnquotedNormalizedMinusKeywords = modelChanges -> {
            List<String> minusKeywords = modelChanges.getChangedProp(AdGroup.MINUS_KEYWORDS);
            minusKeywords = keywordStopwordsFixer.unquoteAndFixStopwords(minusKeywords, WordKind.PLUS);
            List<String> normalizedWords = keywordNormalizer.normalizeKeywords(minusKeywords);

            StreamEx.of(origPhrases.get(modelChanges.getId()))
                    .zipWith(normalizedWords.stream())
                    .forEach(el -> el.getKey().setNormalizedPhrase(el.getValue()));

            return normalizedWords;
        };
        normalMinusKeywords = StreamEx.of(validModelChanges)
                .mapToEntry(getUnquotedNormalizedMinusKeywords)
                .mapKeys(ModelChanges::getId)
                .toMap();

        origPhraseToNormPhrase = StreamEx.ofValues(origPhrases)
                .flatMapToEntry(
                        phraseInfoList -> StreamEx.of(phraseInfoList)
                                .toMap(NormalizedPhrasePair::getOrigPhrase, NormalizedPhrasePair::getNormalizedPhrase,
                                        (el1, el2) -> el1)
                )
                .toMap((el1, el2) -> el1);
    }

    private void appendNewMinusKeywordsToExisting(Collection<ModelChanges<AdGroup>> validModelChanges) {
        List<Long> adGroupIds = mapList(validModelChanges, ModelChanges::getId);
        Map<Long, List<String>> adGroupIdToExistingMinusKeywordsMap =
                EntryStream.of(minusKeywordsPackRepository.getMinusKeywordsByAdGroupIds(shard, adGroupIds))
                        .mapValues(MinusKeywordsPack::getMinusKeywords)
                        .toMap();

        BiFunction<Long, List<String>, List<String>> appendMinusKeywords = (id, newMinusKeywords) -> {
            List<String> existingMinusKeywords =
                    adGroupIdToExistingMinusKeywordsMap.getOrDefault(id, emptyList());
            List<String> updatedMinusKeywords = new ArrayList<>(existingMinusKeywords);
            updatedMinusKeywords.addAll(newMinusKeywords);
            return updatedMinusKeywords;
        };

        BiFunction<Long, List<String>, List<String>> removeMinusKeywords = (id, newMinusKeywords) -> {
            List<String> existingMinusKeywords =
                    adGroupIdToExistingMinusKeywordsMap.getOrDefault(id, emptyList());
            return minusKeywordPreparingTool.removeMinusKeywords(existingMinusKeywords, newMinusKeywords);
        };

        BiFunction<Long, List<String>, List<String>> replacedMinusKeywords = (id, newMinusKeywords) ->
                minusKeywordPreparingTool.replaceMinusKeywords(newMinusKeywords);

        switch (updateMode) {
            case ADD:
                validModelChanges.forEach(propertyModifier(AdGroup.MINUS_KEYWORDS, appendMinusKeywords));
                break;
            case REMOVE:
                validModelChanges.forEach(propertyModifier(AdGroup.MINUS_KEYWORDS, removeMinusKeywords));
                break;
            case REPLACE:
                validModelChanges.forEach(propertyModifier(AdGroup.MINUS_KEYWORDS, replacedMinusKeywords));
                break;
            default:
                throw new IllegalStateException("No such value: " + updateMode);
        }
    }

    /**
     * Удаляет дубликаты из полного списка минус-фраз (имеющиеся + добавляемые), затем сортирует его.
     * <p>
     * Помимо удаления дубликатов и сортировки, вычисляет количество реально добавляемых минус-фраз
     * для каждой группы. Эту информацию затем можно получить с помощью метода
     * {@link #getActualAddedMinusKeywordsCount}.
     * <p>
     * К моменту вызова в ModelChanges должны лежать полные списки минус-фраз.
     */
    private void removeDuplicatesAndSortMinusKeywords(Collection<ModelChanges<AdGroup>> validModelChanges) {
        Map<Long, Integer> minusKeywordsCountBeforeDeduplicating = StreamEx.of(validModelChanges)
                .mapToEntry(mc -> mc.getChangedProp(AdGroup.MINUS_KEYWORDS).size())
                .mapKeys(ModelChanges::getId)
                .toMap();

        validModelChanges
                .forEach(propertyModifier(AdGroup.MINUS_KEYWORDS, minusKeywordPreparingTool::removeDuplicatesAndSort));

        Function<ModelChanges<AdGroup>, Integer> calcActualAddedMinusKeywordsCount = modelChanges -> {
            long id = modelChanges.getId();
            int newMinusKeywordsCount = normalMinusKeywords.get(id).size();
            int allBeforeDeduplicatingCount = minusKeywordsCountBeforeDeduplicating.get(id);
            int allAfterDeduplicatingCount = modelChanges.getChangedProp(AdGroup.MINUS_KEYWORDS).size();
            int removedDuplicatesCount = allBeforeDeduplicatingCount - allAfterDeduplicatingCount;
            int actualNewMinusKeywordsCount = newMinusKeywordsCount - removedDuplicatesCount;
            return actualNewMinusKeywordsCount > 0 ? actualNewMinusKeywordsCount : 0;
        };

        actualAddedMinusKeywordsCount = StreamEx.of(validModelChanges)
                .mapToEntry(calcActualAddedMinusKeywordsCount)
                .mapKeys(ModelChanges::getId)
                .toMap();
    }

    @Override
    protected void beforeExecution(ExecutionStep<AdGroup> executionStep) {
        Collection<AppliedChanges<AdGroup>> appliedChangesCollection = executionStep.getAppliedChangesForExecution();
        rememberMinusKeywordsLength(appliedChangesCollection);
        super.beforeExecution(executionStep);
    }

    /**
     * Запоминает суммарную длину минус-фраз (без учета спец. символов),
     * которые будут выставлены в группах в базе после обновления.
     */
    private void rememberMinusKeywordsLength(Collection<AppliedChanges<AdGroup>> validAppliedChanges) {
        Function<AdGroup, Integer> calcLength = adgroup ->
                KeywordProcessingUtils.getLengthWithoutSpecSymbolsAndSpaces(adgroup.getMinusKeywords());
        sumMinusKeywordsLength = StreamEx.of(validAppliedChanges)
                .map(AppliedChanges::getModel)
                .mapToEntry(AdGroup::getId, calcLength)
                .toMap();
    }

    public Map<String, String> getOrigPhraseToNormPhrase() {
        return origPhraseToNormPhrase;
    }
}
