package ru.yandex.direct.core.entity.adgroup.service.bstags;

import java.util.List;

import ru.yandex.direct.core.entity.adgroup.model.PageGroupTagEnum;
import ru.yandex.direct.core.entity.adgroup.model.TargetTagEnum;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsUtils.concatTags;
import static ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsUtils.tagNames;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class AdGroupBsTagsSettings {

    /**
     * Обязательные теги.
     */
    private final List<String> requiredPageGroupTags;
    /**
     * Все разрешенные теги.
     */
    private final List<String> allowedPageGroupTags;
    /**
     * Дефолтные теги. Проставляются в случае, если с фронта ничего не пришло и список обязательных тегов пуст.
     * По умолчанию defaultPageGroupTags == requiredPageGroupTags.
     */
    private final List<String> defaultPageGroupTags;

    /**
     * Target теги аналогичны PageGroup тегам.
     */
    private final List<String> requiredTargetTags;
    private final List<String> allowedTargetTags;
    private final List<String> defaultTargetTags;

    private AdGroupBsTagsSettings(List<String> requiredPageGroupTags,
                                  List<String> allowedPageGroupTags,
                                  List<String> defaultPageGroupTags,
                                  List<String> requiredTargetTags,
                                  List<String> allowedTargetTags,
                                  List<String> defaultTargetTags) {
        this.requiredPageGroupTags = requiredPageGroupTags;
        this.allowedPageGroupTags = allowedPageGroupTags;
        this.defaultPageGroupTags = defaultPageGroupTags;
        this.requiredTargetTags = requiredTargetTags;
        this.allowedTargetTags = allowedTargetTags;
        this.defaultTargetTags = defaultTargetTags;
    }

    public List<String> getRequiredPageGroupTags() {
        return requiredPageGroupTags;
    }

    public List<String> getAllowedPageGroupTags() {
        return allowedPageGroupTags;
    }

    public List<String> getDefaultPageGroupTags() {
        return defaultPageGroupTags;
    }

    public List<String> getRequiredTargetTags() {
        return requiredTargetTags;
    }

    public List<String> getAllowedTargetTags() {
        return allowedTargetTags;
    }

    public List<String> getDefaultTargetTags() {
        return defaultTargetTags;
    }

    public static class Builder {

        private List<String> requiredPageGroupTags = emptyList();
        private List<String> allowedPageGroupTags = emptyList();
        private List<String> defaultPageGroupTags;

        private List<String> requiredTargetTags = emptyList();
        private List<String> allowedTargetTags = emptyList();
        private List<String> defaultTargetTags;

        public Builder withRequiredPageGroupTags(List<String> requiredPageGroupTags) {
            this.requiredPageGroupTags = requiredPageGroupTags;
            return this;
        }

        public Builder withRequiredPageGroupTagsEnum(List<PageGroupTagEnum> requiredPageGroupTags) {
            this.requiredPageGroupTags = tagNames(requiredPageGroupTags);
            return this;
        }

        public Builder withAllowedPageGroupTags(List<String> allowedPageGroupTags) {
            this.allowedPageGroupTags = allowedPageGroupTags;
            return this;
        }

        public Builder withAllowedPageGroupTagsEnum(List<PageGroupTagEnum> allowedPageGroupTags) {
            this.allowedPageGroupTags = tagNames(allowedPageGroupTags);
            return this;
        }

        public Builder withDefaultPageGroupTags(List<String> defaultPageGroupTags) {
            this.defaultPageGroupTags = defaultPageGroupTags;
            return this;
        }

        public Builder withDefaultPageGroupTagsEnum(List<PageGroupTagEnum> defaultPageGroupTags) {
            this.defaultPageGroupTags = tagNames(defaultPageGroupTags);
            return this;
        }

        public Builder withRequiredTargetTags(List<String> requiredTargetTags) {
            this.requiredTargetTags = requiredTargetTags;
            return this;
        }

        public Builder withRequiredTargetTagsEnum(List<TargetTagEnum> requiredTargetTags) {
            this.requiredTargetTags = tagNames(requiredTargetTags);
            return this;
        }

        public Builder withAllowedTargetTags(List<String> allowedTargetTags) {
            this.allowedTargetTags = allowedTargetTags;
            return this;
        }

        public Builder withAllowedTargetTagsEnum(List<TargetTagEnum> allowedTargetTags) {
            this.allowedTargetTags = tagNames(allowedTargetTags);
            return this;
        }

        public Builder withDefaultTargetTags(List<String> defaultTargetTags) {
            this.defaultTargetTags = defaultTargetTags;
            return this;
        }

        public Builder withDefaultTargetTagsEnum(List<TargetTagEnum> defaultTargetTags) {
            this.defaultTargetTags = tagNames(defaultTargetTags);
            return this;
        }

        public AdGroupBsTagsSettings build() {
            return new AdGroupBsTagsSettings(
                    requiredPageGroupTags,
                    concatTags(requiredPageGroupTags, allowedPageGroupTags, defaultPageGroupTags),
                    nvl(defaultPageGroupTags, requiredPageGroupTags),
                    requiredTargetTags,
                    concatTags(requiredTargetTags, allowedTargetTags, defaultTargetTags),
                    nvl(defaultTargetTags, requiredTargetTags)
            );
        }
    }
}
