package ru.yandex.direct.core.entity.adgroup.service.bstags;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PageGroupTagEnum;
import ru.yandex.direct.core.entity.adgroup.model.TargetTagEnum;
import ru.yandex.direct.utils.JsonUtils;

@ParametersAreNonnullByDefault
public class AdGroupBsTagsUtils {

    private AdGroupBsTagsUtils() {
    }

    /**
     * Конкатенирует списки тегов.
     * Если в результате тегов нет - возвращает пустой список.
     */
    static List<String> concatTags(List<String>... tagsLists) {
        return StreamEx.of(tagsLists)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .distinct()
                .toList();
    }

    /**
     * Рассчитывает список тегов для простановки в базу.
     */
    @Nullable
    public static List<String> calcBsTagsForDb(@Nullable List<String> adGroupTags,
                                               @Nullable List<String> requiredTags,
                                               @Nullable List<String> defaultTags) {
        if (adGroupTags == null) {
            return defaultTags;
        }

        var concatenatedTags = concatTags(adGroupTags, requiredTags);
        return concatenatedTags.isEmpty() ? defaultTags : concatenatedTags;
    }

    /**
     * Получение имен тегов по их значениям в {@link PageGroupTagEnum} или {@link TargetTagEnum}.
     */
    public static List<String> tagNames(List<? extends Enum> tags) {
        return Arrays.asList(JsonUtils.fromJson(JsonUtils.toJson(tags), String[].class));
    }

    /**
     * Возвраащет стрим AdGroup -> специфичный тип AdGroup.
     * При этом в стрим попадают уникальные группы по System.identityHashCode.
     * Зачем это надо см. {@link AdGroupBsTagsTypeSupport#getAdGroupsBsTagsSettings}
     */
    public static <T extends AdGroup> EntryStream<AdGroup, T> entryStreamOfAdGroups(List<T> adGroups) {
        return EntryStream.of(adGroups)
                .mapToKey((i, adGroup) -> (AdGroup) adGroup)
                .distinct(entry -> System.identityHashCode(entry.getKey()));
    }


}
