package ru.yandex.direct.core.entity.adgroup.service.bstags;

import java.util.List;
import java.util.Map;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.bsTagNotAllowed;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class AllowedBsTagsValidator implements Validator<AdGroup, Defect> {

    private final Map<AdGroup, AdGroupBsTagsSettings> adGroupsBsTagsSettings;

    private final boolean anyBsTagAllowed;

    public AllowedBsTagsValidator(Map<AdGroup, AdGroupBsTagsSettings> adGroupsBsTagsSettings,
                                  FeatureService featureService, ClientId clientId, Long operatorUid) {
        this.adGroupsBsTagsSettings = adGroupsBsTagsSettings;
        this.anyBsTagAllowed = featureService.isEnabledForClientId(clientId, FeatureName.TARGET_TAGS_ALLOWED) ||
                featureService.isEnabledForUid(operatorUid, singletonList(FeatureName.TARGET_TAGS_ALLOWED));
    }

    @Override
    public ValidationResult<AdGroup, Defect> apply(AdGroup adGroup) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        var adGroupBsTagsSettings = adGroupsBsTagsSettings.get(adGroup);

        vb.item(AdGroup.PAGE_GROUP_TAGS)
                .check(areBsTagsAllowed(adGroupBsTagsSettings.getAllowedPageGroupTags()));
        vb.item(AdGroup.TARGET_TAGS)
                .check(areBsTagsAllowed(adGroupBsTagsSettings.getAllowedTargetTags()));

        return vb.getResult();
    }

    private Constraint<List<String>, Defect> areBsTagsAllowed(List<String> allowedTags) {
        return fromPredicate(tags -> tags.stream().allMatch(tag -> isBsTagAllowed(tag, allowedTags)), bsTagNotAllowed());
    }

    public boolean isBsTagAllowed(String tag, List<String> allowedTags) {
        return anyBsTagAllowed || allowedTags.contains(tag);
    }
}
