package ru.yandex.direct.core.entity.adgroup.service.bstags;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoproductAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PageGroupTagEnum;
import ru.yandex.direct.core.entity.adgroup.model.TargetTagEnum;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsUtils.entryStreamOfAdGroups;
import static ru.yandex.direct.feature.FeatureName.ZERO_SPEED_PAGE_ENABLED_FOR_GEOPRODUCT;

@Component
@ParametersAreNonnullByDefault
class CpmGeoproductAdGroupBsTagsTypeSupport implements AdGroupBsTagsTypeSupport<CpmGeoproductAdGroup> {

    private static final List<PageGroupTagEnum> PAGE_GROUP_TAGS_REQUIRED = emptyList();
    private static final List<PageGroupTagEnum> PAGE_GROUP_TAGS_DEFAULT = List.of(PageGroupTagEnum.APP_METRO_TAG);
    private static final Map<String, List<PageGroupTagEnum>> PAGE_GROUP_TAGS_ALLOWED_BY_FEATURE = Map.of(
            ZERO_SPEED_PAGE_ENABLED_FOR_GEOPRODUCT.getName(), List.of(PageGroupTagEnum.APP_NAVI_TAG));

    private static final List<TargetTagEnum> TARGET_TAGS_REQUIRED = emptyList();
    private static final List<TargetTagEnum> TARGET_TAGS_DEFAULT = List.of(TargetTagEnum.APP_METRO_TAG);
    private static final Map<String, List<TargetTagEnum>> TARGET_TAGS_ALLOWED_BY_FEATURE = Map.of(
            ZERO_SPEED_PAGE_ENABLED_FOR_GEOPRODUCT.getName(), List.of(TargetTagEnum.APP_NAVI_TAG));

    private final FeatureService featureService;

    @Autowired
    public CpmGeoproductAdGroupBsTagsTypeSupport(FeatureService featureService) {
        this.featureService = featureService;
    }

    @Override
    public Class<CpmGeoproductAdGroup> getAdGroupClass() {
        return CpmGeoproductAdGroup.class;
    }

    @Override
    public IdentityHashMap<AdGroup, AdGroupBsTagsSettings> getAdGroupsBsTagsSettings(
            List<CpmGeoproductAdGroup> adGroups, GetAdGroupsBsTagsSettingsParams params) {
        ClientId clientId = params.getClientId();
        Set<String> clientFeatures = featureService.getEnabledForClientId(clientId);

        var pageGroupTagsAllowed = getAllowedTagsByClientFeatures(clientFeatures, PAGE_GROUP_TAGS_ALLOWED_BY_FEATURE);
        var targetTagsAllowed = getAllowedTagsByClientFeatures(clientFeatures, TARGET_TAGS_ALLOWED_BY_FEATURE);

        var bsTagsSettings = new AdGroupBsTagsSettings.Builder()
                .withRequiredPageGroupTagsEnum(PAGE_GROUP_TAGS_REQUIRED)
                .withAllowedPageGroupTagsEnum(pageGroupTagsAllowed)
                .withDefaultPageGroupTagsEnum(PAGE_GROUP_TAGS_DEFAULT)
                .withRequiredTargetTagsEnum(TARGET_TAGS_REQUIRED)
                .withAllowedTargetTagsEnum(targetTagsAllowed)
                .withDefaultTargetTagsEnum(TARGET_TAGS_DEFAULT)
                .build();

        return entryStreamOfAdGroups(adGroups)
                .mapValues(adGroup -> bsTagsSettings)
                .toCustomMap(IdentityHashMap::new);
    }

    private <T> List<T> getAllowedTagsByClientFeatures(Set<String> clientFeatures,
                                                       Map<String, List<T>> allowedTagsByFeature) {
        return EntryStream.of(allowedTagsByFeature)
                .filterKeys(clientFeatures::contains)
                .values()
                .flatMap(StreamEx::of)
                .toList();
    }
}
