package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ContiguousSet;
import com.google.common.collect.DiscreteDomain;
import com.google.common.collect.Range;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.PartiallyApplicableModelOperation;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkComplexAdGroupsConsistency;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.preparePartialModelOperationAndGetValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItemsWithIndex;

@ParametersAreNonnullByDefault
public abstract class AbstractComplexAdGroupOperation<T extends ComplexAdGroup, O extends PartiallyApplicableModelOperation<AdGroup, Long>> implements PartiallyApplicableOperation<Long> {
    protected final Applicability applicability;
    protected final List<T> complexAdGroups;
    protected final List<AdGroup> adGroups;
    protected final ClientId clientId;
    protected final O adGroupsOperation;

    private Map<Integer, AdGroup> validGroupsMap;
    private ValidationResult<List<AdGroup>, Defect> adGroupsResult;
    private MassResult<Long> result;

    private boolean prepared;
    private boolean executed;

    protected AbstractComplexAdGroupOperation(Applicability applicability, List<T> complexAdGroups, ClientId clientId, O operation) {
        checkComplexAdGroupsConsistency(complexAdGroups);

        this.applicability = applicability;
        this.complexAdGroups = complexAdGroups;
        this.adGroups = mapList(complexAdGroups, ComplexAdGroup::getAdGroup);
        this.clientId = clientId;
        this.adGroupsOperation = operation;
    }

    /**
     * Подготовка/валидация отдельных подопераций.
     * <p>
     * Она состоит из двух частей:
     * 1. валидация/подготовка отдельных сущностей с помощью соответствующих
     * низкоуровневых операций.
     * 2. специфическая валидация дерева, реализованная в потомках.
     * <p>
     * Основные правила:
     * 1. Объекты, входящие в группу, могут валидироваться, только если их группа валидна,
     * так как сама их валидация/подготовка зависит от данных группы.
     * 2. Специфическая валидация так же проводится только в том случае,
     * когда группы валидны, это упрощает код.
     * 3. При валидации вложенных в группу сущностей не проверяются права, так как они
     * проверяются на уровне группы, а все вложенные объекты явно привязываются к тем
     * группам, в которые они вложены в запросе, в самой комплексной операции.
     * 4. Взаимная валидация объектов, как между баннерами и сайтлинками, баннерами
     * и визитками, баннерами и группами, проводится в отдельном сервисе специфической валидации дерева.
     * Сюда же входит взаимное соответствие типов.
     * <p>
     * Особенности:
     * 1. количество баннеров на группу валидируется в отдельном сервисе,
     * а количество условий показа валидируется в их низкоуровневых операциях.
     * Возможно, количество баннеров тоже нужно валидировать в низкоуровневой операции.
     * 2. соответствие типов баннеров типу группы валидируется в отдельном сервисе,
     * как и допустимый тип групп, допустимость сайтлинков и визиток.
     * 3. в случае обновления, валидация прав проводится в каждой саб-операции, кроме операции баннеров, то есть
     * по сути дублируется. Стоит подумать над ее отключением в низкоуровневых операциях.
     */
    @Override
    public Optional<MassResult<Long>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        ValidationResult<List<AdGroup>, Defect> adGroupsResult =
                preparePartialModelOperationAndGetValidationResult(adGroupsOperation, adGroups);

        // Невалидность групп может означать невозможность передачи корректных данных группы
        // в нижележащие операции для их валидации/подготовки, поэтому на данный момент
        // сделано по-простому: если хоть одна группа невалидна, дальше не идем.
        // При этом есть теоретические варианты с продолжением валидации для валидных групп.
        //
        // Для поддержки смарт-групп в API разрешаем комплексные операции с неполной применимостью.
        // В таком случае подобъекты будут валидироваться в любом случае. Пользоваться пока что стоит с осторожностью.
        if (!adGroupsResult.hasAnyErrors() || !isFull(applicability)) {
            afterAdGroupsPrepare(adGroupsResult);
        }

        List<AdGroup> validGroups = getValidItems(adGroupsResult);

        if (validGroups.isEmpty() ||
                (validGroups.size() < complexAdGroups.size() && isFull(applicability))) {
            result = MassResult.brokenMassAction(mapList(adGroups, AdGroup::getId), adGroupsResult);
            return Optional.of(result);
        }

        this.adGroupsResult = adGroupsResult;
        this.validGroupsMap = getValidItemsWithIndex(adGroupsResult);

        return Optional.empty();
    }

    protected void afterAdGroupsPrepare(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
    }

    /**
     * Обновление больших групп.
     * <p>
     * Сначала проводится валидация/подготовка: см. javadoc к {@link #prepare()}.
     * <p>
     * Затем проводится непосредственно выполнение: см. javadoc к {@link #apply()}.
     *
     * @return результат, который в случае провала валидации содержит результат
     * валидации списка простых групп, но при этом содержит результаты валидации
     * простых подмоделей, как если бы группа на самом деле их содержала. Таким образом,
     * {@link ComplexAdGroup} используется лишь как контейнер для линковки групп
     * с подобъектами. На уровне баннеров это работает точно так же.
     */
    private MassResult<Long> applyInternal(Set<Integer> elementIndexesToApply) {
        executed = true;

        Map<Integer, AdGroup> validModelsMapToApply = EntryStream.of(validGroupsMap)
                .filterKeys(elementIndexesToApply::contains)
                .toMap();

        // до исполнения операции добавления группы, нельзя исполнять какие-либо связанные модифицирующие запросы
        // потому как операция добавления группы может свалиться с исключением
        // иначе в результате получится, что мы добавили какие то данные для группы, но саму группу не добавили
        MassResult<Long> addAdGroupsResult = adGroupsOperation.apply(validModelsMapToApply.keySet());

        afterAdGroupsApply(addAdGroupsResult);

        checkState(addAdGroupsResult.isSuccessful(), "sub-operation must be successful");
        result = MassResult.successfulMassAction(mapList(addAdGroupsResult.getResult(), Result::getResult),
                adGroupsResult);
        return result;
    }

    @Override
    public MassResult<Long> apply() {
        checkApplyOrCancelPreconditions();
        Set<Integer> allIndexesSet = ContiguousSet.create(
                Range.closedOpen(0, complexAdGroups.size()), DiscreteDomain.integers());
        return applyInternal(allIndexesSet);
    }

    @Override
    public MassResult<Long> apply(Set<Integer> elementIndexesToApply) {
        checkApplyOrCancelPreconditions();
        checkArgument(Sets.difference(elementIndexesToApply, validGroupsMap.keySet()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");
        return applyInternal(elementIndexesToApply);
    }

    protected void afterAdGroupsApply(MassResult<Long> adGroupsResult) {
    }

    @Override
    public MassResult<Long> cancel() {
        checkApplyOrCancelPreconditions();

        result = adGroupsOperation.cancel();
        return result;
    }

    @Override
    public Optional<MassResult<Long>> getResult() {
        return Optional.ofNullable(result);
    }

    @Override
    public Set<Integer> getValidElementIndexes() {
        checkState(prepared, "prepare() must be called before getValidElementIndexes()");

        return validGroupsMap != null ? validGroupsMap.keySet() : emptySet();
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
    }
}
