package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.List;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.type.language.BannerLanguageValidator;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.hrefOrVcardOrTurboOrPermalinkIsSet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.sitelinksCanExistOnlyWithBannerHref;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.validateSitelinkSet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.validateVcard;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_IN_ADGROUP;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstraints.getLimitBannersInGroup;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstraints.isBannerClassCorrespondTo;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxBannersInAdGroup;

@Service
public class AddComplexAdGroupValidationService {
    private final BannerLanguageValidator bannerLanguageValidator;
    private final FeatureService featureService;

    @Autowired
    public AddComplexAdGroupValidationService(BannerLanguageValidator bannerLanguageValidator,
                                              FeatureService featureService) {
        this.bannerLanguageValidator = bannerLanguageValidator;
        this.featureService = featureService;
    }

    public ValidationResult<List<BannerWithSystemFields>, Defect> validateComplexBanners(
            AdGroup adGroup,
            List<ComplexBanner> complexBanners,
            List<BannerWithSystemFields> banners,
            Language campaignLanguage,
            ClientId clientId,
            Long clientRegionId) {
        return validateComplexBanners(
                adGroup, complexBanners, banners, campaignLanguage, clientId, clientRegionId, null
        );
    }

    public ValidationResult<List<BannerWithSystemFields>, Defect> validateComplexBanners(
            AdGroup adGroup,
            List<ComplexBanner> complexBanners,
            List<BannerWithSystemFields> banners,
            Language campaignLanguage,
            ClientId clientId,
            Long clientRegionId,
            @Nullable CampaignType uacCampaignType
    ) {
        ListValidationBuilder<BannerWithSystemFields, Defect> vb = ListValidationBuilder.of(banners);

        vb
                .check(bannersCountIsInBound(uacCampaignType))
                .checkEachBy((index, banner) -> {
                    //noinspection ConstantConditions: в комплексной операции должно проверяться, что список баннеров
                    // не null
                    ComplexBanner complexBanner = complexBanners.get(index);
                    return validateComplexBanner(complexBanner, adGroup, campaignLanguage,
                            clientId, clientRegionId);
                }, When.isValid());

        return vb.getResult();
    }

    public <T extends BannerWithSystemFields> ValidationResult<List<T>, Defect> validateBanners(
            AdGroup adGroup, List<T> banners) {
        ListValidationBuilder<T, Defect> vb = ListValidationBuilder.of(banners);

        vb
                .check(bannersCountIsInBound())
                .checkEachBy(banner -> validateBanner(banner, adGroup));

        return vb.getResult();
    }

    private ValidationResult<BannerWithSystemFields, Defect> validateComplexBanner(
            ComplexBanner complexBanner, AdGroup adGroup,
            Language campaignLanguage, ClientId clientId,
            Long clientRegionId) {

        ModelItemValidationBuilder<BannerWithSystemFields> vb =
                ModelItemValidationBuilder.of(complexBanner.getBanner());

        vb.checkBy(banner -> validateBanner(banner, adGroup));

        Boolean clientHasDesktopTurbolandingFeature =
                featureService.isEnabledForClientId(clientId, FeatureName.DESKTOP_LANDING);
        vb.check(bannerLanguageValidator.newLanguageIsFromGeo(
                adGroup.getGeo(), campaignLanguage, clientId, clientRegionId))
                .check(hrefOrVcardOrTurboOrPermalinkIsSet(complexBanner, clientHasDesktopTurbolandingFeature))
                .check(sitelinksCanExistOnlyWithBannerHref(complexBanner, clientHasDesktopTurbolandingFeature));

        vb.item(complexBanner.getSitelinkSet(), ComplexBanner.SITELINK_SET.name())
                .checkBy(sitelinkSet -> validateSitelinkSet(complexBanner, sitelinkSet));

        vb.item(complexBanner.getVcard(), ComplexBanner.VCARD.name())
                .checkBy(vcard -> validateVcard(complexBanner, vcard));

        return vb.getResult();
    }

    private <T extends BannerWithSystemFields> ValidationResult<T, Defect> validateBanner(T banner, AdGroup adGroup) {
        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(banner);

        vb.check(isBannerClassCorrespondTo(adGroup.getType()), When.isValid());

        return vb.getResult();
    }

    /**
     * Проверяет, что количество баннеров в группе не превышает максимально допустимое количество.
     */
    private static <T extends BannerWithSystemFields> Constraint<List<T>, Defect> bannersCountIsInBound() {
        return Constraint.fromPredicate(banners -> banners.size() <= MAX_BANNERS_IN_ADGROUP,
                maxBannersInAdGroup(MAX_BANNERS_IN_ADGROUP));
    }

    /**
     * Проверяет, что количество баннеров в группе не превышает максимально допустимое количество
     * с учётом особых лимитов UAC.
     */
    @SuppressWarnings("rawtypes")
    private static Constraint<List<BannerWithSystemFields>, Defect> bannersCountIsInBound(
            @Nullable CampaignType uacCampaignType) {
        int limit = getLimitBannersInGroup(false, uacCampaignType != null, uacCampaignType == CampaignType.TEXT);
        return Constraint.fromPredicate(banners -> banners.size() <= limit,
                maxBannersInAdGroup(limit));
    }
}
