package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nonnull;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexMobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.banner.type.language.BannerLanguageValidator;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageRetargetingSubCategory;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupDoesNotContainThisBanner;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupDoesNotContainThisKeyword;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupDoesNotContainThisRelevanceMatch;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.bidModifiersNotAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.eitherKeywordsOrRetargetingsAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.inconsistentGeoWithBannerLanguages;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.keywordsNotAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.notAllowedValue;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.priceSalesDisallowedAudienceSegments;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.priceSalesDisallowedMetrikaSegments;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.priceSalesTooFewTargetings;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.priceSalesTooManyTargetings;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_IN_ADGROUP;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxBannersInAdGroup;
import static ru.yandex.direct.core.entity.banner.type.language.BannerLanguageConverter.convertLanguage;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ComplexAdGroupConstraints {
    private static final Logger logger = LoggerFactory.getLogger(ComplexAdGroupConstraints.class);

    private static final Set<AdGroupType> BID_MODIFIERS_DEMOGRAPHY_ALLOWED_ADGROUP_TYPES =
            Set.of(AdGroupType.CPM_INDOOR, AdGroupType.MOBILE_CONTENT);
    private static final Set<AdGroupType> BID_MODIFIERS_MOBILE_ALLOWED_ADGROUP_TYPES =
            Set.of(AdGroupType.MOBILE_CONTENT);

    private ComplexAdGroupConstraints() {
    }

    public static Constraint<List<Long>, Defect> adGroupGeoMatchesWithUntouchedBannersLanguage(
            BannerLanguageValidator bannerLanguageValidator, QueryrecService queryrecService,
            BannerTextExtractor bannerTextExtractor,
            Language campaignLanguage, List<BannerWithSystemFields> untouchedBanners, ClientId clientId,
            Long clientRegionId) {
        return geo -> {

            if (CollectionUtils.isEmpty(untouchedBanners)) {
                return null;
            }

            Constraint<BannerWithSystemFields, Defect> languageConstraint =
                    bannerLanguageValidator.newLanguageIsFromGeo(geo, campaignLanguage, clientId, clientRegionId);

            for (var untouchedBanner : untouchedBanners) {
                Defect defect = languageConstraint.apply(untouchedBanner);
                if (defect == null) {
                    continue;
                }
                Language bannerLanguage = convertLanguage(untouchedBanner.getLanguage());
                Language language;

                if (bannerLanguage == Language.UNKNOWN) {
                    logger.warn("Banner with bid {} has no language", untouchedBanner.getId());

                    String text =
                            bannerTextExtractor.extractTexts(Collections.singletonList(untouchedBanner)).get(untouchedBanner);
                    Preconditions.checkState(text != null, "extracted banner text cannot be null "
                            + "if language constraint returned error");
                    language = queryrecService.recognize(text, clientId, clientRegionId);
                } else {
                    language = bannerLanguage;
                }

                return inconsistentGeoWithBannerLanguages(language, untouchedBanner.getId());
            }
            return null;
        };
    }

    /**
     * Проверяет, что суммарное количество баннеров после обновления группы
     * будет в допустимых пределах. Добавляемые баннеры отличаются от обновляемых по наличию id.
     *
     * @param adGroupId                        id группы объявлений
     * @param adGroupIdsToExistingBannersCount мапа id группы -> количество существующих баннеров
     * @return констрейнт
     */
    public static <T extends BannerWithSystemFields> Constraint<List<T>, Defect> newBannersCountIsInBound(
            Long adGroupId, Map<Long, Integer> adGroupIdsToExistingBannersCount) {
        // если id группы == null, то проверка не актуальна
        if (adGroupId == null) {
            return banners -> null;
        }

        Predicate<List<T>> predicate = banners -> {
            long newBannersCount = StreamEx.of(banners)
                    .filter(b -> b.getId() == null)
                    .count();
            Integer existingBannersCount = adGroupIdsToExistingBannersCount.getOrDefault(adGroupId, 0);
            return existingBannersCount + newBannersCount <= MAX_BANNERS_IN_ADGROUP;
        };
        return Constraint.fromPredicate(predicate, maxBannersInAdGroup(MAX_BANNERS_IN_ADGROUP));
    }

    public static <T extends BannerWithSystemFields> Constraint<T, Defect> updatedBannerCorrespondsToAdGroup(
            Long adGroupId, Multimap<Long, Long> bannerIdsByAdGroupIds) {
        Predicate<T> predicate = banner -> banner.getId() == null ||
                bannerIdsByAdGroupIds.get(adGroupId).contains(banner.getId());
        return Constraint.fromPredicate(predicate, adGroupDoesNotContainThisBanner());
    }

    /**
     * Проверяет, что ключевая фраза принадлежит группе
     */
    public static Constraint<Keyword, Defect> updatedKeywordCorrespondsToAdGroup(
            Long adGroupId, SetMultimap<Long, Long> keywordIdsByAdGroupIds) {
        Predicate<Keyword> predicate = keyword -> keyword.getId() == null ||
                keywordIdsByAdGroupIds.get(adGroupId).contains(keyword.getId());
        return Constraint.fromPredicate(predicate, adGroupDoesNotContainThisKeyword());
    }

    /**
     * Проверяет, что бесфразный таргетинг принадлежит группе
     */
    public static Constraint<RelevanceMatch, Defect> modifiedRelevanceMatchCorrespondsToAdGroup(
            Long adGroupId, Multimap<Long, Long> relevanceIdsByAdGroupIds) {
        Predicate<Long> relevanceMatchIdOfGroup =
                id -> id == null || relevanceIdsByAdGroupIds.get(adGroupId).contains(id);

        Predicate<RelevanceMatch> predicateForUpdate = relevanceMatches -> StreamEx.of(relevanceMatches)
                .map(RelevanceMatch::getId)
                .filter(Objects::nonNull)
                .allMatch(relevanceMatchIdOfGroup);

        return Constraint.fromPredicate(predicateForUpdate, adGroupDoesNotContainThisRelevanceMatch());
    }


    public static Constraint<AdGroup, Defect> retargetingsOnlyToDemographics(ComplexCpmAdGroup cpmAdGroup) {
        return Constraint.fromPredicate(
                adGroup -> RetargetingConstraints.onlyToDemographics(cpmAdGroup.getRetargetingConditions()),
                notAllowedValue());
    }

    public static Constraint<AdGroup, Defect> eitherKeywordsOrRetargetingsLinked(ComplexCpmAdGroup cpmAdGroup) {
        return Constraint
                .fromPredicate(adGroup -> cpmAdGroup.getKeywords() == null || cpmAdGroup.getTargetInterests() == null,
                        eitherKeywordsOrRetargetingsAllowed());
    }

    public static Constraint<AdGroup, Defect> keywordsNotLinked(ComplexCpmAdGroup cpmAdGroup) {
        return Constraint.fromPredicate(adGroup -> cpmAdGroup.getKeywords() == null, keywordsNotAllowed());
    }

    public static Constraint<AdGroup, Defect> bidModifiersNotLinkedWithoutKeywords(ComplexCpmAdGroup cpmAdGroup) {
        return Constraint.fromPredicate(adGroup -> {
                    boolean keywordsExist = cpmAdGroup.getKeywords() != null;
                    return keywordsExist || !hasAnyBidModifiers(cpmAdGroup);
                },
                bidModifiersNotAllowed());
    }

    @Nonnull
    public static Constraint<AdGroup, Defect> deviceBidModifiersAllowed() {
        return Constraint.fromPredicate(group -> {
            Set<AdGroupType> allowedTypes =
                    ImmutableSet.of(AdGroupType.CPM_BANNER, AdGroupType.CPM_GEOPRODUCT, AdGroupType.CPM_GEO_PIN,
                            AdGroupType.CPM_VIDEO);
            return allowedTypes.contains(group.getType());
        }, bidModifiersNotAllowed());
    }

    @Nonnull
    public static Constraint<AdGroup, Defect> weatherBidModifiersAllowed() {
        return Constraint.fromPredicate(group -> {
                    Set<AdGroupType> allowedTypes =
                            ImmutableSet.of(AdGroupType.CPM_BANNER, AdGroupType.CPM_VIDEO, AdGroupType.CPM_OUTDOOR,
                                    AdGroupType.CPM_AUDIO, AdGroupType.BASE);
                    return allowedTypes.contains(group.getType());
                },
                bidModifiersNotAllowed());
    }

    @Nonnull
    public static Constraint<AdGroup, Defect> demographyBidModifiersAllowed() {
        return Constraint.fromPredicate(group ->
                        BID_MODIFIERS_DEMOGRAPHY_ALLOWED_ADGROUP_TYPES.contains(group.getType()),
                bidModifiersNotAllowed());
    }

    @Nonnull
    public static Constraint<AdGroup, Defect> mobileBidModifiersAllowed() {
        return Constraint.fromPredicate(group ->
                        BID_MODIFIERS_MOBILE_ALLOWED_ADGROUP_TYPES.contains(group.getType()),
                bidModifiersNotAllowed());
    }

    // Проверка ретаргетингов
    // Проверить, что цели из ретаргетинга входят во множество целей из пакета пакета.
    @Nonnull
    public static Constraint<AdGroup, Defect> pricePackageRetargetingsAllowed(ComplexCpmAdGroup cpmAdGroup, PricePackage pricePackage) {
        return Constraint.fromPredicate(
                adGroup -> areRetargetingGoalsAllowedInPricePackage(pricePackage, cpmAdGroup.getRetargetingConditions()),
                notAllowedValue());
    }

    // Проверка лимитов по таргетингам
    @Nonnull
    public static Constraint<AdGroup, Defect> adGroupTargetingsAreInPricePackageLimit(ComplexCpmAdGroup cpmAdGroup, PricePackage pricePackage) {
        return adGroup -> areTargetingGoalTypesAmountInPricePackageLimit(
                        pricePackage,
                        cpmAdGroup.getRetargetingConditions(),
                        adGroup.getContentCategoriesRetargetingConditionRules());
    }

    public static Constraint<AdGroup, Defect> pricePackageAdGroupMetricaSegmentsDisallowed(ComplexCpmAdGroup cpmAdGroup) {
        return Constraint.fromPredicate(
                adGroup -> !hasMetricaSegments(cpmAdGroup.getRetargetingConditions()),
                priceSalesDisallowedMetrikaSegments());
    }

    public static Constraint<AdGroup, Defect> pricePackageAdGroupAudienceSegmentsDisallowed(ComplexCpmAdGroup cpmAdGroup) {
        return Constraint.fromPredicate(
                adGroup -> !hasAudienceSegments(cpmAdGroup.getRetargetingConditions()),
                priceSalesDisallowedAudienceSegments());
    }

    private static boolean hasMetricaSegments(List<? extends RetargetingConditionBase> retargetingConditions) {
        var retargetingGoals = flatMap(retargetingConditions,
                retCond -> flatMap(retCond.getRules(), Rule::getGoals));

        var onlyMetricaGoals = filterList(retargetingGoals,
                goal -> !goal.getType().equals(GoalType.AUDIENCE) && goal.getType().isMetrika());

        return !onlyMetricaGoals.isEmpty();
    }

    private static boolean hasAudienceSegments(List<? extends RetargetingConditionBase> retargetingConditions) {
        var retargetingGoals = flatMap(retargetingConditions,
                retCond -> flatMap(retCond.getRules(), Rule::getGoals));

        var audienceGoals = filterList(retargetingGoals, goal -> goal.getType().equals(GoalType.AUDIENCE));

        return !audienceGoals.isEmpty();
    }

    public static boolean hasAnyBidModifiers(ComplexCpmAdGroup complexCpmAdGroup) {
        ComplexBidModifier complexBidModifier = complexCpmAdGroup.getComplexBidModifier();
        if (complexBidModifier == null) {
            return false;
        }
        return StreamEx.of(ComplexBidModifier.allModelProperties())
                .map(p -> p.getRaw(complexBidModifier))
                .select(BidModifier.class)
                .anyMatch(Objects::nonNull);
    }

    /**
     * Проверить если ли в ComplexCpmAdGroup корректировки на устройства (mobile, desktop)
     */
    public static boolean complexAdGroupHasDeviceBidModifiers(ComplexCpmAdGroup complexCpmAdGroup) {
        ComplexBidModifier complexBidModifier = complexCpmAdGroup.getComplexBidModifier();
        if (complexBidModifier == null) {
            return false;
        }
        return complexBidModifier.getDesktopModifier() != null
                || complexBidModifier.getMobileModifier() != null
                || complexBidModifier.getSmartTVModifier() != null;
    }

    public static boolean complexAdGroupHasModifiersWhereKeywordsRequired(ComplexCpmAdGroup complexCpmAdGroup) {
        ComplexBidModifier complexBidModifier = complexCpmAdGroup.getComplexBidModifier();
        if (complexBidModifier == null) {
            return false;
        }
        AdGroupType adGroupType = complexCpmAdGroup.getAdGroup().getType();

        return StreamEx.of(ComplexBidModifier.allModelProperties())
                .map(p -> p.getRaw(complexBidModifier))
                .select(BidModifier.class)
                .remove(m -> m instanceof BidModifierDesktop || m instanceof BidModifierMobile
                        || m instanceof BidModifierDesktopOnly || m instanceof BidModifierTablet
                        || m instanceof BidModifierSmartTV || m instanceof BidModifierWeather
                        || (m instanceof BidModifierDemographics && BID_MODIFIERS_DEMOGRAPHY_ALLOWED_ADGROUP_TYPES.contains(adGroupType)))
                .anyMatch(Objects::nonNull);
    }

    /**
     * Проверить если ли в ComplexCpmAdGroup корректировки на погоду (weather)
     */
    public static boolean complexAdGroupHasWeatherBidModifiers(ComplexCpmAdGroup complexCpmAdGroup) {
        ComplexBidModifier complexBidModifier = complexCpmAdGroup.getComplexBidModifier();
        return (complexBidModifier != null) && (complexBidModifier.getWeatherModifier() != null);
    }

    /**
     * Проверить если ли в ComplexTextAdGroup корректировки на погоду (weather)
     */
    public static boolean complexAdGroupHasWeatherBidModifiers(ComplexTextAdGroup complexAdGroup) {
        ComplexBidModifier complexBidModifier = complexAdGroup.getComplexBidModifier();
        return (complexBidModifier != null) && (complexBidModifier.getWeatherModifier() != null);
    }

    /**
     * Проверить если ли в ComplexCpmAdGroup корректировки на соцдем (demography)
     */
    public static boolean complexAdGroupHasDemographyBidModifiers(ComplexCpmAdGroup complexCpmAdGroup) {
        ComplexBidModifier complexBidModifier = complexCpmAdGroup.getComplexBidModifier();
        return (complexBidModifier != null) && (complexBidModifier.getDemographyModifier() != null);
    }

    /**
     * Проверить если ли в ComplexMobileContentAdGroup корректировки на соцдем (demography)
     */
    public static boolean complexAdGroupHasDemographyBidModifiers(ComplexMobileContentAdGroup complexMobileContentAdGroup) {
        ComplexBidModifier complexBidModifier = complexMobileContentAdGroup.getComplexBidModifier();
        return (complexBidModifier != null) && (complexBidModifier.getDemographyModifier() != null);
    }

    /**
     * Проверить если ли в ComplexMobileContentAdGroup мобильные корректировки на погоду (mobile)
     */
    public static boolean complexAdGroupHasMobileBidModifiers(ComplexMobileContentAdGroup complexMobileContentAdGroup) {
        ComplexBidModifier complexBidModifier = complexMobileContentAdGroup.getComplexBidModifier();
        return (complexBidModifier != null) && (complexBidModifier.getMobileModifier() != null);
    }

    /**
     * Проверить если ли в ComplexCpmAdGroup ключевые слова
     */
    public static boolean complexAdGroupHasKeywords(ComplexCpmAdGroup complexCpmAdGroup) {
        return complexCpmAdGroup.getKeywords() != null;
    }

    public static boolean areRetargetingGoalsAllowedInPricePackage(
            PricePackage pricePackage,
            List<? extends RetargetingConditionBase> retargetingConditions) {
        if (isEmpty(retargetingConditions)) {
            return true;
        }
        List<Long> packageGoalIds = nvl(
                pricePackage.getTargetingsCustom().getRetargetingCondition().getCryptaSegments(),
                emptyList());

        var adGroupGoalIds = flatMap(retargetingConditions,
                retCond -> flatMap(retCond.getRules(), rule ->
                        filterAndMapList(rule.getGoals(), goal -> goal.getType().isCrypta(), Goal::getId)));
        return packageGoalIds.containsAll(adGroupGoalIds);
    }

    public static Defect areTargetingGoalTypesAmountInPricePackageLimit(
            PricePackage pricePackage,
            List<? extends RetargetingConditionBase> retargetingConditions,
            List<Rule> contendCategoriesRules) {
        Integer lowerLimit = pricePackage.getTargetingsCustom().getRetargetingCondition().getLowerCryptaTypesCount();
        Integer upperLimit = pricePackage.getTargetingsCustom().getRetargetingCondition().getUpperCryptaTypesCount();

        // ноль - значит без ограничений
        if (lowerLimit == 0 && upperLimit == 0) {
            return null;
        }

        var retargetingConditionsGoalTypes = mapList(flatMap(nvl(retargetingConditions, emptyList()),
                retCond -> flatMap(retCond.getRules(), Rule::getGoals)), Goal::getType);

        Map<Boolean, List<GoalType>> retargetingGoalTypes = StreamEx.of(retargetingConditionsGoalTypes)
                .partitioningBy(GoalType::isMetrika);

        var metrikaGoalTypes = retargetingGoalTypes.get(true);

        var cryptaRetargetingConditionsGoals = retargetingGoalTypes.get(false);

        List<PricePackageRetargetingSubCategory> selectedSubCategories = new ArrayList<>();

        // Соцдем (пол/возраст/доход)
        if (cryptaRetargetingConditionsGoals.contains(GoalType.SOCIAL_DEMO)) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.SOCIAL_DEMO);
        }
        // Семейное положение, дети, профессия
        if (cryptaRetargetingConditionsGoals.contains(GoalType.FAMILY)) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.FAMILY);
        }
        // Поведенческие признаки
        if (cryptaRetargetingConditionsGoals.contains(GoalType.BEHAVIORS)) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.BEHAVIORS);
        }
        // Поведенческие признаки
        if (cryptaRetargetingConditionsGoals.contains(GoalType.INTERESTS)) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.INTERESTS);
        }

        int amountAudience = metrikaGoalTypes.contains(GoalType.AUDIENCE) ? 1 : 0;
        int amountMetrika = (metrikaGoalTypes.size() - amountAudience > 0) ? 1 : 0;

        var contendCategoriesGoalTypes = flatMap(nvl(contendCategoriesRules, emptyList()),
                rule -> mapList(rule.getGoals(), goal -> Goal.computeType(goal.getId())));

        if (amountAudience > 0) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.AUDIENCE);
        }
        // Всю метрику, кроме AUDIENCE надо считать одним типом
        if (amountMetrika > 0) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.METRIKA);
        }

        // По идее тут только CONTENT_CATEGORY и CONTENT_GENRE должны быть.
        // Но на всякий случай подстраховываемся на случай внезапного добавления новых типов.
        List<GoalType> genreCategoriesGoalTypes = filterList(contendCategoriesGoalTypes,
                goalType -> goalType.equals(GoalType.CONTENT_CATEGORY) || goalType.equals(GoalType.CONTENT_GENRE));

        //CONTENT_CATEGORY и CONTENT_GENRE надо считать одной группой целей
        if (!genreCategoriesGoalTypes.isEmpty()) {
            selectedSubCategories.add(PricePackageRetargetingSubCategory.GENRE_CATEGORIES);
        }

        int typesAmount = selectedSubCategories.size();

        if (typesAmount <  lowerLimit && lowerLimit != 0) {
            return priceSalesTooFewTargetings(typesAmount, lowerLimit, upperLimit, listToSet(selectedSubCategories));
        }
        if (typesAmount > upperLimit && upperLimit != 0) {
            return priceSalesTooManyTargetings(typesAmount, lowerLimit, upperLimit, listToSet(selectedSubCategories));
        }
        return null;
    }
}
