package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.UnaryOperator;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.logging.log4j.util.TriConsumer;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ComplexAdGroupModelUtils {

    private ComplexAdGroupModelUtils() {
    }

    public static <T extends ComplexAdGroup> void checkComplexAdGroupsConsistency(List<T> complexAdGroups) {
        checkArgument(!isEmpty(complexAdGroups), "должна быть передана минимум одна группа");
        for (int adGroupIndex = 0; adGroupIndex < complexAdGroups.size(); adGroupIndex++) {
            T complexAdGroup = complexAdGroups.get(adGroupIndex);
            checkArgument(complexAdGroup != null, "complexAdGroups[%s] is null", adGroupIndex);
            checkArgument(complexAdGroup.getAdGroup() != null, "complexAdGroups[%s].adGroup is null", adGroupIndex);
        }
    }

    public static <T extends ComplexAdGroup> void checkKeywordsConsistency(
            List<T> complexAdGroups,
            ModelProperty<? super T, List<Keyword>> keywordsProperty
    ) {
        checkModelsInAdGroupsConsistency(complexAdGroups, keywordsProperty,
                (keyword, adGroupIndex, keywordIndex) ->
                        checkArgument(keyword != null,
                                "complexAdGroups[%s].keywords[%s] is null", adGroupIndex, keywordIndex));
    }

    public static <T extends ComplexAdGroup> void checkRelevanceMatchesConsistency(
            List<T> complexAdGroups,
            ModelProperty<? super T, List<RelevanceMatch>> relevanceMatchProperty
    ) {
        checkModelsInAdGroupsConsistency(complexAdGroups, relevanceMatchProperty,
                (relevanceMatch, adGroupIndex, relevanceMatchIndex) ->
                        checkArgument(relevanceMatch != null,
                                "complexAdGroups[%s].relevanceMatches[%s] is null", adGroupIndex, relevanceMatchIndex));
    }

    public static <T extends ComplexAdGroup> void checkOfferRetargetingsConsistency(
            List<T> complexAdGroups,
            ModelProperty<? super T, List<OfferRetargeting>> offerRetargetingProperty
    ) {
        checkModelsInAdGroupsConsistency(complexAdGroups, offerRetargetingProperty,
                (offerRetargeting, adGroupIndex, offerRetargetingIndex) ->
                        checkArgument(offerRetargeting != null,
                                "complexAdGroups[%s].offerRetargetings[%s] is null", adGroupIndex,
                                offerRetargetingIndex));
    }

    public static <T extends ComplexAdGroup> void checkTargetInterestsConsistency(
            List<T> complexAdGroups,
            ModelProperty<? super T, List<TargetInterest>> targetInterestProperty
    ) {
        checkModelsInAdGroupsConsistency(complexAdGroups, targetInterestProperty,
                (targetInterest, adGroupIndex, targetInterestIndex) ->
                        checkArgument(targetInterest != null,
                                "complexAdGroups[%s].targetInterests[%s] is null", adGroupIndex, targetInterestIndex));
    }

    public static <T extends ComplexAdGroup> void checkComplexBannersConsistency(
            List<T> complexAdGroups,
            ModelProperty<? super T, List<ComplexBanner>> complexBannerProperty
    ) {
        checkModelsInAdGroupsConsistency(complexAdGroups, complexBannerProperty,
                (complexBanner, adGroupIndex, bannerIndex) -> {
                    checkArgument(complexBanner != null,
                            "complexAdGroups[%s].complexBanners[%s] is null", adGroupIndex, bannerIndex);
                    checkArgument(complexBanner.getBanner() != null,
                            "complexAdGroups[%s].complexBanners[%s].banner is null", adGroupIndex, bannerIndex);
                }
        );
    }

    private static <T extends ComplexAdGroup, M> void checkModelsInAdGroupsConsistency(
            List<T> complexAdGroups,
            ModelProperty<? super T, List<M>> modelProperty,
            TriConsumer<M, Integer, Integer> checker
    ) {
        for (int adGroupIndex = 0; adGroupIndex < complexAdGroups.size(); adGroupIndex++) {
            T complexAdGroup = complexAdGroups.get(adGroupIndex);

            List<M> models = modelProperty.get(complexAdGroup);
            if (!isEmpty(models)) {
                for (int modelIndex = 0; modelIndex < models.size(); modelIndex++) {
                    M model = models.get(modelIndex);
                    checker.accept(model, adGroupIndex, modelIndex);
                }
            }
        }
    }

    public static ComplexCpmAdGroup cloneComplexCpmAdGroup(ComplexCpmAdGroup complexCpmAdGroup, Long campaignId) {
        AdGroup copiedAdGroup = cloneAdGroupObject(complexCpmAdGroup, campaignId);

        if (!complexCpmAdGroup.getAdGroup().getCampaignId().equals(campaignId)) {
            copiedAdGroup.setTags(null);
        }

        UnaryOperator<BannerWithSystemFields> cloneCpmBanner =
                cpmBanner -> ifNotNull(cloneObject(cpmBanner), t -> t.withCampaignId(campaignId));

        return new ComplexCpmAdGroup()
                .withAdGroup(copiedAdGroup)
                .withKeywords(mapList(complexCpmAdGroup.getKeywords(), cloneKeywordFunc(campaignId)))
                .withTargetInterests(
                        mapList(complexCpmAdGroup.getTargetInterests(), cloneTargetInterestFunc(campaignId)))
                .withComplexBidModifier(
                        cloneComplexBidModifierWithoutIds(complexCpmAdGroup.getComplexBidModifier(), campaignId))
                .withBanners(mapList(complexCpmAdGroup.getBanners(), cloneCpmBanner))
                .withRetargetingConditions(
                        mapList(complexCpmAdGroup.getRetargetingConditions(), ComplexAdGroupModelUtils::cloneObject))
                .withUsersSegments(
                        mapList(complexCpmAdGroup.getUsersSegments(), ComplexAdGroupModelUtils::cloneObject));
    }

    /**
     * Клонируем группу с подобъектами, с учётом ссылок или клонирования новых зависимых объектов
     * (оставляем старые сайтлинки, визитки. Копируем новые баннеры, изображения.)
     */
    public static ComplexTextAdGroup cloneComplexAdGroupInCampaignOnOversize(ComplexTextAdGroup complexAdGroup,
                                                                             boolean isOperatorInternal) {
        return cloneComplexAdGroup(complexAdGroup, isOperatorInternal, complexAdGroup.getAdGroup().getCampaignId());
    }

    /**
     * Клонируем группу с подобъектами, с учётом ссылок или клонирования новых зависимых объектов
     * Оставляем старые сайтлинки. Копируем новые баннеры, изображения.
     * Если клонируем в другую кампанию, то визитки тоже клонируем, а теги группы нет
     */
    public static ComplexTextAdGroup cloneComplexAdGroup(ComplexTextAdGroup complexAdGroup, boolean isOperatorInternal,
                                                         Long campaignId) {
        UnaryOperator<RelevanceMatch> cloneRelevanceMatch =
                relevanceMatch -> ifNotNull(cloneObject(relevanceMatch), r -> r.withCampaignId(campaignId));
        UnaryOperator<OfferRetargeting> cloneOfferRetargeting =
                offerRetargeting -> ifNotNull(cloneObject(offerRetargeting), r -> r.withCampaignId(campaignId));

        AdGroup copiedAdGroup = cloneAdGroupObject(complexAdGroup, campaignId);
        if (!complexAdGroup.getAdGroup().getCampaignId().equals(campaignId)) {
            copiedAdGroup.setTags(null);
        }

        List<TargetInterest> clonedTargetInterestList =
                mapList(complexAdGroup.getTargetInterests(), cloneTargetInterestFunc(campaignId));

        RetargetingConditionBase clonedRetargetingCondition =
                ifNotNull(complexAdGroup.getRetargetingCondition(), ComplexAdGroupModelUtils::cloneObject);

        if (clonedRetargetingCondition != null && clonedTargetInterestList != null) {
            StreamEx.of(clonedTargetInterestList)
                    .filter(interest -> Objects.equals(interest.getRetargetingConditionId(),
                            clonedRetargetingCondition.getId()))
                    .findFirst()
                    .ifPresent(interest -> {
                        interest.withId(null)
                                .withAdGroupId(0L)
                                .withRetargetingConditionId(null);
                        clonedRetargetingCondition.withId(null).withTargetInterest(interest);
                    });
        }

        return new ComplexTextAdGroup()
                .withAdGroup(copiedAdGroup)
                .withKeywords(mapList(complexAdGroup.getKeywords(), cloneKeywordFunc(campaignId)))
                .withComplexBanners(cloneComplexBannersWithOrder(complexAdGroup.getComplexBanners(),
                        isOperatorInternal, campaignId))
                .withComplexBidModifier(
                        cloneComplexBidModifierWithoutIds(complexAdGroup.getComplexBidModifier(), campaignId))
                .withRelevanceMatches(mapList(complexAdGroup.getRelevanceMatches(), cloneRelevanceMatch))
                .withOfferRetargetings(mapList(complexAdGroup.getOfferRetargetings(), cloneOfferRetargeting))
                .withRetargetingCondition(clonedRetargetingCondition)
                .withTargetInterests(clonedTargetInterestList);
    }

    private static AdGroup cloneAdGroupObject(ComplexAdGroup complexAdGroup, Long campaignId) {
        return cloneObject(complexAdGroup.getAdGroup())
                .withCampaignId(campaignId)
                // признак "мало показов" импортируется извне, инкрементально (только при изменении состояния)
                // поэтому затираем значение, чтобы копия группы с "мало показов" не зависала в таком состоянии
                // когда с показами у нее все хорошо и признака во внешней системе нет
                .withBsRarelyLoaded(false)
                .withMinusKeywordsId(null);
    }

    private static UnaryOperator<TargetInterest> cloneTargetInterestFunc(Long campaignId) {
        return targetInterest -> ifNotNull(cloneObject(targetInterest), t -> t.withCampaignId(campaignId));
    }

    private static UnaryOperator<Keyword> cloneKeywordFunc(Long campaignId) {
        return keyword -> ifNotNull(cloneObject(keyword), k -> k.withCampaignId(campaignId));
    }

    /**
     * клонирует баннеры с сортировкой по ID, чтобы порядок баннеров соответствовал порядку их создания
     */
    @Nullable
    private static List<ComplexBanner> cloneComplexBannersWithOrder(@Nullable List<ComplexBanner> complexBanners,
                                                                    boolean isOperatorInternal, Long campaignId) {
        if (complexBanners == null) {
            return null;
        }

        UnaryOperator<ComplexBanner> cloneComplexBanner = complexBanner -> {
            ComplexBanner clonedComplexBanner = new ComplexBanner()
                    .withBanner(cloneObject(complexBanner.getBanner()))
                    .withCreative(cloneObject(complexBanner.getCreative()));

            if (complexBanner.getVcard() != null && !complexBanner.getVcard().getCampaignId().equals(campaignId)) {
                //noinspection ConstantConditions
                Vcard clonedVcard = cloneObject(complexBanner.getVcard()).withCampaignId(campaignId);
                clonedComplexBanner.setVcard(clonedVcard);
            }

            if (clonedComplexBanner.getBanner() != null) {
                clonedComplexBanner.getBanner().setCampaignId(campaignId);
            }

            // https://st.yandex-team.ru/DIRECT-82381
            if (!isOperatorInternal && clonedComplexBanner.getBanner() instanceof BannerWithHref) {
                ((BannerWithHref) clonedComplexBanner.getBanner()).withDomain(null);
            }

            return clonedComplexBanner;
        };

        return StreamEx.of(complexBanners.stream())
                .map(cloneComplexBanner)
                .sortedByLong(complexBanner -> complexBanner.getBanner().getId())
                .toList();
    }

    private static ComplexBidModifier cloneComplexBidModifierWithoutIds(ComplexBidModifier source, Long campaignId) {
        if (source == null) {
            return null;
        }
        ComplexBidModifier result = new ComplexBidModifier()
                // geo не копируем, операция копирования группы ограничивает.
                .withRetargetingModifier(cloneObject(source.getRetargetingModifier()))
                .withDemographyModifier(cloneObject(source.getDemographyModifier()))
                .withWeatherModifier(cloneObject(source.getWeatherModifier()))
                .withPerformanceTgoModifier(cloneObject(source.getPerformanceTgoModifier()))
                .withVideoModifier(cloneObject(source.getVideoModifier()))
                .withMobileModifier(cloneObject(source.getMobileModifier()))
                .withTabletModifier(cloneObject(source.getTabletModifier()))
                .withSmartTVModifier(cloneObject(source.getSmartTVModifier()))
                .withDesktopModifier(cloneObject(source.getDesktopModifier()))
                .withDesktopOnlyModifier(cloneObject(source.getDesktopOnlyModifier()))
                .withAbSegmentModifier(cloneObject(source.getAbSegmentModifier()))
                .withBannerTypeModifier(cloneObject(source.getBannerTypeModifier()))
                .withInventoryModifier(cloneObject(source.getInventoryModifier()))
                .withExpressionModifiers(mapList(
                        source.getExpressionModifiers(),
                        ComplexAdGroupModelUtils::cloneExpressionModifier
                ))
                .withTrafaretPositionModifier(cloneObject(source.getTrafaretPositionModifier()))
                .withRetargetingFilterModifier(cloneObject(source.getRetargetingFilterModifier()));

        List<BidModifier> bidModifiers = new ArrayList<>();
        if (result.getRetargetingModifier() != null) {
            BidModifierRetargeting modifier = result.getRetargetingModifier();
            result.getRetargetingModifier().setRetargetingAdjustments(
                    mapList(modifier.getRetargetingAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }
        if (result.getDemographyModifier() != null) {
            BidModifierDemographics modifier = result.getDemographyModifier();
            modifier.setDemographicsAdjustments(
                    mapList(modifier.getDemographicsAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }
        if (result.getWeatherModifier() != null) {
            BidModifierWeather modifier = result.getWeatherModifier();
            modifier.setWeatherAdjustments(
                    mapList(modifier.getWeatherAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }
        if (result.getPerformanceTgoModifier() != null) {
            BidModifierPerformanceTgo modifier = result.getPerformanceTgoModifier();
            modifier.setPerformanceTgoAdjustment(cloneObject(modifier.getPerformanceTgoAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getVideoModifier() != null) {
            BidModifierVideo modifier = result.getVideoModifier();
            modifier.setVideoAdjustment(cloneObject(modifier.getVideoAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getMobileModifier() != null) {
            BidModifierMobile modifier = result.getMobileModifier();
            modifier.setMobileAdjustment(cloneObject(modifier.getMobileAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getDesktopModifier() != null) {
            BidModifierDesktop modifier = result.getDesktopModifier();
            modifier.setDesktopAdjustment(cloneObject(modifier.getDesktopAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getTabletModifier() != null) {
            BidModifierTablet modifier = result.getTabletModifier();
            modifier.setTabletAdjustment(cloneObject(modifier.getTabletAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getDesktopOnlyModifier() != null) {
            BidModifierDesktopOnly modifier = result.getDesktopOnlyModifier();
            modifier.setDesktopOnlyAdjustment(cloneObject(modifier.getDesktopOnlyAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getSmartTVModifier() != null) {
            BidModifierSmartTV modifier = result.getSmartTVModifier();
            modifier.setSmartTVAdjustment(cloneObject(modifier.getSmartTVAdjustment()));
            bidModifiers.add(modifier);
        }
        if (result.getAbSegmentModifier() != null) {
            BidModifierABSegment modifier = result.getAbSegmentModifier();
            modifier.setAbSegmentAdjustments(
                    mapList(modifier.getAbSegmentAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }

        if (result.getBannerTypeModifier() != null) {
            BidModifierBannerType modifier = result.getBannerTypeModifier();
            modifier.setBannerTypeAdjustments(
                    mapList(modifier.getBannerTypeAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }

        if (result.getInventoryModifier() != null) {
            BidModifierInventory modifier = result.getInventoryModifier();
            modifier.setInventoryAdjustments(
                    mapList(modifier.getInventoryAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }

        if (result.getExpressionModifiers() != null && !result.getExpressionModifiers().isEmpty()) {
            bidModifiers.addAll(result.getExpressionModifiers());
        }

        if (result.getTrafaretPositionModifier() != null) {
            BidModifierTrafaretPosition modifier = result.getTrafaretPositionModifier();
            modifier.setTrafaretPositionAdjustments(
                    mapList(modifier.getTrafaretPositionAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }
        if (result.getRetargetingFilterModifier() != null) {
            BidModifierRetargetingFilter modifier = result.getRetargetingFilterModifier();
            modifier.setRetargetingAdjustments(
                    mapList(modifier.getRetargetingAdjustments(), ComplexAdGroupModelUtils::cloneObject));
            bidModifiers.add(modifier);
        }

        //set common params
        bidModifiers.forEach(bidModifier -> bidModifier
                .withId(null)
                .withCampaignId(campaignId));

        return result;
    }

    private static BidModifierExpression cloneExpressionModifier(BidModifierExpression bidmod) {
        BidModifierExpression result = cloneObject(bidmod);
        result.setExpressionAdjustments(
                mapList(bidmod.getExpressionAdjustments(), ComplexAdGroupModelUtils::cloneExpressionAdjustment)
        );
        return result;
    }

    private static BidModifierExpressionAdjustment cloneExpressionAdjustment(BidModifierExpressionAdjustment bidmoda) {
        BidModifierExpressionAdjustment result = cloneObject(bidmoda);
        result.setCondition(mapList(
                result.getCondition(),
                l -> mapList(l, ComplexAdGroupModelUtils::cloneObject)
        ));
        return result;
    }

    @SuppressWarnings("unchecked")
    @Nullable
    private static <T> T cloneObject(@Nullable T object) {
        if (object == null) {
            return null;
        }

        try {
            return (T) BeanUtils.cloneBean(object);
        } catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            throw new IllegalStateException("cannot clone bean object", e);
        }
    }
}
