package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.ComplexBannerService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting;
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.bidmodifiers.container.ComplexBidModifierConverter.convertToComplexBidModifier;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class ComplexAdGroupService {

    private final ShardHelper shardHelper;
    private final RetargetingService retargetingService;
    private final BidModifierService bidModifierService;
    private final ComplexBannerService complexBannerService;
    private final AdGroupRepository adGroupRepository;
    private final KeywordRepository keywordRepository;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final OfferRetargetingRepository offerRetargetingRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final UsersSegmentRepository usersSegmentRepository;

    public ComplexAdGroupService(
            ShardHelper shardHelper,
            RetargetingService retargetingService,
            BidModifierService bidModifierService,
            ComplexBannerService complexBannerService,
            AdGroupRepository adGroupRepository,
            KeywordRepository keywordRepository,
            RelevanceMatchRepository relevanceMatchRepository,
            OfferRetargetingRepository offerRetargetingRepository,
            BannerTypedRepository bannerTypedRepository,
            RetargetingConditionRepository retargetingConditionRepository,
            UsersSegmentRepository usersSegmentRepository
    ) {
        this.shardHelper = shardHelper;
        this.retargetingService = retargetingService;
        this.bidModifierService = bidModifierService;
        this.complexBannerService = complexBannerService;
        this.adGroupRepository = adGroupRepository;
        this.keywordRepository = keywordRepository;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.offerRetargetingRepository = offerRetargetingRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.usersSegmentRepository = usersSegmentRepository;
    }

    public List<ComplexTextAdGroup> getComplexAdGroups(long operatorUid, UidAndClientId uidAndClientId,
                                                       Collection<Long> adGroupIds) {
        return getComplexAdGroups(operatorUid, uidAndClientId, adGroupIds, ComplexAdGroupFetchParams.fetchAll());
    }

    public List<ComplexTextAdGroup> getComplexAdGroupsWithoutKeywords(long operatorUid, UidAndClientId uidAndClientId,
                                                                      Collection<Long> adGroupIds) {
        return getComplexAdGroups(operatorUid, uidAndClientId, adGroupIds,
                ComplexAdGroupFetchParams.fetchAll().withFetchKeywords(false));
    }

    public List<ComplexTextAdGroup> getComplexAdGroups(long operatorUid, UidAndClientId uidAndClientId,
                                                       Collection<Long> adGroupIds,
                                                       ComplexAdGroupFetchParams fetchParams) {
        ClientId clientId = uidAndClientId.getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);

        Map<Long, List<Keyword>> keywordsByAdGroupIds = fetchParams.isFetchKeywords()
                ? keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds)
                : Collections.emptyMap();

        Map<Long, RelevanceMatch> relevanceMatchMap = fetchParams.isFetchRelevanceMatches()
                ? relevanceMatchRepository.getRelevanceMatchesByAdGroupIds(shard, clientId, adGroupIds)
                : Collections.emptyMap();

        Map<Long, OfferRetargeting> offerRetargetingMap = fetchParams.isFetchOfferRetargetings()
                ? offerRetargetingRepository.getOfferRetargetingsByAdGroupIds(shard, clientId, adGroupIds)
                : Collections.emptyMap();

        Map<Long, List<BidModifier>> bidModifiersByAdGroupIdMap = fetchParams.isFetchBidModifiers()
                ? getBidModifiersMap(operatorUid, clientId, adGroupIds)
                : Collections.emptyMap();

        List<ComplexBanner> complexBanners =
                complexBannerService.getComplexBannersByAdGroupIds(clientId, uidAndClientId.getUid(), adGroupIds);
        Map<Long, List<ComplexBanner>> complexBannersByAdGroupId = complexBanners.stream()
                .collect(groupingBy(e -> e.getBanner().getAdGroupId(), mapping(e -> e, toList())));

        Map<Long, List<TargetInterest>> targetInterestsByAdGroupId =
                fetchParams.isFetchTargetInterests() || fetchParams.isFetchRetargetingConditions()
                        ? getTargetInterestsMap(shard, clientId, adGroupIds)
                        : Collections.emptyMap();
        Map<Long, List<Long>> retCondIdToAdGroupIdsMap = EntryStream.of(targetInterestsByAdGroupId)
                .mapValues(targetInterests -> mapList(targetInterests, TargetInterest::getRetargetingConditionId))
                .flatMapValues(Collection::stream)
                .invert()
                .grouping();
        // ожидается не больше одного условия с интересами на одну группу
        // и не более одной группы на условие с интересами
        List<RetargetingCondition> retargetingConditions = fetchParams.isFetchRetargetingConditions()
                ? retargetingConditionRepository.getFromRetargetingConditionsTable(shard, clientId,
                        retCondIdToAdGroupIdsMap.keySet(), adGroupIds, null,
                        singleton(RetargetingConditionsRetargetingConditionsType.interests),
                        LimitOffset.limited(adGroupIds.size()))
                : Collections.emptyList();
        Map<Long, RetargetingConditionBase> retargetingConditionsMap = retargetingConditions.stream()
                .map(RetargetingConditionBase.class::cast)
                .collect(toMap(rc -> retCondIdToAdGroupIdsMap.get(rc.getId()).get(0), Function.identity()));

        List<ComplexTextAdGroup> complexAdGroups = new ArrayList<>();
        for (AdGroup adGroup : adGroups) {
            Long adGroupId = adGroup.getId();

            ComplexTextAdGroup complexAdGroup = new ComplexTextAdGroup()
                    .withAdGroup(adGroup)
                    .withKeywords(keywordsByAdGroupIds.get(adGroupId))
                    .withTargetInterests(targetInterestsByAdGroupId.get(adGroupId))
                    .withRetargetingCondition(retargetingConditionsMap.get(adGroupId))
                    .withComplexBidModifier(convertToComplexBidModifier(bidModifiersByAdGroupIdMap.get(adGroupId)))
                    .withComplexBanners(complexBannersByAdGroupId.get(adGroupId))
                    .withRelevanceMatches(relevanceMatchMap.containsKey(adGroupId)
                            ? Collections.singletonList(relevanceMatchMap.get(adGroupId))
                            : null)
                    .withOfferRetargetings(offerRetargetingMap.containsKey(adGroupId)
                            ? Collections.singletonList(offerRetargetingMap.get(adGroupId))
                            : null);

            complexAdGroups.add(complexAdGroup);
        }

        return complexAdGroups;
    }

    public List<ComplexCpmAdGroup> getComplexCpmAdGroups(
            Long operatorUid, UidAndClientId uidAndClientId, List<Long> adGroupIds) {

        var clientId = uidAndClientId.getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);

        Map<Long, List<Keyword>> keywordsByAdGroupIds =
                keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds);
        Map<Long, List<TargetInterest>> targetInterestsByAdGroupId = getTargetInterestsMap(shard, clientId, adGroupIds);
        Map<Long, List<BidModifier>> bidModifiersByAdGroupIdMap = getBidModifiersMap(operatorUid, clientId, adGroupIds);
        Map<Long, List<BannerWithSystemFields>> bannersByAdGroupId =
                StreamEx.of(bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds))
                        .map(b -> (BannerWithSystemFields) b)
                        .collect(groupingBy(BannerWithAdGroupId::getAdGroupId));
        Map<Long, List<RetargetingCondition>> retargetingConditionsMap =
                retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, adGroupIds);
        Map<Long, List<UsersSegment>> usersSegments = usersSegmentRepository.getSegmentsAsMap(shard, adGroupIds);

        List<ComplexCpmAdGroup> complexCpmAdGroups = new ArrayList<>();
        for (var adGroup : adGroups) {

            var adGroupId = adGroup.getId();

            List<RetargetingConditionBase> retargetingConditions =
                    Optional.ofNullable(retargetingConditionsMap.get(adGroupId)).orElse(emptyList())
                            .stream()
                            .map(c -> (RetargetingConditionBase) c)
                            .collect(toList());

            var complexCpmAdGroup = new ComplexCpmAdGroup()
                    .withAdGroup(adGroup)
                    .withKeywords(keywordsByAdGroupIds.get(adGroupId))
                    .withTargetInterests(targetInterestsByAdGroupId.get(adGroupId))
                    .withComplexBidModifier(convertToComplexBidModifier(bidModifiersByAdGroupIdMap.get(adGroupId)))
                    .withBanners(mapList(bannersByAdGroupId.get(adGroupId), b -> {
                        if (b instanceof BannerWithHref) {
                            ((BannerWithHref) b).withDomain(null);
                        }
                        return b;
                    }))
                    .withRetargetingConditions(retargetingConditions)
                    .withUsersSegments(usersSegments.get(adGroupId));

            complexCpmAdGroups.add(complexCpmAdGroup);
        }

        return complexCpmAdGroups;
    }

    private Map<Long, List<BidModifier>> getBidModifiersMap(long operatorUid, ClientId clientId,
                                                            Collection<Long> adGroupIds) {
        Set<Long> adGroupIdsSet = new HashSet<>(adGroupIds);
        List<BidModifier> bidModifiers = bidModifierService.getByAdGroupIds(clientId, adGroupIdsSet, emptySet(),
                Constants.ALL_TYPES, singleton(BidModifierLevel.ADGROUP), operatorUid);
        return bidModifiers.stream()
                .collect(groupingBy(BidModifier::getAdGroupId, mapping(e -> e, toList())));
    }

    private Map<Long, List<TargetInterest>> getTargetInterestsMap(
            int shard, ClientId clientId, Collection<Long> adGroupIds) {
        List<TargetInterest> targetInterests =
                retargetingService.getTargetInterestsWithInterestByAdGroupIds(adGroupIds, clientId, shard);
        return targetInterests.stream()
                .collect(groupingBy(TargetInterest::getAdGroupId));
    }

    public static class ComplexAdGroupFetchParams {
        private boolean fetchKeywords = true;
        private boolean fetchRelevanceMatches = true;
        private boolean fetchOfferRetargetings = true;
        private boolean fetchTargetInterests = true;
        private boolean fetchRetargetingConditions = true;
        private boolean fetchBidModifiers = true;

        public static ComplexAdGroupFetchParams fetchAll() {
            return new ComplexAdGroupFetchParams();
        }

        public boolean isFetchKeywords() {
            return fetchKeywords;
        }

        public ComplexAdGroupFetchParams withFetchKeywords(boolean fetchKeywords) {
            this.fetchKeywords = fetchKeywords;
            return this;
        }

        public boolean isFetchRelevanceMatches() {
            return fetchRelevanceMatches;
        }

        public ComplexAdGroupFetchParams withFetchRelevanceMatches(boolean fetchRelevanceMatches) {
            this.fetchRelevanceMatches = fetchRelevanceMatches;
            return this;
        }

        public boolean isFetchOfferRetargetings() {
            return fetchOfferRetargetings;
        }

        public ComplexAdGroupFetchParams withFetchOfferRetargetings(boolean fetchOfferRetargetings) {
            this.fetchOfferRetargetings = fetchOfferRetargetings;
            return this;
        }

        public boolean isFetchTargetInterests() {
            return fetchTargetInterests;
        }

        public ComplexAdGroupFetchParams withFetchTargetInterests(boolean fetchTargetInterests) {
            this.fetchTargetInterests = fetchTargetInterests;
            return this;
        }

        public boolean isFetchRetargetingConditions() {
            return fetchRetargetingConditions;
        }

        public ComplexAdGroupFetchParams withFetchRetargetingConditions(boolean fetchRetargetingConditions) {
            this.fetchRetargetingConditions = fetchRetargetingConditions;
            return this;
        }

        public boolean isFetchBidModifiers() {
            return fetchBidModifiers;
        }

        public ComplexAdGroupFetchParams withFetchBidModifiers(boolean fetchBidModifiers) {
            this.fetchBidModifiers = fetchBidModifiers;
            return this;
        }
    }
}
