package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmAudioAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoPinAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoproductAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmIndoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmOutdoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmYndxFrontpageAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class ComplexAdGroupUpdateConverter {

    private ComplexAdGroupUpdateConverter() {
        // no instantiation
    }

    public static List<ModelChanges<AdGroup>> adGroupsToModelChanges(List<AdGroup> adGroups) {
        return mapList(adGroups, ComplexAdGroupUpdateConverter::adGroupToModelChanges);
    }

    public static ModelChanges<AdGroup> adGroupToModelChanges(AdGroup adGroup) {
        checkArgument(adGroup.getType() != null, "adGroup type must be non-null");

        // TODO появился класс AdGroupTypesMapper, который создает пустой ModelChanges с нужным типом
        // возможно стоит заиспользовать его здесь
        switch (adGroup.getType()) {
            case MCBANNER:
            case CONTENT_PROMOTION_VIDEO:
                return toCommonModelChanges(adGroup, AdGroup.class);
            case BASE:
                return toTextModelChanges((TextAdGroup) adGroup);
            case CPM_AUDIO:
                return toCpmAudioModelChanges((CpmAudioAdGroup) adGroup);
            case CPM_BANNER:
                return toCpmBannerModelChanges((CpmBannerAdGroup) adGroup);
            case CONTENT_PROMOTION:
                return toContentPromotionModelChanges((ContentPromotionAdGroup) adGroup);
            case CPM_GEOPRODUCT:
                return toCpmGeoproductModelChanges((CpmGeoproductAdGroup) adGroup);
            case CPM_GEO_PIN:
                return toCpmGeoPinModelChanges((CpmGeoPinAdGroup) adGroup);
            case CPM_YNDX_FRONTPAGE:
                return toCpmYndxFrontpageModelChanges((CpmYndxFrontpageAdGroup) adGroup);
            case CPM_VIDEO:
                return toCpmVideoModelChanges((CpmVideoAdGroup) adGroup);
            case CPM_OUTDOOR:
                return toModelChanges((CpmOutdoorAdGroup) adGroup);
            case CPM_INDOOR:
                return toModelChanges((CpmIndoorAdGroup) adGroup);
            case PERFORMANCE:
                return toPerformanceModelChanges((PerformanceAdGroup) adGroup);
            case DYNAMIC:
                return toDynamicModelChanges((DynamicAdGroup) adGroup);
            case MOBILE_CONTENT:
                return toMobileContentModelChanges((MobileContentAdGroup) adGroup);

            default:
                throw new UnsupportedOperationException(
                        format("Ad group of type %s are not supported", adGroup.getType().name()));
        }
    }

    private static ModelChanges<AdGroup> toTextModelChanges(TextAdGroup adGroup) {
        return toCommonModelChanges(adGroup, TextAdGroup.class)
                .process(adGroup.getFilteredFeedId(), TextAdGroup.FILTERED_FEED_ID)
                .process(adGroup.getFieldToUseAsName(), TextAdGroup.FIELD_TO_USE_AS_NAME)
                .process(adGroup.getFieldToUseAsBody(), TextAdGroup.FIELD_TO_USE_AS_BODY)
                .process(adGroup.getFeedId(), TextAdGroup.FEED_ID)
                .process(adGroup.getFeedFilter(), TextAdGroup.FEED_FILTER)
                .process(adGroup.getUsersSegments(), TextAdGroup.USERS_SEGMENTS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toCpmBannerModelChanges(CpmBannerAdGroup adGroup) {
        return toCommonModelChanges(adGroup, CpmBannerAdGroup.class)
                .process(adGroup.getForInBanners(), CpmBannerAdGroup.FOR_IN_BANNERS)
                .processNotNull(adGroup.getPriority(), CpmBannerAdGroup.PRIORITY)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toCpmAudioModelChanges(CpmAudioAdGroup adGroup) {
        return toCommonModelChanges(adGroup, CpmAudioAdGroup.class)
                .processNotNull(adGroup.getPriority(), CpmAudioAdGroup.PRIORITY)
                .castModelUp(AdGroup.class);
    }

    private static <T extends AdGroup> ModelChanges<T> toCommonModelChanges(T adGroup, Class<T> clazz) {
        return new ModelChanges<>(adGroup.getId(), clazz)
                .process(adGroup.getName(), AdGroup.NAME)
                .process(adGroup.getGeo(), AdGroup.GEO)
                .process(adGroup.getMinusKeywords(), AdGroup.MINUS_KEYWORDS)
                .processNotNull(adGroup.getLibraryMinusKeywordsIds(), AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .process(adGroup.getTags(), AdGroup.TAGS)
                .processNotNull(adGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), AdGroup.TARGET_TAGS)
                .processNotNull(adGroup.getProjectParamConditions(), AdGroup.PROJECT_PARAM_CONDITIONS)
                .process(adGroup.getHyperGeoId(), AdGroup.HYPER_GEO_ID)
                .process(adGroup.getContentCategoriesRetargetingConditionRules(),
                        AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES);
    }

    private static ModelChanges<AdGroup> toModelChanges(CpmOutdoorAdGroup cpmOutdoorAdGroup) {
        return new ModelChanges<>(cpmOutdoorAdGroup.getId(), CpmOutdoorAdGroup.class)
                .process(cpmOutdoorAdGroup.getName(), AdGroup.NAME)
                .process(cpmOutdoorAdGroup.getTags(), AdGroup.TAGS)
                .processNotNull(cpmOutdoorAdGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(cpmOutdoorAdGroup.getTargetTags(), CpmOutdoorAdGroup.TARGET_TAGS)
                .process(cpmOutdoorAdGroup.getPageBlocks(), CpmOutdoorAdGroup.PAGE_BLOCKS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toModelChanges(CpmIndoorAdGroup cpmIndoorAdGroup) {
        return new ModelChanges<>(cpmIndoorAdGroup.getId(), CpmIndoorAdGroup.class)
                .process(cpmIndoorAdGroup.getName(), AdGroup.NAME)
                .process(cpmIndoorAdGroup.getTags(), AdGroup.TAGS)
                .processNotNull(cpmIndoorAdGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(cpmIndoorAdGroup.getTargetTags(), CpmIndoorAdGroup.TARGET_TAGS)
                .process(cpmIndoorAdGroup.getPageBlocks(), CpmIndoorAdGroup.PAGE_BLOCKS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toCpmGeoproductModelChanges(CpmGeoproductAdGroup adGroup) {
        return new ModelChanges<>(adGroup.getId(), CpmGeoproductAdGroup.class)
                .process(adGroup.getName(), AdGroup.NAME)
                .process(adGroup.getGeo(), AdGroup.GEO)
                .process(adGroup.getTags(), AdGroup.TAGS)
                .processNotNull(adGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), AdGroup.TARGET_TAGS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toCpmGeoPinModelChanges(CpmGeoPinAdGroup adGroup) {
        return new ModelChanges<>(adGroup.getId(), CpmGeoPinAdGroup.class)
                .process(adGroup.getName(), AdGroup.NAME)
                .process(adGroup.getGeo(), AdGroup.GEO)
                .process(adGroup.getTags(), AdGroup.TAGS)
                .processNotNull(adGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), AdGroup.TARGET_TAGS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toCpmYndxFrontpageModelChanges(
            CpmYndxFrontpageAdGroup cpmYndxFrontpageAdGroup) {
        return new ModelChanges<>(cpmYndxFrontpageAdGroup.getId(), CpmYndxFrontpageAdGroup.class)
                .process(cpmYndxFrontpageAdGroup.getName(), AdGroup.NAME)
                .process(cpmYndxFrontpageAdGroup.getGeo(), AdGroup.GEO)
                .process(cpmYndxFrontpageAdGroup.getTags(), AdGroup.TAGS)
                .processNotNull(cpmYndxFrontpageAdGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(cpmYndxFrontpageAdGroup.getTargetTags(), CpmYndxFrontpageAdGroup.TARGET_TAGS)
                .processNotNull(cpmYndxFrontpageAdGroup.getPriority(), CpmYndxFrontpageAdGroup.PRIORITY)
                .process(cpmYndxFrontpageAdGroup.getContentCategoriesRetargetingConditionRules(),
                        AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toCpmVideoModelChanges(
            CpmVideoAdGroup cpmVideoAdGroup) {
        return new ModelChanges<>(cpmVideoAdGroup.getId(), CpmVideoAdGroup.class)
                .process(cpmVideoAdGroup.getName(), AdGroup.NAME)
                .process(cpmVideoAdGroup.getGeo(), AdGroup.GEO)
                .process(cpmVideoAdGroup.getTags(), AdGroup.TAGS)
                .processNotNull(cpmVideoAdGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(cpmVideoAdGroup.getTargetTags(), CpmVideoAdGroup.TARGET_TAGS)
                .processNotNull(cpmVideoAdGroup.getProjectParamConditions(), CpmVideoAdGroup.PROJECT_PARAM_CONDITIONS)
                .processNotNull(cpmVideoAdGroup.getPriority(), CpmVideoAdGroup.PRIORITY)
                .process(cpmVideoAdGroup.getIsNonSkippable(), CpmVideoAdGroup.IS_NON_SKIPPABLE)
                .process(cpmVideoAdGroup.getContentCategoriesRetargetingConditionRules(),
                        AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toPerformanceModelChanges(PerformanceAdGroup adGroup) {
        return new ModelChanges<>(adGroup.getId(), PerformanceAdGroup.class)
                .process(adGroup.getFieldToUseAsName(), PerformanceAdGroup.FIELD_TO_USE_AS_NAME)
                .process(adGroup.getFieldToUseAsBody(), PerformanceAdGroup.FIELD_TO_USE_AS_BODY)
                .process(adGroup.getName(), PerformanceAdGroup.NAME)
                .process(adGroup.getGeo(), PerformanceAdGroup.GEO)
                .process(adGroup.getHyperGeoId(), PerformanceAdGroup.HYPER_GEO_ID)
                .process(adGroup.getMinusKeywords(), PerformanceAdGroup.MINUS_KEYWORDS)
                .processNotNull(adGroup.getLibraryMinusKeywordsIds(), PerformanceAdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .process(adGroup.getTrackingParams(), PerformanceAdGroup.TRACKING_PARAMS)
                .processNotNull(adGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), AdGroup.TARGET_TAGS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toDynamicModelChanges(DynamicAdGroup adGroup) {
        return new ModelChanges<>(adGroup.getId(), DynamicAdGroup.class)
                .process(adGroup.getFieldToUseAsName(), DynamicAdGroup.FIELD_TO_USE_AS_NAME)
                .process(adGroup.getFieldToUseAsBody(), DynamicAdGroup.FIELD_TO_USE_AS_BODY)
                .process(adGroup.getName(), DynamicAdGroup.NAME)
                .process(adGroup.getGeo(), DynamicAdGroup.GEO)
                .process(adGroup.getHyperGeoId(), DynamicAdGroup.HYPER_GEO_ID)
                .process(adGroup.getMinusKeywords(), DynamicAdGroup.MINUS_KEYWORDS)
                .processNotNull(adGroup.getLibraryMinusKeywordsIds(), DynamicAdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .processNotNull(adGroup.getPageGroupTags(), DynamicAdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), DynamicAdGroup.TARGET_TAGS)
                .processNotNull(adGroup.getRelevanceMatchCategories(), DynamicAdGroup.RELEVANCE_MATCH_CATEGORIES)
                .process(adGroup.getTrackingParams(), DynamicAdGroup.TRACKING_PARAMS)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toMobileContentModelChanges(MobileContentAdGroup adGroup) {
        return new ModelChanges<>(adGroup.getId(), MobileContentAdGroup.class)
                .process(adGroup.getName(), MobileContentAdGroup.NAME)
                .process(adGroup.getGeo(), MobileContentAdGroup.GEO)
                .process(adGroup.getMinusKeywords(), MobileContentAdGroup.MINUS_KEYWORDS)
                .processNotNull(adGroup.getLibraryMinusKeywordsIds(),
                        MobileContentAdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .processNotNull(adGroup.getPageGroupTags(), MobileContentAdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), MobileContentAdGroup.TARGET_TAGS)
                .process(adGroup.getStoreUrl(), MobileContentAdGroup.STORE_URL)
                .process(adGroup.getMinimalOperatingSystemVersion(),
                        MobileContentAdGroup.MINIMAL_OPERATING_SYSTEM_VERSION)
                .process(adGroup.getDeviceTypeTargeting(), MobileContentAdGroup.DEVICE_TYPE_TARGETING)
                .process(adGroup.getNetworkTargeting(), MobileContentAdGroup.NETWORK_TARGETING)
                .process(adGroup.getMobileContent(), MobileContentAdGroup.MOBILE_CONTENT)
                .castModelUp(AdGroup.class);
    }

    private static ModelChanges<AdGroup> toContentPromotionModelChanges(ContentPromotionAdGroup adGroup) {

        return new ModelChanges<>(adGroup.getId(), ContentPromotionAdGroup.class)
                .process(adGroup.getName(), ContentPromotionAdGroup.NAME)
                .process(adGroup.getGeo(), ContentPromotionAdGroup.GEO)
                .process(adGroup.getMinusKeywords(), ContentPromotionAdGroup.MINUS_KEYWORDS)
                .processNotNull(adGroup.getLibraryMinusKeywordsIds(),
                        ContentPromotionAdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .process(adGroup.getTags(), ContentPromotionAdGroup.TAGS)
                .processNotNull(adGroup.getPageGroupTags(), AdGroup.PAGE_GROUP_TAGS)
                .processNotNull(adGroup.getTargetTags(), AdGroup.TARGET_TAGS)
                .castModelUp(AdGroup.class);
    }
}
